/*
 * Copyright (c) 2015 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.runner;

import org.mule.api.MuleContext;
import org.mule.api.context.MuleContextAware;
import org.mule.munit.common.extensions.MunitPlugin;

import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Enumeration;
import java.util.List;
import java.util.Properties;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

/**
 * <p>
 * Loads all the {@link MunitPlugin} that are in the Class Loader, those plugins are set with the {@link MuleContext}
 * (if necessary) but are not initialised
 * </p>
 *
 * @author Mulesoft Inc.
 * @since 3.4
 */
public class MunitPluginFactory
{

    private static Log log = LogFactory.getLog(MunitPluginFactory.class);

    public Collection<MunitPlugin> loadPlugins(MuleContext context)
    {
        List<MunitPlugin> munitPlugins = new ArrayList<MunitPlugin>();
        ClassLoader contextClassLoader = Thread.currentThread().getContextClassLoader();
        try
        {
            Enumeration<URL> resources = contextClassLoader.getResources("META-INF/munit-plugin.properties");
            while (resources.hasMoreElements())
            {
                Properties properties = new Properties();
                Object content = resources.nextElement().getContent();
                properties.load((InputStream) content);
                MunitPlugin plugin = createMunitPlugin(contextClassLoader, properties);
                if (plugin != null)
                {
                    if (plugin instanceof MuleContextAware)
                    {
                        ((MuleContextAware) plugin).setMuleContext(context);
                    }
                    munitPlugins.add(plugin);
                }
            }
        }
        catch (IOException e)
        {
            log.error("Could not read the Classpath in order to get the plugin configurations");
        }


        return munitPlugins;
    }

    private MunitPlugin createMunitPlugin(ClassLoader contextClassLoader, Properties properties)
    {
        String property = properties.getProperty("plugin.className");
        try
        {
            if (property != null && !property.isEmpty())
            {
                return (MunitPlugin) contextClassLoader.loadClass(property).newInstance();
            }

        }
        catch (Throwable e)
        {
            log.error("The class " + property + " could not be load");
        }
        return null;
    }
}
