/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.plugin.maven;

import org.apache.commons.lang.StringUtils;
import org.apache.maven.artifact.Artifact;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugin.MojoFailureException;
import org.apache.maven.plugins.annotations.Component;
import org.apache.maven.plugins.annotations.Parameter;
import org.apache.maven.project.MavenProjectHelper;
import org.codehaus.plexus.archiver.ArchiverException;

import java.io.File;
import java.io.IOException;
import java.util.List;
import java.util.Set;

public abstract class AbstractPackagePluginMojo extends AbstractMuleMojo
{

    protected static final String PLUGIN_PROPERTIES_FILE_NAME = "plugin.properties";

    protected abstract void doAddPluginProperties(File pluginPropertiesFile) throws MojoFailureException;

    @Component
    private MavenProjectHelper projectHelper;
    /**
     * List of exclusion elements (having groupId and artifactId children) to exclude from the
     * application archive.
     */
    @Parameter
    private List<Exclusion> exclusions;
    /**
     * List of inclusion elements (having groupId and artifactId children) to include from the
     * application archive.
     */
    @Parameter
    private List<Inclusion> inclusions;

    public AbstractPackagePluginMojo()
    {
        super();
    }

    @Override
    public final void execute() throws MojoExecutionException, MojoFailureException
    {
        initialize();
        doExecute();
        File plugin = getMuleZipFile();
        try
        {
            createMulePlugin(plugin);
        }
        catch (ArchiverException e)
        {
            throw new MojoExecutionException("Exception creating the Mule Plugin", e);
        }

        this.projectHelper.attachArtifact(this.project, "zip", StringUtils.EMPTY, plugin);
    }

    protected void createMulePlugin(final File plugin) throws MojoExecutionException, MojoFailureException, ArchiverException
    {
        ModuleArchiver archiver = new ModuleArchiver();
        addToArchiver(archiver);

        archiver.setDestFile(plugin);

        try
        {
            plugin.delete();
            archiver.createArchive();
        }
        catch (IOException e)
        {
            getLog().error("Cannot create archive", e);
        }
    }

    protected void addToArchiver(ModuleArchiver archiver) throws MojoFailureException
    {
        addDependencies(archiver);
        addPluginProperties(archiver);
    }

    private void addPluginProperties(ModuleArchiver moduleArchiver) throws MojoFailureException
    {
        File pluginPropertiesFile = new File(outputDirectory, PLUGIN_PROPERTIES_FILE_NAME);

        doAddPluginProperties(pluginPropertiesFile);

        moduleArchiver.add(pluginPropertiesFile);
    }

    private void addDependencies(ModuleArchiver archiver) throws ArchiverException
    {
        for (Artifact artifact : getArtifactsToArchive())
        {
            getLog().info(String.format("Adding <%1s> as a lib", artifact.getId()));
            archiver.addLib(artifact.getFile());
        }
    }

    private Set<Artifact> getArtifactsToArchive()
    {
        ArtifactFilter filter = new ArtifactFilter(project, inclusions, exclusions);
        return filter.getArtifactsToArchive();
    }

    /**
     * Method to give the capability of initialize implementations of this class as the first action of the
     * {@link AbstractPackagePluginMojo#execute()}
     *
     * @throws MojoFailureException   when an initialization failure occurs
     * @throws MojoExecutionException when an initialization error occurs
     */
    protected void initialize() throws MojoFailureException, MojoExecutionException
    {
    }

    /**
     * Method to give the capability of contribute with the {@link AbstractPackagePluginMojo#execute()}
     *
     * @throws MojoFailureException   when an execution failure occurs
     * @throws MojoExecutionException when an execution error occurs
     */
    protected void doExecute() throws MojoFailureException, MojoExecutionException
    {
    }
}