/**
* Support functions for "transformed operations"
*/
%dw 2.8
import HttpResponse from com::mulesoft::connectivity::transport::Http
import mapInput, mapSuccessResult from com::mulesoft::connectivity::decorator::Executor

import PaginatedOperation, Operation, ResultFailure, Page, ResultSuccess from com::mulesoft::connectivity::Model
import drop from dw::core::Arrays
import fail from dw::Runtime

/**
* Maps a field between two types.
*
* If `from` is not present on the source object during mapping the behaviour varies. On mappings where `defaultValue`
* is present it will will be written instead. Otherwise the field mapping will be skipped.
*
* Fields:
* - `from`: The path to the field in the source type
* - `to`: The path to the field in the destination type
* - `defaultValue`: The value to write if `from` is absent on the source object
*/
type FieldMapping = {
    from: Array<String>,
    to: Array<String>,
    defaultValue?: Any
}

/**
* Maps a field assigning a given constant when necessary.
*
* By default constant field mapping are always written. In order to implement "default values" (fields that are written
* if no inner mappings are executed) the `isFallback` property can be set to `true`.
*
* Fields:
* - `value`: The constant to be put there
* - `to`: The path to the field in the destination type
* - `isFallback`: `false` for `'always'` mappings and true for regular default values (assumed `false` if absent)
*/
type ConstantFieldMapping = {
    value: Any,
    to: Array<String>,
    isFallback?: Boolean
}

/**
* A `mapsTo` destination that includes a default value used if the source is absent
*/
type DestinationWithDefault = {
    destination: String | Array<String>,
    defaultValue: Any
}

/**
* Creates a new `FieldMapping` from its arguments
*
* === Parameters
*
* [%header, cols="1,1,3"]
* |===
* | Name | Type | Description
* | `from` | Array<String&#62; | A list of path parts describing the source field
* | `to` | Array<String&#62; | A list of path parts describing the destination field
* |===
*
*/
fun mapsTo(from: Array<String>, to: Array<String>): FieldMapping = {
    from: from,
    to: to
}

/**
* Creates a new `FieldMapping` from its arguments
*
* === Parameters
*
* [%header, cols="1,1,3"]
* |===
* | Name | Type | Description
* | `from` | String | A dot-delimited string describing the path parts of the source field
* | `to` | String | A dot-delimited string describing the path parts of the destination field
* |===
*
*/
fun mapsTo(from: String, to: String): FieldMapping = {
    from: from splitBy '.',
    to: to splitBy '.'
}

/**
* Creates a new `FieldMapping` from its arguments
*
* === Parameters
*
* [%header, cols="1,1,3"]
* |===
* | Name | Type | Description
* | `from` | Array<String&#62; | A list of path parts describing the source field
* | `to` | String | A dot-delimited string describing the path parts of the destination field
* |===
*
*/
fun mapsTo(from: Array<String>, to: String): FieldMapping = {
    from: from,
    to: to splitBy '.'
}

/**
* Creates a new `FieldMapping` from its arguments
*
* === Parameters
*
* [%header, cols="1,1,3"]
* |===
* | Name | Type | Description
* | `from` | String | A dot-delimited string describing the path parts of the source field
* | `to` | Array<String&#62; | A list of path parts describing the destination field
* |===
*
*/
fun mapsTo(from: String, to: Array<String>): FieldMapping = {
    from: from splitBy '.',
    to: to
}

/**
* Creates a new `FieldMapping` from its arguments
*
* === Parameters
*
* [%header, cols="1,1,3"]
* |===
* | Name | Type | Description
* | `from` | String | A field source: either a dot-delimited string describing the path parts of the source field of a list of those path parts
* | `to` | DestinationWithDefault | An object describing both the field sink and the default value to use when the source is absent
* |===
*
*/
fun mapsTo(from: String | Array<String>, to: DestinationWithDefault): FieldMapping =
    from mapsTo to.destination
    update { case .defaultValue! -> to.defaultValue }

/**
* Creates a new `ConstantFieldMapping` from its arguments
*
* The created mapping will replace its destination even if some sub-mapping writes to it before.
*
* === Parameters
*
* [%header, cols="1,1,3"]
* |===
* | Name | Type | Description
* | `value` | Any | The value to be used for that field
* | `to` | String | A dot-delimited string describing the path parts of the destination field
* |===
*
*/
fun alwaysMapsTo(value: Any, to: String): ConstantFieldMapping = {
    value: value,
    to: to splitBy '.',
    isFallback: false
}

/**
* Creates a new `ConstantFieldMapping` from its arguments
*
* The created mapping will replace its destination even if some sub-mapping writes to it before.
*
* === Parameters
*
* [%header, cols="1,1,3"]
* |===
* | Name | Type | Description
* | `value` | Any | The value to be used for that field
* | `to` | Array<String&#62; | A list of path parts describing the destination field
* |===
*
*/
fun alwaysMapsTo(value: Any, to: Array<String>): ConstantFieldMapping = {
    value: value,
    to: to,
    isFallback: false
}

/**
* Creates a new `ConstantFieldMapping` from its arguments
*
* The created mapping will only be applied if no sub-mapping writes to its destination.
*
* === Parameters
*
* [%header, cols="1,1,3"]
* |===
* | Name | Type | Description
* | `value` | Any | The value to be used for that field
* | `to` | String | A dot-delimited string describing the path parts of the destination field
* |===
*
*/
fun defaultOf(value: Any, to: String): ConstantFieldMapping = {
    value: value,
    to: to splitBy '.',
    isFallback: true
}


/**
* Creates a new `ConstantFieldMapping` from its arguments
*
* The created mapping will only be applied if no sub-mapping writes to its destination.
*
* === Parameters
*
* [%header, cols="1,1,3"]
* |===
* | Name | Type | Description
* | `value` | Any | The value to be used for that field
* | `to` | Array<String&#62; | A list of path parts describing the destination field
* |===
*
*/
fun defaultOf(value: Any, to: Array<String>): ConstantFieldMapping = {
    value: value,
    to: to,
    isFallback: true
}

@Internal(permits = [])
fun writeNested(obj: Any, field: Array<String>, value: Any): Any =
    sizeOf(field) match {
        case 0 -> value
        case 1 -> obj update {
            case .'$(field[0])'! -> value
        }
        case 2 -> obj update {
            case .'$(field[0])'.'$(field[1])'! -> value
        }
        case 3 -> obj update {
            case .'$(field[0])'.'$(field[1])'.'$(field[2])'! -> value
        }
        case 4 -> obj update {
            case .'$(field[0])'.'$(field[1])'.'$(field[2])'.'$(field[3])'! -> value
        }
        else -> obj update {
            case .'$(field[0])'.'$(field[1])'.'$(field[2])'.'$(field[3])'.'$(field[4])'! -> writeNested($ default {}, field drop 5, value)
        }
    }

@Internal(permits = [])
type NestedReadResult = {
    exists: Boolean,
    value: Any
}

@Internal(permits = [])
@TailRec()
fun tryReadNested(obj: Any, field: Array<String>): NestedReadResult =
    sizeOf(field) match {
        case 0 -> {
            exists: true,
            value: obj
        }
        case 1 -> {
            exists: obj[field[0]]?,
            value: obj[field[0]]
        }
        case 2 -> {
            exists: obj[field[0]][field[1]]?,
            value: obj[field[0]][field[1]]
        }
        case 3 -> {
            exists: obj[field[0]][field[1]][field[2]]?,
            value: obj[field[0]][field[1]][field[2]]
        }
        case 4 -> {
            exists: obj[field[0]][field[1]][field[2]][field[3]]?,
            value: obj[field[0]][field[1]][field[2]][field[3]]
        }
        else -> tryReadNested(obj[field[0]][field[1]][field[2]][field[3]][field[4]], field drop 5)
    }

/**
* Creates a new type transformer from the given mapping. The type parameters
* are not checked and should be explicitly given.
*
* This is an **unsafe** operation that can result in badly-typed values. It
* should be used with extreme caution (or by autogenerated code).
*
* === Parameters
*
* [%header, cols="1,1,3"]
* |===
* | Name | Type | Description
* | `mapping` | Array<FieldMapping &#124; ConstantFieldMapping&#62; | The mapping configuration (an array of field mappings)
* |===
*
*/
fun fromMapping<A <: Object, B <: Object>(
    mapping: Array<FieldMapping | ConstantFieldMapping>
) = (v: A): B -> do {
    fun unsafeCast<T>(v: Any): T = v as T
    var sortedMapping = mapping orderBy (entry, index) -> do {
        // 0 - `alwaysMapsTo` default value for fields with no mapping nor sub-mappings
        // 1 - `mapsTo` without default, regular mappings
        // 2 - `mapsTo` with default, mappings with fallback
        // 3 - `defaultOf` default value for a field with sub-mappings (Executed only if no sub-mappings are done)
        //
        // NOTE: `alwaysMapsTo` are run first for backwards compatibility reasons. They were previously used as a
        //       fallback for the absence of `defaultOf` mappings.
        // NOTE 2: **Correct** implementation of this sorting should instead do a toposort. This way mappings for inner
        //         fields are always processed before mappings for outer fields.
        entry match {
            case entry is FieldMapping ->
                if (entry.defaultValue?) 2 else 1
            case entry is ConstantFieldMapping ->
                if (entry.isFallback default false) 3 else 0
        }
    }
    var result = sortedMapping reduce (entry, acc={}) -> do {
        var action = entry match {
            case entry is FieldMapping -> do {
                var field = tryReadNested(v, entry.from)
                ---
                if (field.exists) { skip: false, value: field.value }
                else if (entry.defaultValue?) { skip: false, value: entry.defaultValue }
                else { skip: true }
            }
            case entry is ConstantFieldMapping -> {
                skip: (entry.isFallback default false) and tryReadNested(acc, entry.to).exists,
                value: entry.value
            }
        }
        ---
        action match {
            case writeAction is { skip: false, value: Any } -> writeNested(acc, entry.to, writeAction.value)
            case skipAction is { skip: true } -> acc
        }
    }
    ---
    unsafeCast<B>(result)
}

/**
* Applies a transformer configuration to a given base connectivity operation.
*
* === Parameters
*
* [%header, cols="1,1,3"]
* |===
* | Name | Type | Description
* | `operation` | Operation<OldIn, OldOut, Err, Conn&#62; | The operation to transform
* | `transformer` | { in: &#40;v: NewIn&#41; &#45;&#62; OldIn, out: &#40;v: OldOut&#41; &#45;&#62; NewOut } | The set of transformers to apply
* |===
*
*/
fun withTransformer<OldIn, OldOut, Err <: ResultFailure, Conn, NewIn, NewOut>(
    // This should be `withTransformer<OldIn <: Object, Out, Err <: ResultFailure, Conn, NewIn <: Object>(...)`
    // but there's a DW issue that makes inference end up using the bound
    // instead of the specific type
    operation: Operation<OldIn, OldOut, Err, Conn>,
    transformer: {
      in: (v: NewIn) -> OldIn,
      out: (v: OldOut) -> NewOut
    }
): Operation<NewIn, NewOut, Err, Conn> =
    operation update {
        case executor at .executor -> mapSuccessResult(mapInput(operation.executor, transformer.in), transformer.out)
    }

/**
* Applies a transformer configuration to a given base connectivity operation.
*
* === Parameters
*
* [%header, cols="1,1,3"]
* |===
* | Name | Type | Description
* | `operation` | Operation<OldIn, Out, Err, Conn&#62; | The operation to transform
* | `transformer` | { in: &#40;v: NewIn&#41; &#45;&#62; OldIn } | The set of transformers to apply
* |===
*
*/
fun withTransformer<OldIn, Out, Err <: ResultFailure, Conn, NewIn>(
    operation: Operation<OldIn, Out, Err, Conn>,
    transformer: {
        in: (v: NewIn) -> OldIn
    }
): Operation<NewIn, Out, Err, Conn> = withTransformer<OldIn, Out, Err, Conn, NewIn, Out>(operation, {
  in: transformer.in,
  out: doNothing
})

/**
*
* A simple transformer that extracts the `body` field from a given HTTP Response.
*
* === Parameters
*
* [%header, cols="1,1,3"]
* |===
* | Name | Type | Description
* | `response` | HttpResponse<T> |  The response to process
* |===
*
**/
fun extractRequestBody(response): ? = response.body!

/**
*
* A simple transformer generator that always returns the same value
*
* === Parameters
*
* [%header, cols="1,1,3"]
* |===
* | Name | Type | Description
* | `v` | T |  The value to use instead of the `HttpResponse`
* |===
*
**/
fun replaceResultWith<T>(v: T) = (response: HttpResponse): T -> v

/**
*
* A simple transformer that keeps its parameter as-is
*
* === Parameters
*
* [%header, cols="1,1,3"]
* |===
* | Name | Type | Description
* | `response` | T |  The response to process
* |===
*
**/
fun doNothing<T>(response: T): T = response


fun withTransformerPaginated<OldIn, NextIn, OldOut, Err <: ResultFailure, Conn, NewIn, NewOut>(
    operation: PaginatedOperation<OldIn, NextIn, Page<OldOut, NextIn>, Err, Conn>,
    transformer: {
        in: (v: NewIn) -> OldIn,
        out: (v: Array<OldOut>) -> Array<NewOut>
    }
): PaginatedOperation<NewIn, NextIn, Page<NewOut, NextIn>, Err, Conn> =
    operation update {
        case .executor -> (v: NewIn, c : Conn) ->
            operation.executor(transformer.in(v), c) match {
                case is ResultSuccess<Page<OldOut, NextIn>> -> $ update { case value at .value.items ->  transformer.out(value)}
                case is Err -> $
            }
        case .nextPageExecutor -> (v : NextIn, c : Conn) ->
            operation.nextPageExecutor(v, c) match {
                case is ResultSuccess<Page<OldOut, NextIn>> -> $ update { case value at .value.items ->  transformer.out(value)}
                case is Err -> $
            }
    }


fun withTransformerPaginated<OldIn, NextIn, OldOut, Err <: ResultFailure, Conn, NewIn>(
    operation: PaginatedOperation<OldIn, NextIn, Page<OldOut, NextIn>, Err, Conn>,
    transformer: {
        in: (v: NewIn) -> OldIn
    }
): PaginatedOperation<NewIn, NextIn, Page<OldOut, NextIn>, Err, Conn> =
    withTransformerPaginated(operation, {
        in: transformer.in,
        out: doNothing
    })
