%dw 2.7
import SemanticTerms from com::mulesoft::connectivity::decorator::Annotations
import * from com::mulesoft::connectivity::Model

import sendRequest, createBinaryHttpRequest, readHttpResponseBody,DEFAULT_HTTP_CLIENT_CONFIG, DEFAULT_HTTP_REQUEST_CONFIG from dw::io::http::Client
import HttpRequestCookies, BinaryRequestEncoder, BinaryResponseDecoder from dw::io::http::Types

import * from dw::io::http::Types
import * from dw::core::Objects

import toBase64 from dw::core::Binaries
import encodeURIComponent, decodeURI from dw::core::URL

import * from com::mulesoft::connectivity::transport::Serialization

/******************************************************************************************************************
* Request/Response
*******************************************************************************************************************/

/**
 * DataWeave type for representing a HTTP request methods.
 */
type HttpMethod = "GET" | "POST" | "PUT" | "DELETE" | "PATCH" | "OPTIONS"

/**
* Type that represents an HTTP connection instance
*
* The fields represent a "default" or "initial" value configuring things
* necessary for requests through this connection to succeed.
*
* Fields:
* - `method`: The HTTP method
* - `baseUri`: Optional. The base URI
* - `path`: The specific location of the resource (not including `baseUri` or query parameters)
* - `queryParams`: Optional. The query parameters
* - `headers`: Optional. The initial headers, may contain authentication information
* - `body`: Optional. The initial body
* - `config`: Options for the HTTP request
*/
type HttpRequester = {
    method: HttpMethod,
    baseUri?: String,
    path: String,
    queryParams?: { _?: Any },
    headers?: { _?: String },
    body?: Any,
    config?: {
      contentType?: String,
      followRedirects?: Boolean,
      returnRawResponse?: Boolean,
      readTimeout?: Number,
      requestTimeout?: Number,
      connectionTimeout?: Number
    },
    cookie?: HttpRequestCookies
}

/**
 * Type for representing an HTTP response cookies.
 *
 * Fields:
 * - `name`: The name of the cookie.
 * - `value`: The value of the cookie.
 * - `maxAge`: The maximum age of the cookie, specified in seconds.
 * - `httpOnly`: `true` if this cookie contains the HttpOnly attribute.
 * - `secure`: `true` if sending this cookie should be restricted to a secure protocol, or `false` if the it can be sent using any protocol.
 * - `domain`: The domain name set for this cookie.
 * - `comment`: The comment describing the purpose of this cookie.
 * - `path`: The path on the server to which the browser returns this cookie.
 *
 */
type HttpResponseCookie = {
  name: String,
  value: String,
  maxAge: Number,
  httpOnly: Boolean,
  secure: Boolean,
  domain?: String,
  comment?: String,
  path?: String
}

/**
* Type that represents a connection response whose body is of `BodyType`
*
* Fields:
* - `contentType`: The content type of the response
* - `status`: The status code of the response
* - `statusText`: Optional. A text description of the status code
* - `headers`: The headers of the response
* - `body`: Optional. The contents of the response
* - `cookies`: Optional. Cookies sent along with the response
*/
type HttpResponse<BodyType> = {
    contentType?: String,
    status: Number,
    statusText?: String,
    headers: { _?: String },
    body?: BodyType,
    cookies: { _?: HttpResponseCookie }
}

/**
* Type that represents a connection response without body
*
* Fields:
* - `contentType`: The content type of the response
* - `status`: The status code of the response
* - `statusText`: Optional. A text description of the status code
* - `headers`: The headers of the response
* - `cookies`: Optional. Cookies sent along with the response
*/
type NoContentHttpResponse = {
    contentType?: String,
    status: Number,
    statusText?: String,
    headers: { _?: String },
    cookies: { _?: HttpResponseCookie }
}


/**
* Invokes a native HTTP request based on the HTTP connection instance given
*
* === Parameters
*
* [%header, cols="1,1,3"]
* |===
* | Name | Type | Description
* | `httpRequest` | HttpRequester | An HTTP connection instance
* |===
*
*/
fun doRequest(req: HttpRequester): HttpResponse = do {
    var queryParams =
        if (!isEmpty(req.queryParams))
            "?$(serializeQueryParams(req.queryParams!))"
        else
            ""
    var url = req.baseUri! ++ req.path ++ queryParams
    var serializationConfig = {
      contentType: req.config.contentType default "application/json",
      readerProperties: {},
      writerProperties: {}
    }
    var CUSTOM_HTTP_REQUEST_CONFIG = DEFAULT_HTTP_REQUEST_CONFIG mergeWith {
                                       followRedirects : req.config.followRedirects default false,
                                       readTimeout: req.config.readTimeout default 60000,
                                       requestTimeout: req.config.requestTimeout default 60000,
                                       // Adds streaming support for raw binary responses only (req.config.returnRawResponse = true)
                                       // Has no effect when req.config.returnRawResponse = false
                                       streamResponse: true
    }
    var CUSTOM_HTTP_CLIENT_CONFIG = DEFAULT_HTTP_CLIENT_CONFIG mergeWith { connectionTimeout :  req.config.connectionTimeout default 5000 }
    var binaryRequestEncoder = { encode: (httpRequest) -> createBinaryHttpRequest(httpRequest, serializationConfig) }
    var binaryResponseDecoder =
         if (req.config.returnRawResponse default false)
                 { decode: (httpResponse) -> httpResponse }
          else
                 { decode: (httpResponse) -> readHttpResponseBody(httpResponse, serializationConfig) }
    ---
    sendRequest<Any, { _?: String }>({
     method: req.method,
     url: url,
     (headers: req.headers!) if req.headers?,
     (cookies: req.cookie!) if req.cookie?,
     (body: req.body) if req.body?
   }, binaryRequestEncoder, binaryResponseDecoder, CUSTOM_HTTP_REQUEST_CONFIG, CUSTOM_HTTP_CLIENT_CONFIG)
}

/******************************************************************************************************************
* Connection/Authentication
*******************************************************************************************************************/
/**
* Type that represents an authentication type, it may be a well known type or
* a custom one
*
* Well known authentication types:
* - `{ "type": "basic" }`: Basic authentication as per [RFC7617]
* - `{ "type": "bearer" }`: Bearer authentication as per [RFC6750]
* - `{ "type": "apiKey" }`: API Key authentication
* - `{ "type": "oauth2" }`: OAuth 2 authentication
*
* Custom authentication types are described as objects with a `custom` type and
* a `subType` field naming the authentication type.
*
* [RFC7617]: https://datatracker.ietf.org/doc/html/rfc7617
* [RFC6750]: https://datatracker.ietf.org/doc/html/rfc6750
*/
type HttpAuthenticationType = {
    "type": 'basic' | 'bearer' | 'apiKey' | 'oauth2' // Supported and well known HTTP authentication types
} | { "type": 'custom', subType: String }

/**
* Type for HTTP connection provider descriptions
*
* Has the same shape as a regular `ConnectionProvider` but its instances are required
* to have an `HttpRequester` as their connection request instance and a `HttpResponse` as their response instance.
*/
type HttpConnectionProvider<InputType <: Object, HttpAuthenticationType> =
    ConnectionProvider<HttpConnection, InputType, HttpAuthenticationType> & {extensions?: (InputType) -> Array<ConnectionProviderExtensions | ConnectionProviderOAuthExtensions>}

/**
* Type for HTTP connection
*
* Has the same shape as a regular `Connection` but its instances are required
* to have an `HttpRequester` as their connection request instance and a `HttpResponse` as their response instance.
*/
type HttpConnection = Connection<HttpRequester, HttpResponse>

/**
* Type for HTTP test connection
*
* Has the same shape as a regular `TestConnection` but its instances are required
* to have an `HttpConnection` as their `Connection`.
*/
type HttpTestConnection = TestConnection<HttpConnection>

/**
* Type for HTTP connection schema
*
* Fields:
* - `baseUri`: Optional. The base URI
*/
type HttpConnectionProviderSchema = {|
    baseUri?: String
|}

/**
* Type for HTTP connection extensions for default connections
*
* Fields:
* - `in`: Required. Where the parameter goes
* - `name`: Required. The parameter name
* - `value`: Required. The parameter value
*/
type ConnectionProviderExtensions = {
    in: "query" | "header" | "body" | "path" | "cookie",
    name: String,
    value: String //It must be serialized at this point
}

/**
* Type for HTTP OAuth connection extensions for default OAuth connections
*
* Fields:
* - `inDance`: Required. If the parameter is used for the dance or for each request
* - `ConnectionProviderExtensions`: Required. ConnectionProviderExtensions inherited type
*/
type ConnectionProviderOAuthExtensions = {
    inDance: true | false
} & ConnectionProviderExtensions

/**
* Type for HTTP response interceptor. Given a response, it returns a new response.
*/
type ResponseInterceptor = (HttpResponse) -> HttpResponse

/**
* Type for HTTP connection options
*
* Fields:
* - `extensions`: Optional. A list of extensions to apply to the HTTP connection
* - `responseInterceptors`: Optional. A list of response interceptors to apply to the HTTP connection
*/
type HttpConnectionOptions<InputType <: Object> = {
    extensions?: (InputType) -> Array<ConnectionProviderExtensions | ConnectionProviderOAuthExtensions>,
    responseInterceptors?: Array<ResponseInterceptor>
}
/**
* Generic function to define a `connect` function for an HTTP connection
*
* === Parameters
*
* [%header, cols="1,1,3"]
* |===
* | Name | Type | Description
* | `authSchemaMapper` | &#40;InputType&#41; &#45;&#62; AuthSchema | mapping function from a connection schema to an authentication schema
* | `connectionProviderSchemaMapper` | &#40;InputType&#41; &#45;&#62; HttpConnectionProviderSchema | mapping function for a connection schema
* | `authenticate` | &#40;AuthSchema, HttpRequester&#41; &#45;&#62; HttpRequester | function to build an HTTP connection instance with authentication information
* |===
*/
fun defineHttpConnection<InputType, AuthSchema>(
	authSchemaMapper: (InputType) -> AuthSchema,
    connectionProviderSchemaMapper: (InputType) -> HttpConnectionProviderSchema,
    authenticate: (AuthSchema, HttpRequester) -> HttpRequester
	) = (parameter: InputType) -> do {
	    var httpConnectionSchema = connectionProviderSchemaMapper(parameter)
	    ---
	    (httpRequest: HttpRequester) ->
                doRequest(authenticate(
                                        authSchemaMapper(parameter),
                                       setBaseUri(httpConnectionSchema, httpRequest)))
	}

/**
* Generic function to define a `connect` function for an HTTP connection
*
* === Parameters
*
* [%header, cols="1,1,3"]
* |===
* | Name | Type | Description
* | `authSchemaMapper` | &#40;InputType&#41; &#45;&#62; AuthSchema | mapping function from a connection schema to an authentication schema
* | `connectionProviderSchemaMapper` | &#40;InputType&#41; &#45;&#62; HttpConnectionProviderSchema | mapping function for a connection schema
* | `authenticate` | &#40;AuthSchema, HttpRequester&#41; &#45;&#62; HttpRequester | function to build an HTTP connection instance with authentication information
* | `attributes` | AuthAttributes | Additional attributes required for the authentication function 
* |===
*
*/
fun defineHttpConnection<InputType, AuthSchema, AuthAttributes>(
	authSchemaMapper: (InputType) -> AuthSchema,
    connectionProviderSchemaMapper: (InputType) -> HttpConnectionProviderSchema,
    authenticate: (AuthSchema, AuthAttributes, HttpRequester) -> HttpRequester,
    attributes: AuthAttributes
	) = (parameter: InputType) -> do {
	    var httpConnectionSchema = connectionProviderSchemaMapper(parameter)
	    ---
	    (httpRequest: HttpRequester) ->
                doRequest(
                authenticate(
                            authSchemaMapper(parameter),
                           attributes,
                          setBaseUri(httpConnectionSchema, httpRequest)))
	}

/**
* Function which applies the connection extensions to an existing HttpRequester
*
* === Parameters
*
* [%header, cols="1,1,3"]
* |===
* | Name | Type | Description
* | `extensions` | Array<ConnectionProviderExtensions | ConnectionProviderOAuthExtensions> | List of connection extensions
* | `httpRequest` | HttpRequester | The http request to add the extensions
* |===
*
*/
@Internal(permits=['http::ConnectionExtensionsTest'])
fun applyExtensions(extensions: Array<ConnectionProviderExtensions | ConnectionProviderOAuthExtensions>, httpRequest: HttpRequester): HttpRequester =
    extensions reduce (extension, httpRequest = httpRequest) -> do {
        var processExtension =  if(extension.inDance?)
                                    !extension.inDance
                                else
                                    true
        ---
            if(processExtension)
                extension.in match {
                    case "header" ->
                        httpRequest update {
                            case .headers.`$(extension.name)`! -> extension.value
                            }
                    case "query" ->
                        httpRequest update {
                            case .queryParams.`$(extension.name)`! -> extension.value
                            }
                    case "path" ->
                            if(indexOf(httpRequest.baseUri,'{'++ extension.name ++ '}')>=0)
                                httpRequest update {
                                    case baseUri at .baseUri-> baseUri replace ('{'++ extension.name ++ '}') with(extension.value)
                                }
                            else
                                if(indexOf(httpRequest.path,'{'++ extension.name ++ '}')>=0)
                                    httpRequest update {
                                        case path at .path-> path replace ('{'++ extension.name ++ '}') with(extension.value)
                                    }
                                else
                                    log("WARNING - Defined path extension could not be resolved in path",httpRequest)
                    case "body" ->
                        httpRequest update {
                            case body at .body! -> onNull(body,()->{}) ++ {`$(extension.name)` : extension.value}
                            //TODO: It does not work for many cases. Have to review this and other cases such as complex objects, xml, form-urlencoded, etc
                            }
                    case "cookie" ->
                        httpRequest update {
                            case .cookie.`$(extension.name)`! -> extension.value
                            }
                }
            else
                httpRequest
    }

/**
* Applies a list of response interceptors to a response
*
* === Parameters
*
* [%header, cols="1,1,3"]
* |===
* | Name | Type | Description
* | `responseInterceptors` | Array<ResponseInterceptor> | A list of response interceptors
* | `httpResponse` | HttpResponse | The response to intercept
* |===
*
*/
fun intercept(responseInterceptors: Array<ResponseInterceptor>, httpResponse: HttpResponse): HttpResponse =
    responseInterceptors reduce (interceptor, httpResponse = httpResponse) -> 
        interceptor(httpResponse)

/**
* Generic function to define a `connect` function for an HTTP connection
*
* === Parameters
*
* [%header, cols="1,1,3"]
* |===
* | Name | Type | Description
* | `authSchemaMapper` | &#40;InputType&#41; &#45;&#62; AuthSchema | mapping function from a connection schema to an authentication schema
* | `connectionProviderSchemaMapper` | &#40;InputType&#41; &#45;&#62; HttpConnectionProviderSchema | mapping function for a connection schema
* | `authenticate` | &#40;AuthSchema, HttpRequester&#41; &#45;&#62; HttpRequester | function to build an HTTP connection instance with authentication information
* | `extensions` | (InputType) -> Array<ConnectionProviderExtensions | ConnectionProviderOAuthExtensions> | function which returns a list of extensions from an input type
* |===
*
*/
@Deprecated(since= "05/07/2025", replacement= "Use defineHttpConnection<InputType, AuthSchema>(authSchemaMapper, connectionProviderSchemaMapper, authenticate, options) instead")
fun defineHttpConnection<InputType, AuthSchema>(
	authSchemaMapper: (InputType) -> AuthSchema,
    connectionProviderSchemaMapper: (InputType) -> HttpConnectionProviderSchema,
    authenticate: (AuthSchema, HttpRequester) -> HttpRequester,
    extensions: (InputType) -> Array<ConnectionProviderExtensions | ConnectionProviderOAuthExtensions>
	) = (parameter: InputType) -> do {
	    var httpConnectionSchema = connectionProviderSchemaMapper(parameter)
	    ---
	    (httpRequest: HttpRequester) ->
                doRequest(applyExtensions(extensions(parameter),
                authenticate(
                            authSchemaMapper(parameter),
                          setBaseUri(httpConnectionSchema, httpRequest))))
	}

/**
* Generic function to define a `connect` function for an HTTP connection
*
* === Parameters
*
* [%header, cols="1,1,3"]
* |===
* | Name | Type | Description
* | `authSchemaMapper` | &#40;InputType&#41; &#45;&#62; AuthSchema | mapping function from a connection schema to an authentication schema
* | `connectionProviderSchemaMapper` | &#40;InputType&#41; &#45;&#62; HttpConnectionProviderSchema | mapping function for a connection schema
* | `authenticate` | &#40;AuthSchema, HttpRequester&#41; &#45;&#62; HttpRequester | function to build an HTTP connection instance with authentication information
* | `attributes` | AuthAttributes | Additional attributes required for the authentication function
* | `extensions` | (InputType) -> Array<ConnectionProviderExtensions | ConnectionProviderOAuthExtensions> | function which returns a list of extensions from an input type
* |===
*
*/
@Deprecated(since= "05/07/2025", replacement= "Use defineHttpConnection<InputType, AuthSchema, AuthAttributes>(authSchemaMapper, connectionProviderSchemaMapper, authenticate, attributes, options) instead")
fun defineHttpConnection<InputType, AuthSchema, AuthAttributes>(
    authSchemaMapper: (InputType) -> AuthSchema,
    connectionProviderSchemaMapper: (InputType) -> HttpConnectionProviderSchema,
    authenticate: (AuthSchema, AuthAttributes, HttpRequester) -> HttpRequester,
    attributes: AuthAttributes,
    extensions: (InputType) -> Array<ConnectionProviderExtensions | ConnectionProviderOAuthExtensions>
) = (parameter: InputType) -> do {
    var httpConnectionSchema = connectionProviderSchemaMapper(parameter)
    ---
    (httpRequest: HttpRequester) -> do {
        var authInfo = authenticate(
            authSchemaMapper(parameter),
            attributes,
            setBaseUri(httpConnectionSchema, httpRequest)
        )
        ---
        doRequest(applyExtensions(extensions(parameter), authInfo))
    }
}

/**
* Generic function to define a `connect` function for an HTTP connection
*
* === Parameters
*
* [%header, cols="1,1,3"]
* |===
* | Name | Type | Description
* | `authSchemaMapper` | &#40;InputType&#41; &#45;&#62; AuthSchema | mapping function from a connection schema to an authentication schema
* | `connectionProviderSchemaMapper` | &#40;InputType&#41; &#45;&#62; HttpConnectionProviderSchema | mapping function for a connection schema
* | `authenticate` | &#40;AuthSchema, HttpRequester&#41; &#45;&#62; HttpRequester | function to build an HTTP connection instance with authentication information
* | `options` | HttpConnectionOptions<InputType> | Additional options for the HTTP connection
* |===
*
*/
fun defineHttpConnection<InputType <: Object, AuthSchema, AuthAttributes>(
    authSchemaMapper: (InputType) -> AuthSchema,
    connectionProviderSchemaMapper: (InputType) -> HttpConnectionProviderSchema,
    authenticate: (AuthSchema, HttpRequester) -> HttpRequester,
    options: HttpConnectionOptions<InputType>
) = (parameter: InputType) -> do {
    var httpConnectionSchema = connectionProviderSchemaMapper(parameter)
    ---
    (httpRequest: HttpRequester) -> do {
        var authInfo = authenticate(
            authSchemaMapper(parameter),
            setBaseUri(httpConnectionSchema, httpRequest)
        )
        var requester = if (options.extensions?) applyExtensions(options.extensions(parameter), authInfo) else authInfo
        ---
        if(options.responseInterceptors?)
            intercept(options.responseInterceptors!, doRequest(requester))
        else
            doRequest(requester)
    }
}

/**
* Generic function to define a `connect` function for an HTTP connection
*
* === Parameters
*
* [%header, cols="1,1,3"]
* |===
* | Name | Type | Description
* | `authSchemaMapper` | &#40;InputType&#41; &#45;&#62; AuthSchema | mapping function from a connection schema to an authentication schema
* | `connectionProviderSchemaMapper` | &#40;InputType&#41; &#45;&#62; HttpConnectionProviderSchema | mapping function for a connection schema
* | `authenticate` | &#40;AuthSchema, AuthAttributes, HttpRequester&#41; &#45;&#62; HttpRequester | function to build an HTTP connection instance with authentication information
* | `attributes` | AuthAttributes | Additional attributes required for the authentication function
* | `options` | HttpConnectionOptions<InputType> | Additional options for the HTTP connection
* |===
*/
fun defineHttpConnection<InputType <: Object, AuthSchema, AuthAttributes>(
    authSchemaMapper: (InputType) -> AuthSchema,
    connectionProviderSchemaMapper: (InputType) -> HttpConnectionProviderSchema,
    authenticate: (AuthSchema, AuthAttributes, HttpRequester) -> HttpRequester,
    attributes: AuthAttributes,
    options: HttpConnectionOptions<InputType>
) = (parameter: InputType) -> do {
    var httpConnectionSchema = connectionProviderSchemaMapper(parameter)
    ---
    (httpRequest: HttpRequester) -> do {
        var authInfo = authenticate(
            authSchemaMapper(parameter),
            attributes,
            setBaseUri(httpConnectionSchema, httpRequest)
        )
        var requester = if (options.extensions?) applyExtensions(options.extensions(parameter), authInfo) else authInfo
        ---
        if(options.responseInterceptors?)
            intercept(options.responseInterceptors!, doRequest(requester))
        else
            doRequest(requester)
    }
}

/**
* Sets the base URI based on the HTTP connection provider schema defined by the authentication and the HTTP connection instance defined by the operation.
* The connection schema base URI is used in case the connection instance that the operation provides does not define one.
* 
* === Parameters
*
* [%header, cols="1,1,3"]
* |===
* | Name | Type | Description
* | `httpConnectionProviderSchema` | HttpConnectionProviderSchema | An HTTP connection provider schema
* | `httpRequest` | HttpRequester | An HTTP connection instance
* |===
*
*/
fun setBaseUri(httpConnectionProviderSchema: HttpConnectionProviderSchema, httpRequest: HttpRequester): HttpRequester =
    if (!(httpRequest.baseUri?) and httpConnectionProviderSchema.baseUri?)
        httpRequest update { case .baseUri! -> httpConnectionProviderSchema.baseUri! }
    else
        httpRequest

////////// Basic Authentication //////////

/**
* Type for HTTP connection descriptions with the _basic authentication schema_
*/
type BasicHttpConnectionProvider<InputType <: Object> = HttpConnectionProvider<InputType, {"type": "basic"}>

/**
* Base type for authentication schemas using RFC7617
*
* Fields:
* - `username`: The username to send
* - `password`: The password to send
*/
type BasicAuthSchema = {
    username: @SemanticTerms(value = ["username"]) String,
    password: @SemanticTerms(value = ["password"]) String
}

/**
* Creates an HTTP connection instance from a `BasicAuthSchema`
*
* === Parameters
*
* [%header, cols="1,1,3"]
* |===
* | Name | Type | Description
* | `value` | BasicAuthSchema | The credentials to use
* | `httpRequest` | HttpRequester | An HTTP connection instance
* |===
*
*/
fun basicAuth(value: BasicAuthSchema, httpRequest: HttpRequester): HttpRequester =  do {
    var base = toBase64("$(value.username):$(value.password)" as Binary {encoding: "UTF-8"})
    ---
    httpRequest update {
        case .headers.Authorization! -> "Basic $(base)"
    }
}

/**
* Defines an HTTP connection description that uses the _basic authentication schema_
*
* === Parameters
*
* [%header, cols="1,1,3"]
* |===
* | Name | Type | Description
* | `schemaMapper` | &#40;ConnectionSchemaType&#41; &#45;&#62; BasicAuthSchema | mapping function from a connection schema to a _basic authentication schema_
* | `connectionSchemaMapper` | &#40;ConnectionSchemaType&#41; &#45;&#62; HttpConnectionSchema | mapping function for a connection schema
* |===
*
*/
fun defineBasicHttpConnectionProvider<InputType <: Object>(
    schemaMapper: (InputType) -> BasicAuthSchema,
    connectionProviderSchemaMapper: (InputType) -> HttpConnectionProviderSchema
    ): BasicHttpConnectionProvider<InputType> = {
    connect: defineHttpConnection(schemaMapper, connectionProviderSchemaMapper, basicAuth),
    authenticationType: {"type": "basic"}
}

/**
* Defines an HTTP connection description that uses the _basic authentication schema_
*
* === Parameters
*
* [%header, cols="1,1,3"]
* |===
* | Name | Type | Description
* | `schemaMapper` | &#40;ConnectionSchemaType&#41; &#45;&#62; BasicAuthSchema | mapping function from a connection schema to a _basic authentication schema_
* | `connectionSchemaMapper` | &#40;ConnectionSchemaType&#41; &#45;&#62; HttpConnectionSchema | mapping function for a connection schema
* | `extensions` | (InputType) -> Array<ConnectionProviderExtensions> | function which returns a list of extensions from an input type
* |===
*
*/
@Deprecated(since= "05/07/2025", replacement= "Use defineBasicHttpConnectionProvider<InputType>(schemaMapper, connectionProviderSchemaMapper, options) instead")
fun defineBasicHttpConnectionProvider<InputType <: Object>(
    schemaMapper: (InputType) -> BasicAuthSchema,
    connectionProviderSchemaMapper: (InputType) -> HttpConnectionProviderSchema,
    extensions: (InputType) -> Array<ConnectionProviderExtensions>
    ): BasicHttpConnectionProvider<InputType> = {
    connect: defineHttpConnection(schemaMapper, connectionProviderSchemaMapper, basicAuth, extensions),
    authenticationType: {"type": "basic"},
    extensions: extensions
}

/**
* Generic function to define a `connect` function for an HTTP connection
*
* === Parameters
*
* [%header, cols="1,1,3"]
* |===
* | Name | Type | Description
* | `schemaMapper` | &#40;InputType&#41; &#45;&#62; BasicAuthSchema | mapping function from a connection schema to a _basic authentication schema_
* | `connectionProviderSchemaMapper` | &#40;InputType&#41; &#45;&#62; HttpConnectionProviderSchema | mapping function for a connection schema
* | `options` | HttpConnectionOptions<InputType> | Additional options for the HTTP connection
* |===
*
*/
fun defineBasicHttpConnectionProvider<InputType <: Object>(
    schemaMapper: (InputType) -> BasicAuthSchema,
    connectionProviderSchemaMapper: (InputType) -> HttpConnectionProviderSchema,
    options: HttpConnectionOptions<InputType>
    ): BasicHttpConnectionProvider<InputType> = {
    connect: defineHttpConnection(schemaMapper, connectionProviderSchemaMapper, basicAuth, options),
    authenticationType: {"type": "basic"},
    (extensions: options.extensions!) if options.extensions?
}

////////// Bearer Authentication //////////

/**
* Type for HTTP connection descriptions with the _bearer authentication schema_
*/
type BearerHttpConnectionProvider<InputType <: Object> = HttpConnectionProvider<InputType, {"type": "bearer"}>

/**
* Base type for authentication schemas using RFC6750
*
* Fields:
* - `token`: The token to send
*/
type BearerAuthSchema = {
    token: @SemanticTerms(value = ["secretToken", "password"]) String
}

/**
* Creates an HTTP connection instance from a `BearerAuthSchema`
*
* === Parameters
*
* [%header, cols="1,1,3"]
* |===
* | Name | Type | Description
* | `value` | BearerAuthSchema | The credentials to use
* | `httpRequest` | HttpRequester | An HTTP connection instance
* |===
*
*/
fun bearerAuth(value: BearerAuthSchema, httpRequest: HttpRequester): HttpRequester =
    httpRequest update {
        case .headers.Authorization! -> "Bearer $(value.token)"
    }

/**
* Defines an HTTP connection description that uses the _bearer authentication schema_
*
* === Parameters
*
* [%header, cols="1,1,3"]
* |===
* | Name | Type | Description
* | `schemaMapper` | &#40;ConnectionSchemaType&#41; &#45;&#62; BearerAuthSchema | mapping function from a connection schema to a _bearer authentication schema_
* | `connectionSchemaMapper` | &#40;ConnectionSchemaType&#41; &#45;&#62; HttpConnectionSchema | mapping function for a connection schema
* |===
*
*/
fun defineBearerHttpConnectionProvider<InputType <: Object>(
    schemaMapper: (InputType) -> BearerAuthSchema,
    connectionSchemaProviderMapper: (InputType) -> HttpConnectionProviderSchema
    ): BearerHttpConnectionProvider<InputType> = {
    connect: defineHttpConnection(schemaMapper, connectionSchemaProviderMapper, bearerAuth),
    authenticationType: {"type": "bearer"}
}

/**
* Defines an HTTP connection description that uses the _bearer authentication schema_
*
* === Parameters
*
* [%header, cols="1,1,3"]
* |===
* | Name | Type | Description
* | `schemaMapper` | &#40;ConnectionSchemaType&#41; &#45;&#62; BearerAuthSchema | mapping function from a connection schema to a _bearer authentication schema_
* | `connectionSchemaMapper` | &#40;ConnectionSchemaType&#41; &#45;&#62; HttpConnectionSchema | mapping function for a connection schema
* | `extensions` | (InputType) -> Array<ConnectionProviderExtensions> | function which returns a list of extensions from an input type
* |===
*
*/
@Deprecated(since= "05/07/2025", replacement= "Use defineBearerHttpConnectionProvider<InputType>(schemaMapper, connectionProviderSchemaMapper, options) instead")
fun defineBearerHttpConnectionProvider<InputType <: Object>(
    schemaMapper: (InputType) -> BearerAuthSchema,
    connectionSchemaProviderMapper: (InputType) -> HttpConnectionProviderSchema,
    extensions: (InputType) -> Array<ConnectionProviderExtensions>
    ): BearerHttpConnectionProvider<InputType> = {
    connect: defineHttpConnection(schemaMapper, connectionSchemaProviderMapper, bearerAuth, extensions),
    authenticationType: {"type": "bearer"},
    extensions: extensions
}

/**
* Generic function to define a `connect` function for an HTTP connection
*
* === Parameters
*
* [%header, cols="1,1,3"]
* |===
* | Name | Type | Description
* | `schemaMapper` | &#40;InputType&#41; &#45;&#62; BearerAuthSchema | mapping function from a connection schema to a _bearer authentication schema_
* | `connectionProviderSchemaMapper` | &#40;InputType&#41; &#45;&#62; HttpConnectionProviderSchema | mapping function for a connection schema
* | `options` | HttpConnectionOptions<InputType> | Additional options for the HTTP connection
* |===
*
*/
fun defineBearerHttpConnectionProvider<InputType <: Object>(
    schemaMapper: (InputType) -> BearerAuthSchema,
    connectionSchemaProviderMapper: (InputType) -> HttpConnectionProviderSchema,
    options: HttpConnectionOptions<InputType>
    ): BearerHttpConnectionProvider<InputType> = {
    connect: defineHttpConnection(schemaMapper, connectionSchemaProviderMapper, bearerAuth, options),
    authenticationType: {"type": "bearer"},
    (extensions: options.extensions!) if options.extensions?
}


////////// API Key Authentication //////////

/**
* Type for HTTP connection descriptions with the _api key authentication schema_
*/
type ApiKeyHttpConnectionProvider<InputType <: Object> = HttpConnectionProvider<InputType, {"type": "apiKey"}>

/**
* Base type for authentication schemas using API Key
*
* Fields:
* - `apiKey`: The API Key to send
*/
type ApiKeyAuthSchema = {|
    apiKey: @SemanticTerms(value = ["apiKey"]) String
|}

/**
* Type to represent an API Key attribute
*
* Fields:
* - `in`: HTTP connection parameter type where the API Key attribute is targeted to be present
* - `name`: Attribute name
*/
type ApiKeyAuthAttributesSchema = {|
    in: 'header' | 'cookie' | 'query',
    name: String
|}

/**
* Creates an HTTP connection instance from a `ApiKeyAuthSchema`
*
* === Parameters
*
* [%header, cols="1,1,3"]
* |===
* | Name | Type | Description
* | `value` | ApiKeyAuthSchema | The credentials to use
* | `attributes` | ApiKeyAuthAttributesSchema | attributes information where the API Key value should be located
* | `httpRequest` | HttpRequester | An HTTP connection instance
* |===
*
*/
fun apiKeyAuth(value: ApiKeyAuthSchema, attributes: ApiKeyAuthAttributesSchema, httpRequest: HttpRequester): HttpRequester =
    attributes.in match {
         case 'header' -> httpRequest update {
            case .headers."$(attributes.name)"! -> value.apiKey
         }
         case 'cookie' -> httpRequest update {
            case cookies at .cookie! -> (cookies default {}) update {
                case .`$(attributes.name)`! -> value.apiKey
            }
         }
         case 'query' -> httpRequest update {
            case .queryParams."$(attributes.name)"! -> value.apiKey
         }
}

/**
* Defines an HTTP connection description that uses the _api key authentication schema_
*
* === Parameters
*
* [%header, cols="1,1,3"]
* |===
* | Name | Type | Description
* | `schemaMapper` | &#40;ConnectionSchemaType&#41; &#45;&#62; ApiKeyAuthSchema |  mapping function from a connection schema to a _bearer authentication schema_
* | `connectionSchemaMapper` | &#40;ConnectionSchemaType&#41; &#45;&#62; HttpConnectionSchema | mapping function for a connection schema
* | `attributes` | ApiKeyAuthAttributesSchema | attributes information where the API Key value should be located
* |===
*
*/
fun defineApiKeyHttpConnectionProvider<InputType <: Object>(
    schemaMapper: (InputType) -> ApiKeyAuthSchema,
    connectionProviderSchemaMapper: (InputType) -> HttpConnectionProviderSchema,
    attributes: ApiKeyAuthAttributesSchema
    ): ApiKeyHttpConnectionProvider<InputType> = {
    connect: defineHttpConnection(schemaMapper, connectionProviderSchemaMapper, apiKeyAuth, attributes),
    authenticationType: {"type": "apiKey", attributes: attributes}
}

/**
* Defines an HTTP connection description that uses the _api key authentication schema_
*
* === Parameters
*
* [%header, cols="1,1,3"]
* |===
* | Name | Type | Description
* | `schemaMapper` | &#40;ConnectionSchemaType&#41; &#45;&#62; ApiKeyAuthSchema |  mapping function from a connection schema to a _bearer authentication schema_
* | `connectionSchemaMapper` | &#40;ConnectionSchemaType&#41; &#45;&#62; HttpConnectionSchema | mapping function for a connection schema
* | `attributes` | ApiKeyAuthAttributesSchema | attributes information where the API Key value should be located
* | `extensions` | (InputType) -> Array<ConnectionProviderExtensions> | function which returns a list of extensions from an input type
* |===
*
*/
@Deprecated(since= "05/07/2025", replacement= "Use defineApiKeyHttpConnectionProvider<InputType>(schemaMapper, connectionProviderSchemaMapper, attributes, options) instead")
fun defineApiKeyHttpConnectionProvider<InputType <: Object>(
    schemaMapper: (InputType) -> ApiKeyAuthSchema,
    connectionProviderSchemaMapper: (InputType) -> HttpConnectionProviderSchema,
    attributes: ApiKeyAuthAttributesSchema,
    extensions: (InputType) -> Array<ConnectionProviderExtensions>
    ): ApiKeyHttpConnectionProvider<InputType> = {
    connect: defineHttpConnection(schemaMapper, connectionProviderSchemaMapper, apiKeyAuth, attributes, extensions),
    authenticationType: {"type": "apiKey", attributes: attributes},
    extensions: extensions
}

/**
* Generic function to define a `connect` function for an HTTP connection
*
* === Parameters
*
* [%header, cols="1,1,3"]
* |===
* | Name | Type | Description
* | `schemaMapper` | &#40;InputType&#41; &#45;&#62; ApiKeyAuthSchema | mapping function from a connection schema to a _bearer authentication schema_
* | `connectionProviderSchemaMapper` | &#40;InputType&#41; &#45;&#62; HttpConnectionProviderSchema | mapping function for a connection schema
* | `attributes` | ApiKeyAuthAttributesSchema | attributes information where the API Key value should be located
* | `options` | HttpConnectionOptions<InputType> | Additional options for the HTTP connection
* |===
*
*/
fun defineApiKeyHttpConnectionProvider<InputType <: Object>(
    schemaMapper: (InputType) -> ApiKeyAuthSchema,
    connectionProviderSchemaMapper: (InputType) -> HttpConnectionProviderSchema,
    attributes: ApiKeyAuthAttributesSchema,
    options: HttpConnectionOptions<InputType>
    ): ApiKeyHttpConnectionProvider<InputType> = {
    connect: defineHttpConnection(schemaMapper, connectionProviderSchemaMapper, apiKeyAuth, attributes, options),
    authenticationType: {"type": "apiKey", attributes: attributes},
    (extensions: options.extensions!) if options.extensions?
}

////////// OAuth 2.0 Authentication //////////

/**
* Type for HTTP connection descriptions with the oauth authentication schema_
*/
type OAuth2HttpConnection<InputType <: Object> = HttpConnectionProvider<InputType, {"type": "oauth2"}>

/**
* Type for all OAuth supported types
*/
type GrantType = 'authorizationCode' | 'clientCredentials' | 'password' | 'implicit'

/**
* Type to represent base OAuth attribute for all gran types
*/
type CommonOAuth2GrantTypeSchema<GrantType> = {|
    "grantType": GrantType,
    refreshUrl?: String,
    scopes: Array<String>
|}
/**
* Type to represent authorizationCode OAuth attribute
*/
type AuthorizationCodeGrantType = CommonOAuth2GrantTypeSchema<'authorizationCode'>  & {|
    authorizationUrl: String,
    tokenUrl: String
|}
/**
* Type to represent clientCredentials OAuth attribute
*/
type ClientCredentialsGrantType = CommonOAuth2GrantTypeSchema<'clientCredentials'> & {|
    tokenUrl: String
|}
/**
* Type to represent password OAuth attribute
*/
type PasswordGrantType = CommonOAuth2GrantTypeSchema<'password'> & {|
    tokenUrl: String
|}
/**
* Type to represent implicit OAuth attribute
*/
type ImplicitGrantType = CommonOAuth2GrantTypeSchema<'implicit'>  & {|
    authorizationUrl: String
|}

/**
* Type to represent an OAuth attribute
*
* Fields depends on grant type
*/
type OAuth2Schema<OAuthGranType <: AuthorizationCodeGrantType | ClientCredentialsGrantType | PasswordGrantType | ImplicitGrantType > = OAuthGranType

/**
* Base type for authentication schemas using OAuth
*
* Fields:
* - `accessToken`: The access token gotten from dance mechanism
*/
type OAuth2AuthSchema = {|
    accessToken: @SemanticTerms(value = ["secretToken", "password"]) String
|}

/**
* Creates an HTTP connection instance from a `OAuth2AuthSchema`
*
* === Parameters
*
* [%header, cols="1,1,3"]
* |===
* | Name | Type | Description
* | `value` | OAuth2AuthSchema | The credentials to use
* | `httpRequest` | HttpRequester | An HTTP connection instance
* |===
*
*/
fun oAuth2Auth(value: OAuth2AuthSchema, httpRequest: HttpRequester): HttpRequester =
    httpRequest update {
        case .headers.Authorization! -> "Bearer $(value.accessToken)"
    }

/**
* Defines an HTTP connection description that uses the OAuth key authentication schema_
*
* === Parameters
*
* [%header, cols="1,1,3"]
* |===
* | Name | Type | Description
* | `schemaMapper` | &#40;ConnectionSchemaType&#41; &#45;&#62; OAuth2AuthSchema |  mapping function from a connection schema to a _bearer authentication schema_
* | `connectionSchemaMapper` | &#40;ConnectionSchemaType&#41; &#45;&#62; HttpConnectionProviderSchema | mapping function for a connection schema
* | `attributes` | OAuth2Schema | attributes information where the OAuth authentication information should be located
* |===
*
*/
fun defineOAuth2Connection<InputType <: Object>(
    schemaMapper: (InputType) -> OAuth2AuthSchema,
    connectionSchemaMapper: (InputType) -> HttpConnectionProviderSchema,
    attributes: OAuth2Schema
    ): OAuth2HttpConnection<InputType> = {
    connect: defineHttpConnection(schemaMapper, connectionSchemaMapper, oAuth2Auth),
    authenticationType: {"type": "oauth2", attributes: attributes}
}

/**
* Defines an HTTP connection description that uses the OAuth key authentication schema_
*
* === Parameters
*
* [%header, cols="1,1,3"]
* |===
* | Name | Type | Description
* | `schemaMapper` | &#40;ConnectionSchemaType&#41; &#45;&#62; OAuth2AuthSchema |  mapping function from a connection schema to a _bearer authentication schema_
* | `connectionSchemaMapper` | &#40;ConnectionSchemaType&#41; &#45;&#62; HttpConnectionProviderSchema | mapping function for a connection schema
* | `attributes` | OAuth2Schema | attributes information where the OAuth authentication information should be located
* | `extensions` | (InputType) -> Array<ConnectionProviderExtensions> | function which returns a list of extensions from an input type
* |===
*
*/
@Deprecated(since= "05/07/2025", replacement= "Use defineOAuth2Connection<InputType>(schemaMapper, connectionProviderSchemaMapper, attributes, options) instead")
fun defineOAuth2Connection<InputType <: Object>(
    schemaMapper: (InputType) -> OAuth2AuthSchema,
    connectionSchemaMapper: (InputType) -> HttpConnectionProviderSchema,
    attributes: OAuth2Schema,
    extensions: (InputType) -> Array<ConnectionProviderOAuthExtensions>
    ): OAuth2HttpConnection<InputType> = {
    connect: defineHttpConnection(schemaMapper, connectionSchemaMapper, oAuth2Auth, extensions),
    authenticationType: {"type": "oauth2", attributes: attributes},
    extensions: extensions
}

/**
* Generic function to define a `connect` function for an HTTP connection
*
* === Parameters
*
* [%header, cols="1,1,3"]
* |===
* | Name | Type | Description
* | `schemaMapper` | &#40;InputType&#41; &#45;&#62; OAuth2AuthSchema | mapping function from a connection schema to a _bearer authentication schema_
* | `connectionSchemaMapper` | &#40;InputType&#41; &#45;&#62; HttpConnectionProviderSchema | mapping function for a connection schema
* | `attributes` | OAuth2Schema | attributes information where the OAuth authentication information should be located
* | `options` | HttpConnectionOptions<InputType> | Additional options for the HTTP connection
* |===
*
*/
fun defineOAuth2Connection<InputType <: Object>(
    schemaMapper: (InputType) -> OAuth2AuthSchema,
    connectionSchemaMapper: (InputType) -> HttpConnectionProviderSchema,
    attributes: OAuth2Schema,
    options: HttpConnectionOptions<InputType>
    ): OAuth2HttpConnection<InputType> = {
    connect: defineHttpConnection(schemaMapper, connectionSchemaMapper, oAuth2Auth, options),
    authenticationType: {"type": "oauth2", attributes: attributes},
    (extensions: options.extensions!) if options.extensions?
}

/******************************************************************************************************************
* Operation
*******************************************************************************************************************/
/**
* Shape all HTTP Requests should have
*/
type BaseHttpRequestType = {|
    /*
    * An object containing the name -> value bindings for URI parameters
    */
    uri?: {},
    /*
    * An object containing the name -> value bindings for query parameters
    */
    query: {},
    /*
    * An object containing the name -> value bindings for http headers
    */
    headers: {},
    /*
    * An object containing the name -> value bindings for http cookie header
    */
    cookie: {},
    /*
    * A value to be encoded as body of the request
    */
    body?: Any
|}

/**
* Given a type T checks that it describes a valid HTTP request
*
* See BaseHttpRequestType for a description of what HTTP requests look like
*/
type HttpRequestType<T <: BaseHttpRequestType> = T

/**
* An empty HTTP request
* Given a type T checks that it describes a valid HTTP request
*
* Useful as a parameter for _"validate connection"_ operations
* See BaseHttpRequestType for a description of what HTTP requests look like
*/
var emptyRequest: HttpRequestType = {
    query: {},
    headers: {},
    cookie: {}
}

/**
* Type for HTTP operations
*
* Has the same shape as a regular `Operation` but its instances are required
* to have subtypes of `HttpResponse` as its success and failure types and an
* `HttpConnection` for its connection instances
*/
type HttpOperation<HttpRequestInput <: HttpRequestType, ResponseSuccessBodyType, ResponseFailureBodyType, HttpConnectionType <: HttpConnection> =
    Operation<HttpRequestInput, HttpResponse<ResponseSuccessBodyType>, ResultFailure<HttpResponse<ResponseFailureBodyType>, Error>, HttpConnectionType>

/******************************************************************************************************************
* Utils
*******************************************************************************************************************/
/**
* Parses the query parameters from a given URI string
*
* === Parameters
*
* [%header, cols="1,1,3"]
* |===
* | Name | Type | Description
* | `uri` | String | The URI to parse
* |===
*
*/
fun parseQueryParameters(uri: String): { _?: String | Array | Null} = do {
  var firstPart = if (uri contains "?") ((uri splitBy "&")[0] splitBy "?")[1] else null
  var remainingParts = if ( (uri splitBy "&")[1 to -1] != null and !isEmpty( (uri splitBy "&")[1 to -1]))  (uri splitBy "&")[1 to -1] else []
  var queryParameters = (if (firstPart != null) [firstPart] else []) ++ remainingParts
            reduce ((queryLine, accumulator={}) ->
                accumulator ++
                do
                {
                    var queryParam = queryLine splitBy "="
                    ---
                    (queryParam[0]): if (sizeOf(queryParam) > 1 and queryParam[1] != "") decodeURI(queryParam[1]) else null
                }
            )
        groupBy $$ mapObject (($$): if (sizeOf($) > 1) ($ pluck $) else $[keysOf($)[0]])
        ---
        queryParameters filterObject ((value, key, index) -> !isEmpty(key)) default {}
}

/**
* Generates the value for cookie header from an object
*
* === Parameters
*
* [%header, cols="1,1,3"]
* |===
* | Name | Type | Description
* | `cookie` | Object | The cookie object to be transformed to a cookie header: {Cookie: key1=value;key2=value2}
* |===
*
*/
    fun getCookieHeader(cookie: Object): Object = do {
        // FIXME: Properly escape keys and values
        var cookieValue=((keysOf(cookie default {}) map (elem, idx) -> (elem as String) ++ "=" ++ (cookie[elem] default ""))
                                                                                                  reduce (elem, finalCookie:String) -> finalCookie ++ ";" ++ elem as String) default null
        ---
        if(cookieValue != null)
            {Cookie: cookieValue}
        else
            {}
    }
