%dw 2.0

/**
*
* The `unpack` function unpacks a Protobuf Any into an actual DataWeave object.
* In order to do this, it needs the url for the compiled descriptor where the Any `type_url`
* will be found.
*
* === Parameters
*
* [%header, cols="1,1,3"]
* |===
* | Name | Type | Description
* | `msg` | { type_url : String, value : Binary } | The Protobuf Any object
* | `descriptorUrl` | String | The URL where the compiled descriptor can be found
* |===
*
* === Example
*
* This example shows how the `unpack` function behaves assuming the in0 input has a payload
* with the `Payload` type specified in the schema below, and where the `load` field has a packed `Range` message.
*
* ==== Schema
*
* [source,ProtoBuf,linenums]
* ----
* syntax = "proto3";
*
* package engine.anyPacking;
*
* import "google/protobuf/any.proto";
*
* message Payload {
*   bool flag = 1;
*   google.protobuf.Any load = 2;
* }
*
* message Range {
*   int32 from = 1;
*   int32 to = 2;
* }
* ----
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* input in0 application/x-protobuf messageType='engine.anyPacking.Payload',descriptorUrl="example.dsc"
* output json
* import unpack from protobuf::Any
*
* ---
* unpack(in0.load, "example.dsc")
* ----
*
* ==== Output
*
* [source,Json,linenums]
* ----
* {
*   "from": 1.0,
*   "to": 3.0
* }
* ----
**/
fun unpack(msg: { type_url : String, value : Binary }, descriptorUrl: String): Any = read(
  msg.value,
  "application/protobuf",
  { descriptorUrl: descriptorUrl, messageType: msg.type_url}
)

/**
*
* The `pack` function serializes an object and returns it as a Protobuf Any message.
* It needs the `messageType` and the `descriptorUrl` in order to properly pack the message,
* which can be of `Any` value.
*
* === Parameters
*
* [%header, cols="1,1,3"]
* |===
* | Name | Type | Description
* | `msg` | Any | The object to be serialized
* | `messageType` | String | The protobuf message type to use
* | `descriptorUrl` | String | The URL to find the compiled descriptor
* |===
*
* === Example
* This example shows how the `pack` function behaves assuming the output will be written as a
* protobuf message with the `Payload` type of the specified schema.
* The output will be a serialized protobuf message, not shown here.
*
* ==== Schema
*
* [source,ProtoBuf,linenums]
* ----
* syntax = "proto3";
*
* package engine.anyPacking;
*
* import "google/protobuf/any.proto";
*
* message Payload {
*   bool flag = 1;
*   google.protobuf.Any load = 2;
* }
*
* message Range {
*   int32 from = 1;
*   int32 to = 2;
* }
* ----
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* output application/x-protobuf messageType='engine.anyPacking.Payload',descriptorUrl="example.dsc"
* import unpack from protobuf::Any
*
* ---
* {
*   flag: true,
*   load: pack({from: 1, to: 3}, 'engine.anyPacking.Range', "descriptors/test.dsc")
* }
* ----
**/
fun pack(msg: Any,  messageType: String, descriptorUrl: String): { type_url : String, value : Binary } = {
  type_url: messageType,
  value: write(msg, "application/protobuf",{ descriptorUrl: descriptorUrl, messageType: messageType}) as Binary
}