/*
 * Copyright (c) 2025, Salesforce, Inc.,
 * All rights reserved.
 * For full license text, see the LICENSE.txt file
 */
/**
* This module describes the base Connectivity concepts.
*/
%dw 2.7
import * from dw::core::Types
import fail from dw::Runtime

/******************************************************************************************************************
 * Connections
 *******************************************************************************************************************/
/**
* A type that describes a test connection validation result
*
* Fields:
* - `isValid`: true/false defines if the connection validation was success or failure
* - `message`: An optional String field that in case of isValid:false has a description of the validation error
* - `error`: An optional String field that in case of isValid:false has the error cause.
*/
type ConnectionValidationResult = {|
    isValid: Boolean,
    message?: String,
    error?: String
|}

/**
* A type that describes a global test connection function
* It does not depend on a connection instance or operation, it just receives a connection to validate it.
* It is intended to be unique.
*/
type TestConnection<ConnectionType <: Connection> = {
	validate: (ConnectionType) -> ConnectionValidationResult
}

/**
* A type that describes a connection which is a function that given a type parameter `ConnectionRequest` sends its
* value to a target system and generates a type parameter `ConnectionResponse`.
*/
type Connection<ConnectionRequest <: Object, ConnectionResponse <: Object> = (ConnectionRequest) -> ConnectionResponse

/**
* A type that describes a connection provider.
*
* Fields:
* - `connect`: Function that receives the `InputType` and returns a `Connection` to allow components to make
* invocations through the connection
* - `authenticationType`: Describes the type of authentication used by the connection.
* - `validate`: Optional field that defines an specific test connection function for connections generated by this `ConnectionProvider`.
*/
type ConnectionProvider<ConnectionType <: Connection, InputType <: Object, AuthenticationType <: Object> = {
    connect: (InputType) -> ConnectionType,
    authenticationType: AuthenticationType,
    validate?: (ConnectionType) -> ConnectionValidationResult
}

/**
* Creates a connection validator function from a given operation and assertion
*
* === Parameters
*
* [%header, cols="1,1,3"]
* |===
* | Name | Type | Description
* | `validateOperation` | ValidateOperation | The operation to execute using the connection parameters
* | `validateMapper` | &#40;Result<ResultType, ResultErrorType&#62;&#41; &#45;&#62; ConnectionValidationResult | The assertion that checks whether the invocation was successful or not
* |===
*/
fun defineTestConnection<ConnectionType, ResultType, ResultErrorType <: ResultFailure>(
    validateOperation: Operation<{||}, ResultType, ResultErrorType, ConnectionType>,
    validateMapper: (Result<ResultType, ResultErrorType>) -> ConnectionValidationResult
): (ConnectionType) -> ConnectionValidationResult = (connection: ConnectionType) -> validateMapper(validateOperation.executor({}, connection))

/******************************************************************************************************************
 * Operation
 *******************************************************************************************************************/

type Executor<InputType <: Object, ResultType, ResultErrorType <: ResultFailure, ConnectionType <: Connection> = (InputType, ConnectionType) -> Result<ResultType, ResultErrorType>

/**
* A type that describes an operation
*
* Fields:
* - `name`: A connector-unique string that identifies this operation
* - `displayName`: A user-visible name for this operation
* - `executor`: A function that executes this operation when invoked
*/
type Operation<InputType <: Object, ResultType, ResultErrorType <: ResultFailure, ConnectionType <: Connection> = {
    name: String,
    displayName: String,
    executor: Executor<InputType, ResultType, ResultErrorType, ConnectionType>
}

type PaginatedOperation<InputType <: Object, NextPageInputType <: Object, ResultType, ResultErrorType <: ResultFailure, ConnectionType <: Connection> = Operation<InputType, ResultType, ResultErrorType, ConnectionType> & {
    nextPageExecutor: Executor<NextPageInputType, ResultType, ResultErrorType, ConnectionType>
}

/******************************************************************************************************************
 * Pagination
 *******************************************************************************************************************/
/**
* A type that represents a pagination response.
*
* Fields:
* - `value`: The raw result of the original operation
* - `items`: An array that includes every result for this page
* - `nextPage`: An optional field. If present it represent the parameterization required for the next page
*
* To get the next page just call this same operation again with the value from `nextPage.args` as the input
* parameter.
*/
type Page<ItemType, InputType <: Object> = {
    /**
    * Items from this page
    */
    items: Array<ItemType>,
    nextPage?: {|
        /**
        * Input parameter for the next page.
        */
        args: InputType
    |}
}

/******************************************************************************************************************
 * Trigger
 *******************************************************************************************************************/

/**
* A type that describes a trigger
*
* Fields:
* - `name`: A connector-unique string that identifies this trigger
* - `displayName`: A user-visible name for this trigger
* - `metadata`: A no user-visible values which describe if the trigger is ordered and if it's paginated
* - `strategy` | TriggerStrategy<OperationOutputType,TriggerResulItemType,OperationResultItemType,Watermark> | Trigger extraction strategy
* - `operation` | Operation<OperationInputType, OperationOutputType, ResultErrorType, ConnectionType> OR
*                   PaginatedOperation<OperationInputType, OperationNextPageInputType, OperationOutputType, ResultErrorType, ConnectionType>
*               | Operation which the trigger is based on to execute requests
* - `inputMapper` | (TriggerInputType, Watermark | Null) -> OperationInputType | Function which maps the input trigger parameters
*/
type Trigger<TriggerInputType, OperationInputType, OperationOutputType, OperationNextPageInputType, ResultErrorType <: ResultFailure, ConnectionType <: Connection, TriggerResultItemType, OperationResultItemType, Watermark> = {
    name: String,
    displayName: String,
    metadata?: {
        order?: "ASC" | "DESC" | "NO_ORDER",
        paginated?: true|false
    },
    strategy: TriggerStrategy<OperationOutputType, TriggerResultItemType, OperationResultItemType, Watermark>,
    operation: Operation<OperationInputType, OperationOutputType, ResultErrorType, ConnectionType> |
    PaginatedOperation<OperationInputType, OperationNextPageInputType, OperationOutputType, ResultErrorType, ConnectionType>,
    inputMapper: (TriggerInputType, Watermark) -> OperationInputType,
    initialWatermark: (TriggerInputType, ConnectionType) -> Watermark
}

/**
* A type that describes the trigger item object
*
* Fields:
* - `value`: The trigger item value with all necessary fields defined
* - `watermark`: The watermark for each item
* - `identity`: The identity for each item
*/
type TriggerItem<TriggerResulItemType, Watermark> = {
    value: TriggerResulItemType,
    watermark: Watermark,
    identity?: String
}

/**
* A type that describes the trigger extraction strategy
*
* Fields:
* - `items`: A function to get the array of items
* - `item`: A function to get each item structure, it could involve a transformation to the final item value
* - `watermark`: A function to get each item watermark
* - `identity`: A function to get each item identity
* - `compareWatermark`: A function that compares watermarks. It must returns 0 if the previous value is equal to the new value.
*                       A value less than 0 is returned if the previous value is lower than the new value and
*                       a value greater than 0 if the previous value is greater than the new value.
*/
type TriggerStrategy<OperationOutputType,TriggerResultItemType, OperationResultItemType, Watermark> = {
     // Note that if it's a paginated operation, OperationOutputType must be a
     // Page<OperationResultItemType,OperationInputType> and items will probably gotten as result.items.
     // In case of a non paginated operation, OperationOutputType must be a HttpResponse<OperationOutputType> and
     // items will probably gotten as result.body."items"
    items: (OperationOutputType) -> Array<OperationResultItemType>,
    item: (OperationResultItemType) -> TriggerResultItemType,
    watermark: (OperationOutputType,OperationResultItemType) -> Watermark,
    identity: (OperationResultItemType) -> String,
    watermarkCompareTo: (Watermark,Watermark) -> Number
}


/**
* A type that describes the trigger next poll information
*
* Fields:
* - `latestWatermark`: Last watermark processed in previous poll
* - `greatestWatermark`: The greatest watermark gotten from the page or previous partialGreatestWatermark from the input
*/
type NextPoll<Watermark> = {
        latestWatermark: Watermark,
        greatestWatermark: Watermark
}

/**
* A type that describes the trigger output page
*
* Fields:
* - `page`: A regular page structure
* - `NextPoll`: Next poll information structure
*/
type TriggerPage<OperationInputType, TriggerResulItemType, Watermark> =
Page<TriggerItem<TriggerResulItemType,Watermark>, OperationInputType> & {nextPoll: NextPoll<Watermark>}
/******************************************************************************************************************
 * Execution
 *******************************************************************************************************************/

 /**
 * Type that represents an Error definition
 * Fields :
 *  `kind` : unique identifier
 *  `categories` : to group the errors
*/
 type Error<Kind <: String, Categories <: String> = {
     kind: Kind,
     categories: Array<Categories>
 }

/**
* An error condition that wraps another error condition. It allows representing
* open error sets while still presenting a closed world of possible ResultFailures
* for a given operation.
*/
type UncheckedError<Category <: String> = {
    kind: "__UNCHECKED",
    categories: Array<Category>,
    cause: Error<String, String>
}

/**
* An unexpected error condition, including those not documented by the connector model
*/
type UnexpectedError = UncheckedError<"__UNEXPECTED">

/**
* Type that represents a successful invocation
*
* Fields:
* - `success`: Always `true`. Signals this invocation was successful
* - `value`: The result of this invocation
*/
type ResultSuccess<ResultType> = {|
   success: true,
   value: ResultType
|}

/**
* Generates the response from a successful invocation
*
* === Parameters
*
* [%header, cols="1,1,3"]
* |===
* | Name | Type | Description
* | `value` | ResultType | The result of the invocation
* |===
*
*/
fun success<ResultType>(value: ResultType): ResultSuccess<ResultType> = {
    success: true,
    value: value
}

/**
* Type that represents a failed invocation
*
* Fields:
* - `success`: Always `false`. Signals this invocation failed
* - `error` : containing the error value and error type instance
*/
type ResultFailure<ErrorValueType, ErrorType <: Error> = {|
   success: false,
   error: {
      description: String | Null,
      value: ErrorValueType
   } & ErrorType
|}

/**
* Generates the response from a failed invocation, with empty kind and categories
*
* === Parameters
*
* [%header, cols="1,1,3"]
* |===
* | Name | Type | Description
* | `value` | ResultType |  The error that was found
* | `errorDescription` | String &#124; Null | Optional. A description for the error
* |===
*
*/
fun failure<ResultErrorValueType>(value: ResultErrorValueType, errorDescription: String | Null = null): ResultFailure<ResultErrorValueType, Error> = {
    success: false,
    error: {
        description: errorDescription,
        value: value,
        kind: "",
        categories: []
    }
}

/**
* Generates the response from a failed invocation
*
* === Parameters
*
* [%header, cols="1,1,3"]
* |===
* | Name | Type | Description
* | `value` | ResultType |  The error that was found
* | `err` | ErrorType |  The error type instance
* | `errorDescription` | String &#124; Null | Optional. A description for the error
* |===
*
*/
fun failure<ResultErrorValueType, ErrorType <: Error>(value : ResultErrorValueType, err : ErrorType, description: String | Null = null) : ResultFailure<ResultErrorValueType, ErrorType> = {
    success: false,
    error: {
        description: description,
        value: value
    } ++ err
}

/**
* Generates the response from a failed invocation whose error is not explicitly declared at the link-weave model
*
* === Parameters
*
* [%header, cols="1,1,3"]
* |===
* | Name | Type | Description
* | `value` | T |  The error that was found
* | `cause` | Error<String, String> |  The error type instance to be wrapped
* | `category` | String | A category to distinguish this unchecked error from others
* |===
*
*/
fun uncheckedFailure<T, Category <: String>(value: T, cause: Error<String, String>, category: Category): ResultFailure<T, UncheckedError<Category>> = {
  success: false,
  error: {
    kind: "__UNCHECKED",
    description: "An unchecked error occurred. Check the `cause` field for more details",
    categories: [category],
    value: value,
    cause: cause
  }
}

/**
* Generates the response from a failed invocation that resulted in an unexpected error
*
* === Parameters
*
* [%header, cols="1,1,3"]
* |===
* | Name | Type | Description
* | `value` | T |  The error that was found
* | `cause` | Error<String, String> |  The error type instance to be wrapped
* |===
*
*/
fun unexpectedFailure<T>(value: T, cause: Error<String, String>): ResultFailure<T, UnexpectedError> = uncheckedFailure(value, cause, "__UNEXPECTED")

/**
* Type that represents invocation results, may be either a `ResultSuccess` or a `ResultFailure`
*
* Fields:
* - `success`: Signals if this invocation was successful
* - `value`: The result of the invocation (an instance of either the `SuccessType` or the `FailureType`)
* - `description`: Optional. If the invocation was a failure this field may contain a description of the issue
*/
type Result<ResultSuccessType, ResultFailureType <: ResultFailure> = ResultSuccess<ResultSuccessType> | ResultFailureType

/**
 * Obtains the value of a successful result or executes a lambda if the passed result represents a failure.
 */
fun orElse<T>(result: Result<T, ResultFailure>, elseFunction: () -> T): T = result match {
    case is ResultSuccess<T> -> $.value
    else -> elseFunction()
}

/**
 * Obtains the value of a successful result or fails if the passed result represents a failure.
 */
fun getValueOrFail<T>(result: Result<T, ResultFailure>): T = (result as ResultSuccess<T>).value

/******************************************************************************************************************
 * Value Provider
 *******************************************************************************************************************/

type DisplayValue<T <: Object> =  {|
   label: String,
   properties?: T
|}

type ProvidedValue<T <: Object | SimpleType, P <: Object> = {
    value: T,
    displayValue: DisplayValue<P>
}

type ValueProvider<InputType, ErrorType, ProvidedValueType, DisplayPropertiesType <: Object, ConnectionType <: Connection> =
    Executor<InputType, Array<ProvidedValue<ProvidedValueType, DisplayPropertiesType>>, ResultFailure<ErrorType, Error>, ConnectionType>

type PaginatedValueProvider<InputType, ErrorType, ProvidedValueType, DisplayPropertiesType <: Object, ConnectionType <: Connection> = {
   executor: Executor<InputType, Page<ProvidedValue<ProvidedValueType, DisplayPropertiesType>, InputType>, ResultFailure<ErrorType, Error>, ConnectionType>,
   nextPageExecutor: Executor<InputType, Page<ProvidedValue<ProvidedValueType, Object>, InputType>, ResultFailure<ErrorType, Error>, ConnectionType>
}

/******************************************************************************************************************
 * Metadata Provider
 *******************************************************************************************************************/
type TypeValueDefinition = PrimitiveTypeValueDefinition
                            | ObjectTypeValueDefinition
                            | ArrayTypeValueDefinition
                            | UnionTypeValueDefinition
                            | ExtendedObjectTypeValueDefinition

type EnumValue<T> = { value: T, label: String }

type PrimitiveTypeValueDefinition = StringTypeValueDefinition
  | BooleanTypeValueDefinition
  | NumberTypeValueDefinition
  | NullTypeValueDefinition

type StringFormat = "date" | "date-time"

type StringTypeValueDefinition = {|
  "type": "String",
  format?: StringFormat,
  values?: Array<EnumValue<String>>
|}

type BooleanTypeValueDefinition = {|
  "type": "Boolean"
|}

type NumberFormat = "integer" | "float" | "double" | "int32" | "int64"

type NumberTypeValueDefinition = {|
  "type": "Number",
  values?: Array<EnumValue<Number>>,
  format?: NumberFormat
|}

type NullTypeValueDefinition = {|
  "type": "Null"
|}

type ObjectTypeValueDefinition = {|
  name: String,
  "type": "Object",
  fields : Array<ObjectFieldValueDefinition>
|}

type ObjectFieldValueDefinition = {
    name: QName,
    value: TypeValueDefinition,
    required: Boolean,
    repeated: Boolean,
    annotations?: { _?: Any }
}


type ArrayTypeValueDefinition = {|
  item : TypeValueDefinition,
  "type": "Array",
  name?: String
|}

type UnionTypeValueDefinition = {|
  "type": "Union",
  types : Array<TypeValueDefinition>
|}

type ExtendedObjectTypeValueDefinition = {|
  name: String,
  "type" : "ExtendedObject",
  typeReference : String,
  customFields : Array<ObjectFieldValueDefinition>
|}

type MetadataProviderExecutor<InputType <: Object, ErrorType, ConnectionType <: Connection> =
    Executor<InputType, TypeValueDefinition, ResultFailure<ErrorType, Error>, ConnectionType>
