/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.module.boot.internal;

import static org.mule.test.allure.AllureConstants.ClassloadingIsolationFeature.CLASSLOADING_ISOLATION;
import static org.mule.test.allure.AllureConstants.ClassloadingIsolationFeature.ClassloadingIsolationStory.CLASSLOADER_GENERATION;

import static java.lang.System.getProperty;
import static java.nio.file.Files.copy;
import static java.nio.file.Files.createDirectories;

import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.containsString;
import static org.hamcrest.Matchers.instanceOf;
import static org.hamcrest.Matchers.notNullValue;
import static org.hamcrest.core.Is.is;

import static org.junit.Assert.assertThrows;

import org.mule.runtime.module.deployment.impl.internal.builder.JarFileBuilder;
import org.mule.tck.junit4.AbstractMuleTestCase;
import org.mule.tck.util.CompilerUtils.JarCompiler;

import java.io.File;
import java.lang.reflect.Method;
import java.net.URISyntaxException;
import java.nio.file.Path;

import org.junit.jupiter.api.BeforeAll;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.io.TempDir;

import io.qameta.allure.Feature;
import io.qameta.allure.Issue;
import io.qameta.allure.Story;

@Feature(CLASSLOADING_ISOLATION)
@Story(CLASSLOADER_GENERATION)
class DefaultMuleClassPathConfigTestCase extends AbstractMuleTestCase {

  @TempDir
  private static Path muleHome;

  private ClassLoader containerClassLoader;

  private Class fromMuleModule;
  private Class fromThirdPartyLib;
  private Class fromMuleJavaSpecificModule;
  private Class fromThirdPartyJavaSpecificLib;

  @BeforeAll
  public static void setUpClass() throws Exception {
    Path fromThirdPartyLibJarFile = new JarFileBuilder("third-party-lib",
                                                       new JarCompiler()
                                                           .useClassPath()
                                                           .compiling(getResourceFile("/org/test/opt/FromThirdPartyLib.java"))
                                                           .compile("third-party-lib.jar"))
                                                               .getArtifactFile();

    Path fromMuleModuleJarFile = new JarFileBuilder("mule-module",
                                                    new JarCompiler()
                                                        .useClassPath()
                                                        .compiling(getResourceFile("/org/test/mule/FromMuleModule.java"))
                                                        .dependingOn(fromThirdPartyLibJarFile)
                                                        .compile("mule-module.jar"))
                                                            .getArtifactFile();

    Path fromUserLibJarFile = new JarFileBuilder("user-lib",
                                                 new JarCompiler()
                                                     .useClassPath()
                                                     .compiling(getResourceFile("/org/test/user/FromUserLib.java"))
                                                     .compile("user-lib.jar"))
                                                         .getArtifactFile();

    Path fromUserLibPropertiesFile = getResourceFile("/org/test/user/properties.yaml");

    final var libMule = muleHome.resolve("lib/mule");
    createDirectories(libMule);
    copy(fromMuleModuleJarFile, libMule.resolve(fromMuleModuleJarFile.getFileName().toString()));
    final var libOpt = muleHome.resolve("lib/opt");
    createDirectories(libOpt);
    copy(fromThirdPartyLibJarFile, libOpt.resolve(fromThirdPartyLibJarFile.getFileName().toString()));
    final var libUser = muleHome.resolve("lib/user");
    createDirectories(libUser);
    copy(fromUserLibJarFile, libUser.resolve(fromUserLibJarFile.getFileName().toString()));
    copy(fromUserLibPropertiesFile, libUser.resolve(fromUserLibPropertiesFile.getFileName().toString()));

    Path fromThirdPartyLibJarJavaSpecificFile = new JarFileBuilder("third-party-lib-javaspecific",
                                                                   new JarCompiler()
                                                                       .useClassPath()
                                                                       .compiling(getResourceFile("/org/test/opt/javaspecific/FromThirdPartyLib.java"))
                                                                       .compile("third-party-lib-javaspecific.jar"))
                                                                           .getArtifactFile();

    Path fromMuleModuleJarJavaSpecificFile = new JarFileBuilder("mule-module-javaspecific",
                                                                new JarCompiler()
                                                                    .useClassPath()
                                                                    .compiling(getResourceFile("/org/test/mule/javaspecific/FromMuleModule.java"))
                                                                    .dependingOn(fromThirdPartyLibJarJavaSpecificFile)
                                                                    .compile("mule-module-javaspecific.jar"))
                                                                        .getArtifactFile();

    final var javaSpecificationVersion = getProperty("java.specification.version").split("\\.")[0];

    final var libMuleJdkx = muleHome.resolve("lib/mule/jdk-" + javaSpecificationVersion);
    createDirectories(libMuleJdkx);
    copy(fromMuleModuleJarJavaSpecificFile, libMuleJdkx
        .resolve(fromMuleModuleJarJavaSpecificFile.getFileName().toString()));
    final var libOptJdkx = muleHome.resolve("lib/opt/jdk-" + javaSpecificationVersion);
    createDirectories(libOptJdkx);
    copy(fromThirdPartyLibJarJavaSpecificFile, libOptJdkx
        .resolve(fromThirdPartyLibJarJavaSpecificFile.getFileName().toString()));
  }

  /**
   * Creates an instance of {@link File} with the path specified in the parameter {@code resource}.
   *
   * @param resource the path to the file.
   * @return a {@link File} representing the resource.
   * @throws URISyntaxException if an error occurred while trying to convert the URI.
   */
  public static Path getResourceFile(String resource) throws URISyntaxException {
    return Path.of(DefaultMuleClassPathConfigTestCase.class.getResource(resource).toURI());
  }

  @BeforeEach
  public void setUp() {
    containerClassLoader = new DefaultMuleContainerFactory(null, null)
        .createContainerSystemClassLoader(muleHome.toFile(), muleHome.toFile());

    try {
      fromMuleModule = containerClassLoader.loadClass("org.test.mule.FromMuleModule");
      fromThirdPartyLib = containerClassLoader.loadClass("org.test.opt.FromThirdPartyLib");
      fromMuleJavaSpecificModule = containerClassLoader.loadClass("org.test.mule.javaspecific.FromMuleModule");
      fromThirdPartyJavaSpecificLib = containerClassLoader.loadClass("org.test.opt.javaspecific.FromThirdPartyLib");
    } catch (ClassNotFoundException e) {
      throw new RuntimeException(e);
    }
  }

  @Test
  void muleOptClassLoaderVisibility() throws Exception {
    assertThat(fromMuleModule.getDeclaredMethod("useThirdPartyLib", null)
        .invoke(null),
               is("OK"));
  }

  @Test
  @Issue("W-16010357")
  void optUserClassLoaderVisibility() throws Exception {
    assertThat(fromThirdPartyLib.getDeclaredMethod("useLib", String.class)
        .invoke(null, "org.test.user.FromUserLib"),
               is("OK"));
  }

  @Test
  void optMuleNoClassLoaderVisibility() throws Exception {
    final String muleClassName = "org.test.mule.FromMuleModule";
    final Method useLibMethod = fromThirdPartyLib.getDeclaredMethod("useLib", String.class);

    var thrown = assertThrows(Exception.class, () -> useLibMethod.invoke(null, muleClassName));
    assertThat(thrown.getCause().getCause(), instanceOf(ClassNotFoundException.class));
    assertThat(thrown.getCause().getCause().getMessage(), containsString(muleClassName));
  }

  @Test
  void muleOptJavaSpecificClassLoaderVisibility() throws Exception {
    assertThat(fromMuleJavaSpecificModule.getDeclaredMethod("useThirdPartyLib", null)
        .invoke(null),
               is("OK"));
  }

  @Test
  @Issue("W-16010357")
  void optJavaSpoecificUserClassLoaderVisibility() throws Exception {
    assertThat(fromThirdPartyJavaSpecificLib.getDeclaredMethod("useLib", String.class)
        .invoke(null, "org.test.user.FromUserLib"),
               is("OK"));
  }

  @Test
  void optMuleJavaSpecificNoClassLoaderVisibility() throws Exception {
    final String muleClassName = "org.test.mule.javaspecific.FromMuleModule";
    final Method useLibMethod = fromThirdPartyJavaSpecificLib.getDeclaredMethod("useLib", String.class);

    var thrown = assertThrows(Exception.class, () -> useLibMethod.invoke(null, muleClassName));
    assertThat(thrown.getCause().getCause(), instanceOf(ClassNotFoundException.class));
    assertThat(thrown.getCause().getCause().getMessage(), containsString(muleClassName));
  }

  @Test
  @Issue("W-16188357")
  void findUserResource() {
    assertThat(containerClassLoader.getResource("properties.yaml"), notNullValue());
  }

}
