/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.api.meta.model.declaration.fluent;

import org.mule.runtime.api.meta.model.ModelProperty;
import org.mule.runtime.api.meta.model.stereotype.StereotypeModel;

/**
 * Allows configuring a {@link ComponentDeclaration} through a fluent API
 *
 * @since 1.0
 */
abstract class ComponentDeclarer<T extends ComponentDeclarer, D extends ComponentDeclaration>
    extends ParameterizedDeclarer<T, D>
    implements HasModelProperties<ComponentDeclarer>, HasNestedComponents, HasNestedRoutes {

  /**
   * Creates a new instance
   *
   * @param declaration the {@link ComponentDeclaration} which will be configured
   */
  ComponentDeclarer(D declaration) {
    super(declaration);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public NestedComponentDeclarer withOptionalComponent(String nestedComponentName) {
    NestedComponentDeclarer<?, ?> declarer = withComponent(nestedComponentName);
    declarer.getDeclaration().setRequired(false);
    return declarer;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public NestedComponentDeclarer withComponent(String nestedComponentName) {
    NestedComponentDeclaration nested = new NestedComponentDeclaration(nestedComponentName);
    declaration.addNestedComponent(nested);
    return new NestedComponentDeclarer(nested);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public NestedChainDeclarer withChain() {
    NestedChainDeclaration nested = new NestedChainDeclaration("main");
    declaration.addNestedComponent(nested);
    return new NestedChainDeclarer(nested);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public NestedChainDeclarer withChain(String chainName) {
    NestedChainDeclaration nested = new NestedChainDeclaration(chainName);
    declaration.addNestedComponent(nested);
    return new NestedChainDeclarer(nested);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public NestedRouteDeclarer withRoute(String routeName) {
    NestedRouteDeclaration nested = new NestedRouteDeclaration(routeName);
    declaration.addNestedComponent(nested);
    return new NestedRouteDeclarer(nested);
  }

  /**
   * Adds the given {@code stereotype}
   *
   * @param stereotype a {@link org.mule.runtime.api.meta.model.stereotype.StereotypeModel}
   * @return {@code this} declarer
   */
  public T withStereotype(StereotypeModel stereotype) {
    declaration.addStereotype(stereotype);
    return (T) this;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public T withModelProperty(ModelProperty modelProperty) {
    declaration.addModelProperty(modelProperty);
    return (T) this;
  }

}
