/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.internal.util.collection;

import static java.util.Collections.emptySet;

import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;
import java.util.function.BiConsumer;
import java.util.function.BinaryOperator;
import java.util.function.Function;
import java.util.function.Supplier;
import java.util.stream.Collector;

/**
 * {@link Collector} which returns an immutable {@link Map}
 *
 * @param <T> the generic type of input elements
 * @param <K> the output map's key type
 * @param <V> the output map's values type
 * @since 1.0
 */
public class ImmutableMapCollector<T, K, V> implements Collector<T, Map<K, V>, Map<K, V>> {

  private final Function<T, K> keyMapper;
  private final Function<T, V> valueMapper;

  /**
   * Creates a new instance
   *
   * @param keyMapper   a mapping function to produce keys
   * @param valueMapper a mapping function to produce values
   */
  public ImmutableMapCollector(Function<T, K> keyMapper, Function<T, V> valueMapper) {
    this.keyMapper = keyMapper;
    this.valueMapper = valueMapper;
  }

  @Override
  public Supplier<Map<K, V>> supplier() {
    return HashMap::new;
  }

  @Override
  public BiConsumer<Map<K, V>, T> accumulator() {
    return (map, value) -> map.put(keyMapper.apply(value), valueMapper.apply(value));
  }

  @Override
  public BinaryOperator<Map<K, V>> combiner() {
    return (left, right) -> {
      left.putAll(right);
      return left;
    };
  }

  @Override
  public Function<Map<K, V>, Map<K, V>> finisher() {
    return Collections::unmodifiableMap;
  }

  @Override
  public Set<Characteristics> characteristics() {
    return emptySet();
  }
}
