/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.api.scheduler;

import org.mule.runtime.api.alert.TimedDataAggregation;

/**
 * Provides a read-only view of the state of a {@link Scheduler}.
 *
 * @since 1.0
 */
public interface SchedulerView {

  /**
   * Returns a name that indicates where was this scheduler created.
   * <p>
   * It needs not to be unique.
   *
   * @return the name of this scheduler.
   */
  String getName();

  /**
   * @return the kind of tasks this scheduler is configured to run (ref: org.mule.service.scheduler.ThreadType).
   * @since 1.10
   */
  default String getThreadType() {
    return "UNKNOWN";
  }

  /**
   * The number of tasks waiting in a queue for this scheduler.
   * <p>
   * 0 may mean there is no queue, or it cannot be queried, or there are no queued items.
   *
   * @return the number of tasks waiting in a queue for this scheduler.
   * @since 1.10, 1.9.8, 1.6.21
   */
  default int getQueuedTasks() {
    return 0;
  }

  /**
   * @return the number of tasks being executed by this scheduler.
   * @since 1.10, 1.9.8, 1.6.21
   */
  default int getActiveTasks() {
    return 0;
  }

  /**
   * @return how many tasks were rejected from this scheduler because it was a max capacity.
   * @since 1.10, 1.9.8, 1.6.21
   */
  default TimedDataAggregation<Integer> getRejectionsOverTime() {
    return null;
  }

  /**
   * @return how many tasks were throttled from this scheduler because it was a max capacity.
   * @since 1.10, 1.9.8, 1.6.21
   */
  default TimedDataAggregation<Integer> getThrottlesOverTime() {
    return null;
  }

  /**
   * A shutdown scheduler is not able to accept new tasks.
   * <p>
   * Tasks still running are allowed a graceful period to complete.
   *
   * @return {@code true} if this scheduler has been shutdown.
   * @see Scheduler#isShutdown()
   */
  boolean isShutdown();

  /**
   * A terminated scheduler is shutdown and all of its in-progress tasks have been completed or forcefully terminated after a
   * graceful shutdown period.
   *
   * @return {@code true} if this scheduler is terminated.
   * @see Scheduler#isTerminated()
   */
  boolean isTerminated();

}
