/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.runtime.api.message;

import static java.lang.String.format;
import static java.util.ServiceLoader.load;
import org.mule.runtime.api.message.Message.Builder;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Factory class used to create {@link Builder} objects.
 *
 * @since 1.0
 */
public abstract class AbstractMuleMessageBuilderFactory {

  private static final Logger LOGGER = LoggerFactory.getLogger(AbstractMuleMessageBuilderFactory.class);

  static {
    try {
      final AbstractMuleMessageBuilderFactory factory = load(AbstractMuleMessageBuilderFactory.class,
                                                             AbstractMuleMessageBuilderFactory.class.getClassLoader()).iterator()
                                                                 .next();
      LOGGER.info(format("Loaded MuleMessageBuilderFactory implementation '%s' from classloader '%s'",
                         factory.getClass().getName(), factory.getClass().getClassLoader().toString()));

      DEFAULT_FACTORY = factory;
    } catch (Throwable t) {
      LOGGER.error("Error loading MuleMessageBuilderFactory implementation.", t);
      throw t;
    }
  }

  private static final AbstractMuleMessageBuilderFactory DEFAULT_FACTORY;

  /**
   * The implementation of this abstract class is provided by the Mule Runtime, and loaded during this class initialization.
   * <p>
   * If more than one implementation is found, the classLoading order of those implementations will determine which one is used.
   * Information about this will be logged to aid in the troubleshooting of those cases.
   *
   * @return the implementation of this builder factory provided by the Mule Runtime.
   */
  static final AbstractMuleMessageBuilderFactory getDefaultFactory() {
    return DEFAULT_FACTORY;
  }

  /**
   * @return a fresh {@link Builder} object.
   */
  protected abstract Message.PayloadBuilder create();

  /**
   * @param message existing {@link Message} to use as a template to create a new {@link Builder} instance.
   * @return a fresh {@link Builder} based on the template {@code message} provided.
   */
  protected abstract Message.Builder create(Message message);

}
