/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.runtime.api.notification;

import org.mule.runtime.api.scheduler.Scheduler;

/**
 * Contains information about an event that happened in the Mule Server. The event may happen in any Mule artifact running on the
 * Mule Server (an extension, a plugin, or an inner component of the Runtime).
 * <p>
 * The nature of the event that occurred is identified by the concrete type of the {@link Notification} and its
 * {@link #getAction() action}.
 * <p>
 * The concrete type of the {@link Notification} is an indication of the kind of component that caused the event, and in may
 * contain additional data describing the event.
 * 
 * @since 4.0
 */
public interface Notification {

  /**
   * Indicates the synchronicity of the processing of this notification.
   * 
   * @return Whether the execution of the listeners for this notification will happen on the same thread that called
   *         {@link NotificationDispatcher#dispatch(Notification)} or will be dispatched to a {@link Scheduler} for asynchronous
   *         processing.
   */
  default boolean isSynchronous() {
    return false;
  }

  /**
   * Informs the kind of event that caused this notification to be fired.
   * 
   * @return the kind of action that caused this notification to be fired.
   */
  Action getAction();

  /**
   * Indicates the kind of action that causes a {@link Notification} to be fired.
   */
  interface Action {

    /**
     * @return unique namespace for the action
     */
    String getNamespace();

    /**
     * @return unique ID for this action
     */
    String getIdentifier();

  }
}
