/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.runtime.api.notification;

import java.util.function.Predicate;

/**
 * Allows to register/unregister {@link NotificationListener}s for {@link Notification}s fired by the Mule container, extensions
 * and applications.
 * 
 * @since 4.0
 */
public interface NotificationListenerRegistry {

  String REGISTRY_KEY = "_muleNotificationListenerRegistry";

  /**
   * Registers a {@link NotificationListener}. The listener will be notified when a particular event happens within the server.
   * Typically this is not an event in the same sense as a Mule Event (although there is nothing stopping the implementation of
   * this class triggering listeners when a Mule Event is received).
   *
   * @param listener the listener to register
   * @param <N>      the concrete type of the notification to be handled by the {@code listener}
   */
  <N extends Notification> void registerListener(NotificationListener<N> listener);

  /**
   * Registers a {@link NotificationListener}. The listener will be notified when a particular event happens within the server.
   * Typically this is not an event in the same sense as a Mule Event (although there is nothing stopping the implementation of
   * this class triggering listeners when a Mule Event is received).
   *
   * @param listener the listener to register
   * @param selector a filter to apply on a fired {@link Notification} before calling the {@code listener} with it. Non-null.
   * @param <N>      the concrete type of the notification to be handled by the {@code listener}
   */
  <N extends Notification> void registerListener(NotificationListener<N> listener, Predicate<N> selector);

  /**
   * Unregisters a previously registered {@link NotificationListener}. If the listener has not already been registered, this
   * method should return without exception
   *
   * @param listener the listener to unregister
   * @param <N>      the concrete type of the notification handled by the {@code listener}
   */
  <N extends Notification> void unregisterListener(NotificationListener<N> listener);

}
