/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.ast.internal;

import static com.google.common.collect.Streams.concat;
import static java.util.Arrays.asList;
import static java.util.Collections.emptyList;
import static java.util.Collections.emptyMap;
import static java.util.Collections.singleton;
import static java.util.Optional.empty;
import static java.util.Optional.of;
import static java.util.Optional.ofNullable;
import static org.mule.runtime.api.functional.Either.right;

import org.mule.runtime.api.component.ComponentIdentifier;
import org.mule.runtime.api.component.TypedComponentIdentifier.ComponentType;
import org.mule.runtime.api.component.location.ComponentLocation;
import org.mule.runtime.api.functional.Either;
import org.mule.runtime.api.meta.model.parameter.ParameterModel;
import org.mule.runtime.api.meta.model.parameter.ParameterizedModel;
import org.mule.runtime.ast.api.ComponentAst;
import org.mule.runtime.ast.api.ComponentMetadataAst;
import org.mule.runtime.ast.api.ComponentParameterAst;

import java.util.Collection;
import java.util.Map;
import java.util.NoSuchElementException;
import java.util.Optional;
import java.util.Spliterator;
import java.util.stream.Stream;


public class TestComponentAst implements ComponentAst {

  private final Object model;
  private final Collection<ComponentAst> innerComponents;
  private Map<String, String> rawParams = emptyMap();

  public TestComponentAst(ComponentAst... innerComponents) {
    this.model = null;
    this.innerComponents = asList(innerComponents);
  }

  public TestComponentAst(Object model, ComponentAst... innerComponents) {
    this.model = model;
    this.innerComponents = asList(innerComponents);
  }

  @Override
  public ComponentIdentifier getIdentifier() {
    return null;
  }

  @Override
  public ComponentType getComponentType() {
    return null;
  }

  @Override
  public ComponentLocation getLocation() {
    return null;
  }

  @Override
  public ComponentMetadataAst getMetadata() {
    return null;
  }

  @Override
  public Optional<String> getComponentId() {
    return getRawParameterValue("name");
  }

  @Override
  public <M> Optional<M> getModel(Class<M> modelClass) {
    if (model == null || !modelClass.isAssignableFrom(model.getClass())) {
      return empty();
    } else {
      return (Optional<M>) of(model);
    }
  }

  @Override
  public Optional<String> getRawParameterValue(String paramName) {
    return ofNullable(rawParams.get(paramName));
  }

  @Override
  public ComponentParameterAst getParameter(String paramName) {
    final Optional<ParameterModel> paramModel = getModel(ParameterizedModel.class)
        .flatMap(parameterized -> parameterized.getAllParameterModels()
            .stream()
            .filter(pm -> pm.getName().equals(paramName))
            .findAny());


    if (rawParams.containsKey(paramName) || paramModel.isPresent()) {
      return new ComponentParameterAst() {

        @Override
        public Either<String, Object> getValue() {
          return right(getRawValue() != null ? getRawValue() : getModel().getDefaultValue());
        }

        @Override
        public String getRawValue() {
          return rawParams.get(paramName);
        }

        @Override
        public ParameterModel getModel() {
          return paramModel.get();
        }

        @Override
        public Optional<ComponentMetadataAst> getMetadata() {
          return empty();
        }
      };
    } else {
      throw new NoSuchElementException(paramName);
    }
  }

  @Override
  public Collection<ComponentParameterAst> getParameters() {
    return emptyList();
  }

  public void setParameters(Map<String, String> rawParams) {
    this.rawParams = rawParams;
  }

  @Override
  public Stream<ComponentAst> recursiveStream() {
    return concat(singleton(this).stream(),
                  innerComponents.stream()
                      .flatMap(cm -> cm.recursiveStream()));
  }

  @Override
  public Spliterator<ComponentAst> recursiveSpliterator() {
    return recursiveStream().spliterator();
  }

  @Override
  public Stream<ComponentAst> directChildrenStream() {
    return innerComponents.stream();
  }

  @Override
  public Spliterator<ComponentAst> directChildrenSpliterator() {
    return innerComponents.spliterator();
  }

  @Override
  public String toString() {
    return getComponentId().orElse(super.toString());
  }

}
