/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.ast.graph.api;

import static org.slf4j.LoggerFactory.getLogger;

import org.mule.runtime.api.meta.model.parameter.ParameterModel;
import org.mule.runtime.api.meta.model.parameter.ParameterizedModel;
import org.mule.runtime.ast.api.ArtifactAst;
import org.mule.runtime.ast.api.ComponentAst;
import org.mule.runtime.ast.api.ComponentParameterAst;
import org.mule.runtime.ast.graph.internal.ComponentAstEdge;
import org.mule.runtime.ast.graph.internal.DefaultArtifactAstDependencyGraph;

import java.util.HashSet;
import java.util.Objects;
import java.util.Set;

import org.jgrapht.Graph;
import org.jgrapht.graph.SimpleDirectedGraph;
import org.slf4j.Logger;

/**
 * Contains a factory method for creating {@link ArtifactAstDependencyGraph}s.
 *
 * @since 1.0
 */
public class ArtifactAstDependencyGraphFactory {

  private static final Logger LOGGER = getLogger(ArtifactAstDependencyGraphFactory.class);

  private ArtifactAstDependencyGraphFactory() {
    // Nothing to do
  }

  /**
   * Creates an {@link ArtifactAstDependencyGraph} for the given {@link ArtifactAst}.
   * <p>
   * This method relies on the stereotypes in the extension model for determining the dependencies.
   *
   * @param fullArtifactAst the {@link ArtifactAst} to generate the graph for.
   * @return a graph instance with the dependencies between the artifact components resolved.
   */
  public static ArtifactAstDependencyGraph generateFor(ArtifactAst fullArtifactAst) {

    // The edges of the graph represent a "depends on" relation.
    final Graph<ComponentAst, ComponentAstEdge> graph = new SimpleDirectedGraph<>(ComponentAstEdge.class);

    // First pass, add all components to the graph
    fullArtifactAst.recursiveStream().forEach(graph::addVertex);

    Set<ComponentAstDependency> missingDependencies = new HashSet<>();

    // Second pass, determine the dependencies
    fullArtifactAst.recursiveStream()
        .forEach(comp -> {
          // A component depends on its children...
          comp.directChildrenStream().forEach(child -> addDependency(graph, comp, child));

          // For components that have parameters, add the dependencies when those parameters are references
          comp.getModel(ParameterizedModel.class)
              .ifPresent(parameterizedModel -> parameterizedModel.getAllParameterModels().stream()
                  .filter(paramModel -> !paramModel.getAllowedStereotypes().isEmpty())
                  .filter(paramModel -> parameterValue(comp, paramModel) != null)
                  .map(paramModel -> new ComponentAstDependency(parameterValue(comp, paramModel),
                                                                new HashSet<>(paramModel.getAllowedStereotypes())))
                  .forEach(dependency -> {
                    if (missingDependencies.contains(dependency)) {
                      // If this dependency is already identified as missing, skip it
                      return;
                    }

                    // Iterate through the whole AST looking for a component with a matching id and stereotype
                    fullArtifactAst.recursiveStream()
                        // If there was a dependency already found, skip the stereotype matching
                        .filter(tlcm -> !graph.containsEdge(comp, tlcm))
                        .filter(dependency::isSatisfiedBy)
                        .forEach(tlcm -> addDependency(graph, comp, tlcm));

                    if (graph.outgoingEdgesOf(comp)
                        .stream()
                        .noneMatch(edge -> dependency.isSatisfiedBy(edge.getTarget()))) {
                      missingDependencies.add(dependency);
                    }
                  }));
        });

    if (!missingDependencies.isEmpty() && LOGGER.isDebugEnabled()) {
      LOGGER.debug("Missing dependencies:");
      missingDependencies.forEach(rq -> LOGGER.debug("    {}", rq));
    }

    return new DefaultArtifactAstDependencyGraph(fullArtifactAst, graph, missingDependencies);
  }

  private static String parameterValue(ComponentAst comp, ParameterModel paramModel) {
    final ComponentParameterAst param = comp.getParameter(paramModel.getName());
    if (param.getValue().isRight() && param.getValue().getRight() != null) {
      return Objects.toString(param.getValue().getRight());
    } else {
      return null;
    }
  }

  private static void addDependency(final Graph<ComponentAst, ComponentAstEdge> graph,
                                    final ComponentAst dependencyFrom,
                                    final ComponentAst dependencyTo) {
    LOGGER.debug("Dependency found: '{}' to '{}'", dependencyFrom, dependencyTo);
    graph.addEdge(dependencyFrom, dependencyTo);
  }
}
