/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.ast.graph.api;

import static java.util.Arrays.asList;
import static java.util.Collections.emptyList;
import static java.util.Collections.emptyMap;
import static java.util.Collections.singletonMap;
import static java.util.Collections.sort;
import static java.util.stream.Collectors.toSet;
import static org.hamcrest.Matchers.empty;
import static org.hamcrest.collection.IsCollectionWithSize.hasSize;
import static org.hamcrest.core.Is.is;
import static org.hamcrest.core.IsCollectionContaining.hasItem;
import static org.hamcrest.core.IsNot.not;
import static org.junit.Assert.assertThat;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.when;
import static org.mockito.Mockito.withSettings;
import static org.mule.runtime.ast.graph.api.ArtifactAstDependencyGraphFactory.generateFor;

import org.mule.runtime.api.meta.model.parameter.ParameterModel;
import org.mule.runtime.api.meta.model.parameter.ParameterizedModel;
import org.mule.runtime.api.meta.model.stereotype.HasStereotypeModel;
import org.mule.runtime.api.meta.model.stereotype.ImmutableStereotypeModel;
import org.mule.runtime.api.meta.model.stereotype.StereotypeModel;
import org.mule.runtime.ast.api.ComponentAst;
import org.mule.runtime.ast.internal.TestArtifactAst;
import org.mule.runtime.ast.internal.TestComponentAst;

import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import com.google.common.collect.ImmutableMap;
import io.qameta.allure.Description;
import io.qameta.allure.Issue;
import org.junit.Test;

public class ArtifactAstDependencyGraphFactoryTestCase {

  @Test
  public void simpleTopLevelWithInner() {
    final ComponentAst inner = new TestComponentAst();
    final ComponentAst topLevel = new TestComponentAst(inner);

    final ArtifactAstDependencyGraph graph = generateFor(new TestArtifactAst(topLevel));

    final Collection<ComponentAst> topLevelMinimalComponents =
        graph.minimalArtifactFor(topLevel).recursiveStream().collect(toSet());
    assertThat(topLevelMinimalComponents, hasItem(topLevel));
    assertThat(topLevelMinimalComponents, hasItem(inner));
  }

  @Test
  public void simpleTopLevelWithInnerPredicate() {
    final ComponentAst inner = new TestComponentAst();
    final ComponentAst topLevel = new TestComponentAst(inner);

    final ArtifactAstDependencyGraph graph = generateFor(new TestArtifactAst(topLevel));

    final Collection<ComponentAst> topLevelMinimalComponents =
        graph.minimalArtifactFor(topLevel::equals).recursiveStream().collect(toSet());
    assertThat(topLevelMinimalComponents, hasItem(topLevel));
    assertThat(topLevelMinimalComponents, hasItem(inner));
  }

  @Test
  public void unrelatedComponents() {
    final ComponentAst innerA = new TestComponentAst();
    final ComponentAst innerB = new TestComponentAst();

    final ArtifactAstDependencyGraph graph = generateFor(new TestArtifactAst(innerA, innerB));

    final Collection<ComponentAst> innerAMinimalComponents =
        graph.minimalArtifactFor(innerA).recursiveStream().collect(toSet());
    assertThat(innerAMinimalComponents, hasItem(innerA));
    assertThat(innerAMinimalComponents, not(hasItem(innerB)));

    final Collection<ComponentAst> innerBMinimalComponents =
        graph.minimalArtifactFor(innerB).recursiveStream().collect(toSet());
    assertThat(innerBMinimalComponents, hasItem(innerB));
    assertThat(innerBMinimalComponents, not(hasItem(innerA)));
  }

  @Test
  public void unrelatedComponentsPredicate() {
    final ComponentAst innerA = new TestComponentAst();
    final ComponentAst innerB = new TestComponentAst();

    final ArtifactAstDependencyGraph graph = generateFor(new TestArtifactAst(innerA, innerB));

    final Collection<ComponentAst> innerAMinimalComponents =
        graph.minimalArtifactFor(innerA::equals).recursiveStream().collect(toSet());
    assertThat(innerAMinimalComponents, hasItem(innerA));
    assertThat(innerAMinimalComponents, not(hasItem(innerB)));

    final Collection<ComponentAst> innerBMinimalComponents =
        graph.minimalArtifactFor(innerB::equals).recursiveStream().collect(toSet());
    assertThat(innerBMinimalComponents, hasItem(innerB));
    assertThat(innerBMinimalComponents, not(hasItem(innerA)));
  }

  @Test
  public void noAllowedStereotypes() {
    final ParameterizedModel parameterizedModel = mock(ParameterizedModel.class);
    final ParameterModel parameterModel = mock(ParameterModel.class);
    when(parameterModel.getAllowedStereotypes()).thenReturn(emptyList());
    when(parameterModel.getName()).thenReturn("nameParam");
    when(parameterizedModel.getAllParameterModels()).thenReturn(asList(parameterModel));

    final TestComponentAst innerA = new TestComponentAst(parameterizedModel);
    innerA.setParameters(singletonMap("nameParam", "dependencyId"));
    final TestComponentAst innerB = new TestComponentAst();

    final ArtifactAstDependencyGraph graph = generateFor(new TestArtifactAst(innerA, innerB));

    final Collection<ComponentAst> innerAMinimalComponents =
        graph.minimalArtifactFor(innerA).recursiveStream().collect(toSet());
    assertThat(innerAMinimalComponents, hasItem(innerA));
    assertThat(innerAMinimalComponents, not(hasItem(innerB)));

    final Collection<ComponentAst> innerBMinimalComponents =
        graph.minimalArtifactFor(innerB).recursiveStream().collect(toSet());
    assertThat(innerBMinimalComponents, hasItem(innerB));
    assertThat(innerBMinimalComponents, not(hasItem(innerA)));
  }

  @Test
  public void unmatchingStereotypesUnmatchingName() {
    final ParameterizedModel parameterizedModel = mock(ParameterizedModel.class);
    final ParameterModel parameterModel = mock(ParameterModel.class);
    when(parameterModel.getAllowedStereotypes())
        .thenReturn(asList(new ImmutableStereotypeModel("ST_NAME", "ST_NAMESPACE", null)));
    when(parameterModel.getName()).thenReturn("nameParam");
    when(parameterizedModel.getAllParameterModels()).thenReturn(asList(parameterModel));

    final TestComponentAst innerA = new TestComponentAst(parameterizedModel);
    innerA.setParameters(singletonMap("nameParam", "dependencyId"));
    final TestComponentAst innerB = new TestComponentAst();
    innerB.setParameters(singletonMap("name", "notMatchingDependencyId"));

    final ArtifactAstDependencyGraph graph = generateFor(new TestArtifactAst(innerA, innerB));

    final Collection<ComponentAst> innerAMinimalComponents =
        graph.minimalArtifactFor(innerA).recursiveStream().collect(toSet());
    assertThat(innerAMinimalComponents, hasItem(innerA));
    assertThat(innerAMinimalComponents, not(hasItem(innerB)));

    final Collection<ComponentAst> innerBMinimalComponents =
        graph.minimalArtifactFor(innerB).recursiveStream().collect(toSet());
    assertThat(innerBMinimalComponents, hasItem(innerB));
    assertThat(innerBMinimalComponents, not(hasItem(innerA)));
  }

  @Test
  public void matchingStereotypesUnmatchingName() {
    final ParameterizedModel parameterizedModel = mock(ParameterizedModel.class);
    final ParameterModel parameterModel = mock(ParameterModel.class);
    when(parameterModel.getAllowedStereotypes())
        .thenReturn(asList(new ImmutableStereotypeModel("ST_NAME", "ST_NAMESPACE", null)));
    when(parameterModel.getName()).thenReturn("nameParam");
    when(parameterizedModel.getAllParameterModels()).thenReturn(asList(parameterModel));

    final TestComponentAst innerA = new TestComponentAst(parameterizedModel);
    innerA.setParameters(singletonMap("nameParam", "dependencyId"));

    final HasStereotypeModel hasStereotypeModel = mock(HasStereotypeModel.class);
    when(hasStereotypeModel.getStereotype()).thenReturn(new ImmutableStereotypeModel("ST_NAME", "ST_NAMESPACE", null));
    final TestComponentAst innerB = new TestComponentAst(hasStereotypeModel);
    innerB.setParameters(singletonMap("name", "notMatchingDependencyId"));

    final ArtifactAstDependencyGraph graph = generateFor(new TestArtifactAst(innerA, innerB));

    final Collection<ComponentAst> innerAMinimalComponents =
        graph.minimalArtifactFor(innerA).recursiveStream().collect(toSet());
    assertThat(innerAMinimalComponents, hasItem(innerA));
    assertThat(innerAMinimalComponents, not(hasItem(innerB)));

    final Collection<ComponentAst> innerBMinimalComponents =
        graph.minimalArtifactFor(innerB).recursiveStream().collect(toSet());
    assertThat(innerBMinimalComponents, hasItem(innerB));
    assertThat(innerBMinimalComponents, not(hasItem(innerA)));
  }

  @Test
  public void unmatchingStereotypesMatchingName() {
    final ParameterizedModel parameterizedModel = mock(ParameterizedModel.class);
    final ParameterModel parameterModel = mock(ParameterModel.class);
    when(parameterModel.getAllowedStereotypes())
        .thenReturn(asList(new ImmutableStereotypeModel("ST_NAME", "ST_NAMESPACE", null)));
    when(parameterModel.getName()).thenReturn("nameParam");
    when(parameterizedModel.getAllParameterModels()).thenReturn(asList(parameterModel));

    final TestComponentAst innerA = new TestComponentAst(parameterizedModel);
    innerA.setParameters(singletonMap("nameParam", "dependencyId"));

    final HasStereotypeModel hasStereotypeModel = mock(HasStereotypeModel.class);
    when(hasStereotypeModel.getStereotype()).thenReturn(new ImmutableStereotypeModel("ST_NOT_NAME", "ST_NAMESPACE", null));
    final TestComponentAst innerB = new TestComponentAst(hasStereotypeModel);
    innerB.setParameters(singletonMap("name", "dependencyId"));

    final ArtifactAstDependencyGraph graph = generateFor(new TestArtifactAst(innerA, innerB));

    final Collection<ComponentAst> innerAMinimalComponents =
        graph.minimalArtifactFor(innerA).recursiveStream().collect(toSet());
    assertThat(innerAMinimalComponents, hasItem(innerA));
    assertThat(innerAMinimalComponents, not(hasItem(innerB)));

    final Collection<ComponentAst> innerBMinimalComponents =
        graph.minimalArtifactFor(innerB).recursiveStream().collect(toSet());
    assertThat(innerBMinimalComponents, hasItem(innerB));
    assertThat(innerBMinimalComponents, not(hasItem(innerA)));
  }

  @Test
  public void matchingStereotypesMatchingName() {
    final ParameterizedModel parameterizedModel = mock(ParameterizedModel.class);
    final ParameterModel parameterModel = mock(ParameterModel.class);
    when(parameterModel.getAllowedStereotypes())
        .thenReturn(asList(new ImmutableStereotypeModel("ST_NAME", "ST_NAMESPACE", null)));
    when(parameterModel.getName()).thenReturn("nameParam");
    when(parameterizedModel.getAllParameterModels()).thenReturn(asList(parameterModel));

    final TestComponentAst innerA = new TestComponentAst(parameterizedModel);
    innerA.setParameters(singletonMap("nameParam", "dependencyId"));

    final HasStereotypeModel hasStereotypeModel = mock(HasStereotypeModel.class);
    when(hasStereotypeModel.getStereotype()).thenReturn(new ImmutableStereotypeModel("ST_NAME", "ST_NAMESPACE", null));
    final TestComponentAst innerB = new TestComponentAst(hasStereotypeModel);
    innerB.setParameters(singletonMap("name", "dependencyId"));

    final ArtifactAstDependencyGraph graph = generateFor(new TestArtifactAst(innerA, innerB));

    final Collection<ComponentAst> innerAMinimalComponents =
        graph.minimalArtifactFor(innerA).recursiveStream().collect(toSet());
    assertThat(innerAMinimalComponents, hasItem(innerA));
    assertThat(innerAMinimalComponents, hasItem(innerB));

    final Collection<ComponentAst> innerBMinimalComponents =
        graph.minimalArtifactFor(innerB).recursiveStream().collect(toSet());
    assertThat(innerBMinimalComponents, hasItem(innerB));
    assertThat(innerBMinimalComponents, not(hasItem(innerA)));
  }

  @Test
  public void transitivesHandling() {
    final TestComponentAst inner = new TestComponentAst();
    inner.setParameters(singletonMap("name", "inner"));
    final TestComponentAst intermediateLevel = new TestComponentAst(inner);
    intermediateLevel.setParameters(singletonMap("name", "intermediate"));
    final TestComponentAst topLevel = new TestComponentAst(intermediateLevel);
    topLevel.setParameters(singletonMap("name", "top"));

    final ArtifactAstDependencyGraph graph = generateFor(new TestArtifactAst(topLevel));

    final Collection<ComponentAst> topLevelMinimalComponents =
        graph.minimalArtifactFor(topLevel).recursiveStream().collect(toSet());
    assertThat(topLevelMinimalComponents, hasItem(topLevel));
    assertThat(topLevelMinimalComponents, hasItem(intermediateLevel));
    assertThat(topLevelMinimalComponents, hasItem(inner));

    final Collection<ComponentAst> innerLevelMinimalComponents =
        graph.minimalArtifactFor(inner).recursiveStream().collect(toSet());
    assertThat(innerLevelMinimalComponents, not(hasItem(topLevel)));
    assertThat(innerLevelMinimalComponents, not(hasItem(intermediateLevel)));
    assertThat(innerLevelMinimalComponents, hasItem(inner));
  }

  @Test
  public void transitivesHandlingPredicate() {
    final TestComponentAst inner = new TestComponentAst();
    inner.setParameters(singletonMap("name", "inner"));
    final TestComponentAst intermediateLevel = new TestComponentAst(inner);
    intermediateLevel.setParameters(singletonMap("name", "intermediate"));
    final TestComponentAst topLevel = new TestComponentAst(intermediateLevel);
    topLevel.setParameters(singletonMap("name", "top"));

    final ArtifactAstDependencyGraph graph = generateFor(new TestArtifactAst(topLevel));

    final Collection<ComponentAst> topLevelMinimalComponents =
        graph.minimalArtifactFor(topLevel::equals).recursiveStream().collect(toSet());
    assertThat(topLevelMinimalComponents, hasItem(topLevel));
    assertThat(topLevelMinimalComponents, hasItem(intermediateLevel));
    assertThat(topLevelMinimalComponents, hasItem(inner));

    final Collection<ComponentAst> innerLevelMinimalComponents =
        graph.minimalArtifactFor(inner::equals).recursiveStream().collect(toSet());
    assertThat(innerLevelMinimalComponents, not(hasItem(topLevel)));
    assertThat(innerLevelMinimalComponents, not(hasItem(intermediateLevel)));
    assertThat(innerLevelMinimalComponents, hasItem(inner));
  }

  @Test
  public void missingReported() {
    final ParameterizedModel parameterizedModel = mock(ParameterizedModel.class);
    final ParameterModel parameterModel = mock(ParameterModel.class);
    when(parameterModel.getAllowedStereotypes())
        .thenReturn(asList(new ImmutableStereotypeModel("ST_NAME", "ST_NAMESPACE", null)));
    when(parameterModel.getName()).thenReturn("nameParam");
    when(parameterizedModel.getAllParameterModels()).thenReturn(asList(parameterModel));

    final TestComponentAst innerA = new TestComponentAst(parameterizedModel);
    innerA.setParameters(singletonMap("nameParam", "dependencyId"));

    final ArtifactAstDependencyGraph graph = generateFor(new TestArtifactAst(innerA));

    assertThat(graph.getMissingDependencies(), hasSize(1));

    final ComponentAstDependency missingDep = graph.getMissingDependencies().iterator().next();
    assertThat(missingDep.getName(), is("dependencyId"));
    assertThat(missingDep.getAllowedStereotypes(), hasSize(1));
    assertThat(missingDep.getAllowedStereotypes(), hasItem(new ImmutableStereotypeModel("ST_NAME", "ST_NAMESPACE", null)));
  }

  @Test
  @Issue("MULE-17730")
  @Description("A missing dependency was not reported if another dependency for the same component is present")
  public void missingReportedButAnotherFound() {
    final ParameterizedModel parameterizedModel = mock(ParameterizedModel.class);
    final ParameterModel parameterModel = mock(ParameterModel.class);
    when(parameterModel.getAllowedStereotypes())
        .thenReturn(asList(new ImmutableStereotypeModel("ST_NAME", "ST_NAMESPACE", null)));
    when(parameterModel.getName()).thenReturn("nameParam");
    when(parameterizedModel.getAllParameterModels()).thenReturn(asList(parameterModel));

    final TestComponentAst innerA = new TestComponentAst(parameterizedModel,
                                                         // a child component is a dependency
                                                         new TestComponentAst(mock(Object.class)));
    innerA.setParameters(singletonMap("nameParam", "dependencyId"));

    final ArtifactAstDependencyGraph graph = generateFor(new TestArtifactAst(innerA));

    assertThat(graph.getMissingDependencies(), hasSize(1));

    final ComponentAstDependency missingDep = graph.getMissingDependencies().iterator().next();
    assertThat(missingDep.getName(), is("dependencyId"));
    assertThat(missingDep.getAllowedStereotypes(), hasSize(1));
    assertThat(missingDep.getAllowedStereotypes(), hasItem(new ImmutableStereotypeModel("ST_NAME", "ST_NAMESPACE", null)));
  }

  @Test
  @Description("Dependencies missing more than once are only reported once")
  public void missingReportedTwice() {
    final ParameterizedModel parameterizedModel = mock(ParameterizedModel.class);
    final ParameterModel parameterModel = mock(ParameterModel.class);
    when(parameterModel.getAllowedStereotypes())
        .thenReturn(asList(new ImmutableStereotypeModel("ST_NAME", "ST_NAMESPACE", null)));
    when(parameterModel.getName()).thenReturn("nameParam");
    when(parameterizedModel.getAllParameterModels()).thenReturn(asList(parameterModel));

    final TestComponentAst innerA = new TestComponentAst(parameterizedModel);
    innerA.setParameters(singletonMap("nameParam", "dependencyId"));
    final TestComponentAst innerB = new TestComponentAst(parameterizedModel);
    innerB.setParameters(singletonMap("nameParam", "dependencyId"));

    final ArtifactAstDependencyGraph graph = generateFor(new TestArtifactAst(innerA, innerB));

    assertThat(graph.getMissingDependencies(), hasSize(1));

    final ComponentAstDependency missingDep = graph.getMissingDependencies().iterator().next();
    assertThat(missingDep.getName(), is("dependencyId"));
    assertThat(missingDep.getAllowedStereotypes(), hasSize(1));
    assertThat(missingDep.getAllowedStereotypes(), hasItem(new ImmutableStereotypeModel("ST_NAME", "ST_NAMESPACE", null)));
  }

  @Test
  public void missingDifferentStereotypeReported() {
    final ParameterizedModel parameterizedModel = mock(ParameterizedModel.class);
    final ParameterModel parameterModel = mock(ParameterModel.class);
    when(parameterModel.getAllowedStereotypes())
        .thenReturn(asList(new ImmutableStereotypeModel("ST_NAME", "ST_NAMESPACE", null)));
    when(parameterModel.getName()).thenReturn("nameParam");
    when(parameterizedModel.getAllParameterModels()).thenReturn(asList(parameterModel));

    final TestComponentAst innerA = new TestComponentAst(parameterizedModel);
    innerA.setParameters(singletonMap("nameParam", "dependencyId"));

    final HasStereotypeModel hasStereotypeModel = mock(HasStereotypeModel.class);
    when(hasStereotypeModel.getStereotype()).thenReturn(new ImmutableStereotypeModel("ST_NOT_NAME", "ST_NAMESPACE", null));
    final TestComponentAst innerB = new TestComponentAst(hasStereotypeModel);
    innerB.setParameters(singletonMap("name", "dependencyId"));

    final ArtifactAstDependencyGraph graph = generateFor(new TestArtifactAst(innerA, innerB));

    assertThat(graph.getMissingDependencies(), hasSize(1));

    final ComponentAstDependency missingDep = graph.getMissingDependencies().iterator().next();
    assertThat(missingDep.getName(), is("dependencyId"));
    assertThat(missingDep.getAllowedStereotypes(), hasSize(1));
    assertThat(missingDep.getAllowedStereotypes(), hasItem(new ImmutableStereotypeModel("ST_NAME", "ST_NAMESPACE", null)));
  }

  @Test
  public void missingSameStereotypeReported() {
    final ParameterizedModel parameterizedModel = mock(ParameterizedModel.class);
    final ParameterModel parameterModel = mock(ParameterModel.class);
    when(parameterModel.getAllowedStereotypes())
        .thenReturn(asList(new ImmutableStereotypeModel("ST_NAME", "ST_NAMESPACE", null)));
    when(parameterModel.getName()).thenReturn("nameParam");
    when(parameterizedModel.getAllParameterModels()).thenReturn(asList(parameterModel));

    final TestComponentAst innerA = new TestComponentAst(parameterizedModel);
    innerA.setParameters(singletonMap("nameParam", "dependencyId"));

    final HasStereotypeModel hasStereotypeModel = mock(HasStereotypeModel.class);
    when(hasStereotypeModel.getStereotype()).thenReturn(new ImmutableStereotypeModel("ST_NAME", "ST_NAMESPACE", null));
    final TestComponentAst innerB = new TestComponentAst(hasStereotypeModel);
    innerB.setParameters(singletonMap("name", "notMatchingDependencyId"));

    final ArtifactAstDependencyGraph graph = generateFor(new TestArtifactAst(innerA, innerB));

    assertThat(graph.getMissingDependencies(), hasSize(1));

    final ComponentAstDependency missingDep = graph.getMissingDependencies().iterator().next();
    assertThat(missingDep.getName(), is("dependencyId"));
    assertThat(missingDep.getAllowedStereotypes(), hasSize(1));
    assertThat(missingDep.getAllowedStereotypes(), hasItem(new ImmutableStereotypeModel("ST_NAME", "ST_NAMESPACE", null)));
  }

  @Test
  public void referencingParameterNotSet() {
    final ParameterizedModel parameterizedModel = mock(ParameterizedModel.class);
    final ParameterModel parameterModel = mock(ParameterModel.class);
    when(parameterModel.getAllowedStereotypes())
        .thenReturn(asList(new ImmutableStereotypeModel("ST_NAME", "ST_NAMESPACE", null)));
    when(parameterModel.getName()).thenReturn("nameParam");
    when(parameterizedModel.getAllParameterModels()).thenReturn(asList(parameterModel));

    final TestComponentAst innerA = new TestComponentAst(parameterizedModel);
    // No value set for the parameter with the reference
    innerA.setParameters(emptyMap());

    final HasStereotypeModel hasStereotypeModel = mock(HasStereotypeModel.class);
    when(hasStereotypeModel.getStereotype()).thenReturn(new ImmutableStereotypeModel("ST_NOT_NAME", "ST_NAMESPACE", null));
    final TestComponentAst innerB = new TestComponentAst(hasStereotypeModel);
    innerB.setParameters(singletonMap("name", "dependencyId"));

    final ArtifactAstDependencyGraph graph = generateFor(new TestArtifactAst(innerA, innerB));

    assertThat(graph.getMissingDependencies(), is(empty()));
  }

  @Test
  public void referencingParameterNotSetDefaultValue() {
    final ParameterizedModel parameterizedModel = mock(ParameterizedModel.class);
    final ParameterModel parameterModel = mock(ParameterModel.class);
    when(parameterModel.getAllowedStereotypes())
        .thenReturn(asList(new ImmutableStereotypeModel("ST_NAME", "ST_NAMESPACE", null)));
    when(parameterModel.getName()).thenReturn("nameParam");
    when(parameterModel.getDefaultValue()).thenReturn("dependencyId");
    when(parameterizedModel.getAllParameterModels()).thenReturn(asList(parameterModel));

    final TestComponentAst innerA = new TestComponentAst(parameterizedModel);
    // No value set for the parameter with the reference
    innerA.setParameters(emptyMap());

    final HasStereotypeModel hasStereotypeModel = mock(HasStereotypeModel.class);
    when(hasStereotypeModel.getStereotype()).thenReturn(new ImmutableStereotypeModel("ST_NOT_NAME", "ST_NAMESPACE", null));
    final TestComponentAst innerB = new TestComponentAst(hasStereotypeModel);
    innerB.setParameters(singletonMap("name", "dependencyId"));

    final ArtifactAstDependencyGraph graph = generateFor(new TestArtifactAst(innerA, innerB));

    assertThat(graph.getMissingDependencies(), hasSize(1));
  }

  @Test
  public void transitivesHandlingComparator() {
    final TestComponentAst inner = new TestComponentAst();
    inner.setParameters(singletonMap("name", "inner"));
    final TestComponentAst intermediateLevel = new TestComponentAst(inner);
    intermediateLevel.setParameters(singletonMap("name", "intermediate"));
    final TestComponentAst topLevel = new TestComponentAst(intermediateLevel);
    topLevel.setParameters(singletonMap("name", "top"));

    final ArtifactAstDependencyGraph graph = generateFor(new TestArtifactAst(topLevel));

    final List<TestComponentAst> sorted = asList(inner, intermediateLevel, topLevel);
    final List<TestComponentAst> unorderedA = asList(intermediateLevel, inner, topLevel);
    final List<TestComponentAst> unorderedB = asList(intermediateLevel, topLevel, inner);
    final List<TestComponentAst> unorderedC = asList(inner, topLevel, intermediateLevel);
    final List<TestComponentAst> unorderedD = asList(topLevel, inner, intermediateLevel);
    final List<TestComponentAst> unorderedE = asList(topLevel, intermediateLevel, inner);

    sort(unorderedA, graph.dependencyComparator());
    assertThat(unorderedA, is(sorted));

    sort(unorderedB, graph.dependencyComparator());
    assertThat(unorderedB, is(sorted));

    sort(unorderedC, graph.dependencyComparator());
    assertThat(unorderedC, is(sorted));

    sort(unorderedD, graph.dependencyComparator());
    assertThat(unorderedD, is(sorted));

    sort(unorderedE, graph.dependencyComparator());
    assertThat(unorderedE, is(sorted));
  }

  @Test
  public void transitivesHandlingComparatorNotHandlingCorrectlyTopLevelElements() {
    final TestComponentAst top1 = new TestComponentAst();
    top1.setParameters(singletonMap("name", "top1"));
    final TestComponentAst inner = new TestComponentAst();
    inner.setParameters(singletonMap("name", "inner"));
    final TestComponentAst top2 = new TestComponentAst(top1, inner);
    top2.setParameters(singletonMap("name", "top2"));
    final TestComponentAst top3 = new TestComponentAst(top1);
    top3.setParameters(singletonMap("name", "top3"));

    final ArtifactAstDependencyGraph graph = generateFor(new TestArtifactAst(top1, top2, top3));

    final List<TestComponentAst> sorted = asList(top1, inner, top3, top2);
    final List<TestComponentAst> unordered = asList(top1, top3, top2, inner);

    sort(unordered, graph.dependencyComparator());
    assertThat(unordered, is(sorted));
  }

  @Test
  public void comparatorShouldWorkWithCyclicDependenciesInGraph() {
    final ParameterizedModel parameterizedModel =
        mock(ParameterizedModel.class, withSettings().extraInterfaces(HasStereotypeModel.class));
    ImmutableStereotypeModel stereotypeModel = new ImmutableStereotypeModel("ST_NAME", "ST_NAMESPACE", null);
    when(((HasStereotypeModel) parameterizedModel).getStereotype()).thenReturn(stereotypeModel);
    List<StereotypeModel> stereotypeModelList = asList(stereotypeModel);
    final ParameterModel refParameterModel = mock(ParameterModel.class);
    when(refParameterModel.getAllowedStereotypes()).thenReturn(stereotypeModelList);
    when(refParameterModel.getName()).thenReturn("refParam");
    when(parameterizedModel.getAllParameterModels()).thenReturn(asList(refParameterModel));

    final TestComponentAst innerA = new TestComponentAst(parameterizedModel);
    innerA.setParameters(ImmutableMap.of("name", "dependencyId1", "refParam", "dependencyId2"));

    final TestComponentAst innerB = new TestComponentAst(parameterizedModel);
    innerB.setParameters(ImmutableMap.of("name", "dependencyId2", "refParam", "dependencyId1"));

    final ArtifactAstDependencyGraph graph = generateFor(new TestArtifactAst(innerA, innerB));

    final Collection<ComponentAst> innerAMinimalComponents =
        graph.minimalArtifactFor(innerA).recursiveStream().collect(toSet());
    assertThat(innerAMinimalComponents, hasItem(innerA));
    assertThat(innerAMinimalComponents, hasItem(innerB));

    final Collection<ComponentAst> innerBMinimalComponents =
        graph.minimalArtifactFor(innerB).recursiveStream().collect(toSet());
    assertThat(innerBMinimalComponents, hasItem(innerB));
    assertThat(innerBMinimalComponents, hasItem(innerA));

    final List<TestComponentAst> sorted = asList(innerB, innerA);
    final List<TestComponentAst> unordered = asList(innerA, innerB);

    sort(unordered, graph.dependencyComparator());
    assertThat(unordered, is(sorted));
  }

  @Test
  public void unrelatedComponentsComparator() {
    final ComponentAst innerA = new TestComponentAst();
    final ComponentAst innerB = new TestComponentAst();

    final ArtifactAstDependencyGraph graph = generateFor(new TestArtifactAst(innerA, innerB));

    final List<ComponentAst> unorderedA = asList(innerA, innerB);
    final List<ComponentAst> unorderedB = asList(innerB, innerA);

    sort(unorderedA, graph.dependencyComparator());
    assertThat(unorderedA, hasSize(2));

    sort(unorderedB, graph.dependencyComparator());
    assertThat(unorderedB, hasSize(2));
  }

  @Test
  public void sameComponentsComparator() {
    final ComponentAst innerA = new TestComponentAst();
    final ComponentAst innerB = innerA;

    final ArtifactAstDependencyGraph graph = generateFor(new TestArtifactAst(innerA, innerB));

    final List<ComponentAst> unorderedA = asList(innerA, innerB);
    final List<ComponentAst> unorderedB = asList(innerB, innerA);

    sort(unorderedA, graph.dependencyComparator());
    assertThat(unorderedA, hasSize(2));

    sort(unorderedB, graph.dependencyComparator());
    assertThat(unorderedB, hasSize(2));
  }

  @Test
  public void dependencyCycle() {
    final ImmutableStereotypeModel stereotype1 = new ImmutableStereotypeModel("ST1_NAME", "ST_NAMESPACE", null);
    final ImmutableStereotypeModel stereotype2 = new ImmutableStereotypeModel("ST2_NAME", "ST_NAMESPACE", null);

    final ParameterizedModel parameterizedModel =
        mock(ParameterizedModel.class, withSettings().extraInterfaces(HasStereotypeModel.class));
    final ParameterModel parameterModel = mock(ParameterModel.class);
    when(parameterModel.getAllowedStereotypes()).thenReturn(asList(stereotype1));
    when(parameterModel.getName()).thenReturn("nameParam");
    when(parameterizedModel.getAllParameterModels()).thenReturn(asList(parameterModel));
    when(((HasStereotypeModel) parameterizedModel).getStereotype()).thenReturn(stereotype2);

    final TestComponentAst innerA = new TestComponentAst(parameterizedModel);
    final Map<String, String> rawParamsA = new HashMap<>();
    rawParamsA.put("name", "dependency2Id");
    rawParamsA.put("nameParam", "dependencyId");
    innerA.setParameters(rawParamsA);

    final HasStereotypeModel hasStereotypeModel =
        mock(HasStereotypeModel.class, withSettings().extraInterfaces(ParameterizedModel.class));
    when(hasStereotypeModel.getStereotype()).thenReturn(stereotype1);
    final ParameterModel parameter2Model = mock(ParameterModel.class);
    when(parameter2Model.getAllowedStereotypes()).thenReturn(asList(stereotype2));
    when(parameter2Model.getName()).thenReturn("nameParam");
    when(((ParameterizedModel) hasStereotypeModel).getAllParameterModels()).thenReturn(asList(parameter2Model));

    final TestComponentAst innerB = new TestComponentAst(hasStereotypeModel);
    final Map<String, String> rawParamsB = new HashMap<>();
    rawParamsB.put("name", "dependencyId");
    rawParamsB.put("nameParam", "dependency2Id");
    innerB.setParameters(rawParamsB);

    final ArtifactAstDependencyGraph graph = generateFor(new TestArtifactAst(innerA, innerB));

    final Collection<ComponentAst> innerAMinimalComponents =
        graph.minimalArtifactFor(innerA).recursiveStream().collect(toSet());
    assertThat(innerAMinimalComponents, hasItem(innerA));
    assertThat(innerAMinimalComponents, hasItem(innerB));
  }

}
