/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.ast.graph.api;

import static java.lang.String.format;

import org.mule.api.annotation.NoInstantiate;
import org.mule.runtime.api.meta.model.stereotype.HasStereotypeModel;
import org.mule.runtime.api.meta.model.stereotype.StereotypeModel;
import org.mule.runtime.ast.api.ComponentAst;

import java.util.Set;

/**
 * Object that represents a dependency on a {@link ComponentAst} with a given name and one of a set of given
 * {@link StereotypeModel}s.
 *
 * @since 1.0
 */
@NoInstantiate
public class ComponentAstDependency {

  private final String name;

  private final Set<StereotypeModel> allowedStereotypes;

  public ComponentAstDependency(String name, Set<StereotypeModel> allowedStereotypes) {
    this.name = name;
    this.allowedStereotypes = allowedStereotypes;
  }

  /**
   * @return the name of the dependency.
   */
  public String getName() {
    return name;
  }

  /**
   * @return the possible stereotypes that are accepted by this dependency.
   */
  public Set<StereotypeModel> getAllowedStereotypes() {
    return allowedStereotypes;
  }

  /**
   * Checks if the provided component satisfies this dependency
   *
   * @param component the component to check against
   * @return {@code true} is the provided component satisfies this dependency, {@code false} otherwise.
   */
  public boolean isSatisfiedBy(ComponentAst component) {
    return component.getModel(HasStereotypeModel.class)
        .filter(componentModel -> component.getComponentId()
            .map(name::equals)
            .orElse(false))
        .map(componentModel -> allowedStereotypes.stream()
            .anyMatch(s -> componentModel.getStereotype() != null && componentModel.getStereotype().isAssignableTo(s)))
        .orElse(false);
  }

  @Override
  public String toString() {
    return format("'%s' (%s)", getName(), getAllowedStereotypes().toString());
  }

  @Override
  public int hashCode() {
    final int prime = 31;
    int result = 1;
    result = prime * result + ((allowedStereotypes == null) ? 0 : allowedStereotypes.hashCode());
    result = prime * result + ((name == null) ? 0 : name.hashCode());
    return result;
  }

  @Override
  public boolean equals(Object obj) {
    if (this == obj) {
      return true;
    }
    if (obj == null) {
      return false;
    }
    if (getClass() != obj.getClass()) {
      return false;
    }
    ComponentAstDependency other = (ComponentAstDependency) obj;
    if (allowedStereotypes == null) {
      if (other.allowedStereotypes != null) {
        return false;
      }
    } else if (!allowedStereotypes.equals(other.allowedStereotypes)) {
      return false;
    }
    if (name == null) {
      if (other.name != null) {
        return false;
      }
    } else if (!name.equals(other.name)) {
      return false;
    }
    return true;
  }


}
