/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.ast.graph.api;

import org.mule.runtime.ast.api.ArtifactAst;
import org.mule.runtime.ast.api.ComponentAst;

import java.util.Comparator;
import java.util.Set;
import java.util.function.Predicate;

/**
 * Provides a means to start only a subset of the components of the artifact, relying on a dependency graph generated for its
 * components.
 * <p>
 * Implementations need not be thread safe.
 *
 * @since 1.0
 */
public interface ArtifactAstDependencyGraph {

  /**
   * Creates an {@link ArtifactAst} that only contains the minimally required components for the provided {@code vertex} to work
   * properly.
   *
   * @param vertex the component to calculate the minimal artifact for
   * @return the minimal artifact
   */
  ArtifactAst minimalArtifactFor(ComponentAst vertex);

  /**
   * Creates an {@link ArtifactAst} that only contains the minimally required components for those which the provided
   * {@code vertexPredicate} applies to work properly.
   *
   * @param vertexPredicate the predicate of the components to calculate the minimal artifact for
   * @return the minimal artifact
   */
  ArtifactAst minimalArtifactFor(Predicate<ComponentAst> vertexPredicate);

  /**
   * Provides a set of components that contains the required components for the componentName
   * ({@link ComponentAst#getComponentId()}) provided.
   *
   * @param componentName {@link ComponentAst#getComponentId()}, a potential component name to calculate the direct dependencies
   *                      for
   * @return a set of components that the component is dependent on
   *
   * @since 1.1.0
   */
  Set<ComponentAst> getRequiredComponents(String componentName);

  /**
   * Provides information about components referenced from within the artifact but not declared there.
   *
   * @return the information about the missing dependencies
   */
  Set<ComponentAstDependency> getMissingDependencies();

  /**
   * Provides a {@link Comparator} implementation that when used to sort dependencies.
   * <p>
   * Given 2 components, A and B where A depends on B, {@code compare(A, B)} will return a negative value and
   * {@code compare(B, A)} will return a positive value.
   * <p>
   * Note: this comparator imposes orderings that are inconsistent with equals. For 2 components that do not depend on each other,
   * the returned value will be 0, but that doesn't mean that the components are equal.
   */
  Comparator<ComponentAst> dependencyComparator();
}
