/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.ast.graph;

import static java.lang.String.CASE_INSENSITIVE_ORDER;
import static java.util.Collections.reverse;
import static java.util.stream.Collectors.toList;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.collection.IsIterableContainingInOrder.contains;

import org.mule.runtime.ast.graph.internal.cycle.GraphCycleRemover;

import java.util.ArrayList;
import java.util.List;

import org.jgrapht.Graph;
import org.jgrapht.graph.DefaultEdge;
import org.jgrapht.graph.SimpleDirectedGraph;
import org.jgrapht.traverse.TopologicalOrderIterator;
import org.junit.Test;

public class CycleRemoverTestCase {

  @Test
  public void removeSimpleCycle() {
    Graph<String, DefaultEdge> graph = new SimpleDirectedGraph(DefaultEdge.class);
    graph.addVertex("A");
    graph.addVertex("B");

    graph.addEdge("A", "B");
    graph.addEdge("B", "A");

    graph = new GraphCycleRemover<>(graph, CASE_INSENSITIVE_ORDER).removeCycles();
    List<String> vertexDependencyOrder = new ArrayList<>();
    new TopologicalOrderIterator<>(graph).forEachRemaining(vertex -> vertexDependencyOrder.add(vertex));
    reverse(vertexDependencyOrder);

    assertThat(vertexDependencyOrder, contains("B", "A"));
  }

  // A -> B -> C -> D -> B -> A
  @Test
  public void nestedCycles() {
    Graph<String, DefaultEdge> graph = new SimpleDirectedGraph(DefaultEdge.class);
    graph.addVertex("A");
    graph.addVertex("B");
    graph.addVertex("C");
    graph.addVertex("D");

    graph.addEdge("A", "B");
    graph.addEdge("B", "C");
    graph.addEdge("C", "D");
    graph.addEdge("D", "B");
    graph.addEdge("B", "A");

    graph = new GraphCycleRemover<>(graph, CASE_INSENSITIVE_ORDER).removeCycles();
    List<String> vertexDependencyOrder = new ArrayList<>();
    new TopologicalOrderIterator<>(graph).forEachRemaining(vertex -> vertexDependencyOrder.add(vertex));
    reverse(vertexDependencyOrder);

    assertThat(vertexDependencyOrder, contains("D", "C", "B", "A"));
  }

  // A -> B -> C -> A, D -> E -> F -> D
  @Test
  public void subGraphsWithCycles() {
    Graph<String, DefaultEdge> graph = new SimpleDirectedGraph(DefaultEdge.class);
    graph.addVertex("A");
    graph.addVertex("B");
    graph.addVertex("C");
    graph.addVertex("D");
    graph.addVertex("E");
    graph.addVertex("F");

    graph.addEdge("A", "B");
    graph.addEdge("B", "C");
    graph.addEdge("C", "A");

    graph.addEdge("D", "E");
    graph.addEdge("E", "F");
    graph.addEdge("F", "D");

    graph = new GraphCycleRemover(graph, CASE_INSENSITIVE_ORDER).removeCycles();
    List<String> vertexDependencyOrder = new ArrayList<>();
    new TopologicalOrderIterator<>(graph).forEachRemaining(vertex -> vertexDependencyOrder.add(vertex));
    reverse(vertexDependencyOrder);

    assertThat(graph.edgeSet().stream().map(vertex -> vertex.toString()).collect(toList()),
               contains("(A : B)", "(B : C)", "(D : E)", "(E : F)"));
    assertThat(vertexDependencyOrder, contains("F", "C", "E", "B", "D", "A"));
  }

  // A -> B -> C -> A, D -> E -> F -> D
  @Test
  public void subGraphsWithCyclesReverseOrder() {
    Graph<String, DefaultEdge> graph = new SimpleDirectedGraph(DefaultEdge.class);
    graph.addVertex("A");
    graph.addVertex("B");
    graph.addVertex("C");
    graph.addVertex("D");
    graph.addVertex("E");
    graph.addVertex("F");

    graph.addEdge("A", "B");
    graph.addEdge("B", "C"); // This one will be removed as C comes before B (reverse order)
    graph.addEdge("C", "A");

    graph.addEdge("D", "E");
    graph.addEdge("E", "F"); // This one will be removed as F comes before D (reverse order)
    graph.addEdge("F", "D");

    graph = new GraphCycleRemover(graph, CASE_INSENSITIVE_ORDER.reversed()).removeCycles();
    List<String> vertexDependencyOrder = new ArrayList<>();
    new TopologicalOrderIterator<>(graph).forEachRemaining(vertex -> vertexDependencyOrder.add(vertex));
    reverse(vertexDependencyOrder);

    assertThat(graph.edgeSet().stream().map(vertex -> vertex.toString()).collect(toList()),
               contains("(A : B)", "(C : A)", "(D : E)", "(F : D)"));
    assertThat(vertexDependencyOrder, contains("E", "B", "D", "A", "F", "C"));
  }

  // A -> B -> C -> A, D -> E -> F -> D, B -> F
  @Test
  public void subGraphsWithCyclesReferencesBetweenSubCycles() {
    Graph<String, DefaultEdge> graph = new SimpleDirectedGraph(DefaultEdge.class);
    graph.addVertex("A");
    graph.addVertex("B");
    graph.addVertex("C");
    graph.addVertex("D");
    graph.addVertex("E");
    graph.addVertex("F");

    graph.addEdge("A", "B");
    graph.addEdge("B", "C");
    graph.addEdge("C", "A");

    graph.addEdge("D", "E");
    graph.addEdge("E", "F");
    graph.addEdge("F", "D");

    graph.addEdge("B", "F");

    graph = new GraphCycleRemover(graph, CASE_INSENSITIVE_ORDER).removeCycles();
    List<String> vertexDependencyOrder = new ArrayList<>();
    new TopologicalOrderIterator<>(graph).forEachRemaining(vertex -> vertexDependencyOrder.add(vertex));
    reverse(vertexDependencyOrder);

    assertThat(vertexDependencyOrder, contains("F", "C", "E", "B", "D", "A"));
  }

  // A -> B -> C -> A, D -> E -> F -> D, B -> F
  // Order F, E, D, C, B, A
  @Test
  public void subGraphsWithCyclesReferencesBetweenSubCyclesReverseOrder() {
    Graph<String, DefaultEdge> graph = new SimpleDirectedGraph(DefaultEdge.class);
    graph.addVertex("A");
    graph.addVertex("B");
    graph.addVertex("C");
    graph.addVertex("D");
    graph.addVertex("E");
    graph.addVertex("F");

    graph.addEdge("A", "B");
    graph.addEdge("B", "C");
    graph.addEdge("C", "A");

    graph.addEdge("D", "E");
    graph.addEdge("E", "F");
    graph.addEdge("F", "D");

    graph.addEdge("B", "F");

    graph = new GraphCycleRemover(graph, CASE_INSENSITIVE_ORDER.reversed()).removeCycles();
    List<String> vertexDependencyOrder = new ArrayList<>();
    new TopologicalOrderIterator<>(graph).forEachRemaining(vertex -> vertexDependencyOrder.add(vertex));
    reverse(vertexDependencyOrder);

    assertThat(vertexDependencyOrder, contains("E", "D", "F", "B", "A", "C"));
  }

  @Test
  public void noCyclesNestedDependency() {
    Graph<String, DefaultEdge> graph = new SimpleDirectedGraph(DefaultEdge.class);
    graph.addVertex("A");
    graph.addVertex("B");
    graph.addVertex("C");

    graph.addEdge("A", "C");
    graph.addEdge("A", "B");
    graph.addEdge("B", "C");

    graph = new GraphCycleRemover<>(graph, CASE_INSENSITIVE_ORDER).removeCycles();
    List<String> vertexDependencyOrder = new ArrayList<>();
    new TopologicalOrderIterator<>(graph).forEachRemaining(vertex -> vertexDependencyOrder.add(vertex));
    reverse(vertexDependencyOrder);

    assertThat(vertexDependencyOrder, contains("C", "B", "A"));
  }

}
