/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.ast.graph.api;

import static org.mule.runtime.ast.api.util.MuleAstUtils.doOnParamComponents;

import static org.slf4j.LoggerFactory.getLogger;

import org.mule.runtime.api.meta.model.parameter.ParameterGroupModel;
import org.mule.runtime.api.meta.model.parameter.ParameterModel;
import org.mule.runtime.api.meta.model.parameter.ParameterizedModel;
import org.mule.runtime.api.util.Pair;
import org.mule.runtime.ast.api.ArtifactAst;
import org.mule.runtime.ast.api.ComponentAst;
import org.mule.runtime.ast.api.ComponentParameterAst;
import org.mule.runtime.ast.graph.internal.ComponentAstEdge;
import org.mule.runtime.ast.graph.internal.DefaultArtifactAstDependencyGraph;

import java.util.HashSet;
import java.util.Objects;
import java.util.Set;

import org.jgrapht.Graph;
import org.jgrapht.graph.SimpleDirectedGraph;
import org.slf4j.Logger;

/**
 * Contains a factory method for creating {@link ArtifactAstDependencyGraph}s.
 *
 * @since 1.0
 */
public class ArtifactAstDependencyGraphFactory {

  private static final Logger LOGGER = getLogger(ArtifactAstDependencyGraphFactory.class);

  private ArtifactAstDependencyGraphFactory() {
    // Nothing to do
  }

  /**
   * Creates an {@link ArtifactAstDependencyGraph} for the given {@link ArtifactAst}.
   * <p>
   * This method relies on the stereotypes in the extension model for determining the dependencies.
   *
   * @param fullArtifactAst the {@link ArtifactAst} to generate the graph for.
   * @return a graph instance with the dependencies between the artifact components resolved.
   */
  public static ArtifactAstDependencyGraph generateFor(ArtifactAst fullArtifactAst) {

    // The edges of the graph represent a "depends on" relation.
    final Graph<ComponentAst, ComponentAstEdge> graph = new SimpleDirectedGraph<>(ComponentAstEdge.class);

    // First pass, add all components to the graph
    fullArtifactAst.recursiveStream().forEach(graph::addVertex);

    Set<ComponentAstDependency> missingDependencies = new HashSet<>();

    // Second pass, determine the dependencies
    fullArtifactAst.recursiveStream()
        .forEach(comp -> {
          // A component depends on its children...
          comp.directChildrenStream().forEach(child -> addDependency(graph, comp, child));

          // ... and its parameters if has parameters
          comp.getModel(ParameterizedModel.class).ifPresent(parameterizedModel -> {
            comp.getParameters()
                .forEach(param -> doOnParamComponents(param, child -> addDependency(graph, comp, child)));
          });

          // For components that have parameters, add the dependencies when those parameters are references
          comp.getModel(ParameterizedModel.class)
              .ifPresent(parameterizedModel -> parameterizedModel.getParameterGroupModels().stream()
                  .flatMap(g -> g.getParameterModels().stream().map(p -> new Pair<>(g, p)))
                  .filter(groupAndParam -> !groupAndParam.getSecond().getAllowedStereotypes().isEmpty())
                  .filter(groupAndParam -> parameterValue(comp, groupAndParam.getFirst(), groupAndParam.getSecond()) != null)
                  .map(groupAndParam -> new ComponentAstDependency(comp,
                                                                   comp.getParameter(groupAndParam.getFirst().getName(),
                                                                                     groupAndParam.getSecond().getName()),
                                                                   parameterValue(comp, groupAndParam.getFirst(),
                                                                                  groupAndParam.getSecond()),
                                                                   new HashSet<>(groupAndParam.getSecond()
                                                                       .getAllowedStereotypes())))
                  .forEach(dependency -> {
                    if (missingDependencies.contains(dependency)) {
                      // If this dependency is already identified as missing, skip it
                      return;
                    }

                    // Iterate through the whole AST looking for a component with a matching id and stereotype
                    // If there was a dependency already found, skip the stereotype matching
                    fullArtifactAst
                        .filteredComponents(tlcm -> !graph.containsEdge(comp, tlcm))
                        .filter(dependency::isSatisfiedBy)
                        .forEach(tlcm -> addDependency(graph, comp, tlcm));

                    if (graph.outgoingEdgesOf(comp)
                        .stream()
                        .noneMatch(edge -> dependency.isSatisfiedBy(edge.getTarget()))) {
                      missingDependencies.add(dependency);
                    }
                  }));
        });

    if (!missingDependencies.isEmpty() && LOGGER.isDebugEnabled()) {
      LOGGER.debug("Missing dependencies:");
      missingDependencies.forEach(rq -> LOGGER.debug("    {}", rq));
    }

    return new DefaultArtifactAstDependencyGraph(fullArtifactAst, graph, missingDependencies);
  }

  private static String parameterValue(ComponentAst comp, ParameterGroupModel groupModel, ParameterModel paramModel) {
    final ComponentParameterAst param = comp.getParameter(groupModel.getName(), paramModel.getName());
    if (param != null && param.getValue().isRight() && param.getValue().getRight() != null) {
      return param.getValue().getRight() instanceof String
          ? Objects.toString(param.getValue().getRight())
          : null;
    } else {
      return null;
    }
  }

  private static void addDependency(final Graph<ComponentAst, ComponentAstEdge> graph,
                                    final ComponentAst dependencyFrom,
                                    final ComponentAst dependencyTo) {
    LOGGER.debug("Dependency found: '{}' to '{}'", dependencyFrom, dependencyTo);
    graph.addEdge(dependencyFrom, dependencyTo);
  }
}
