/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.ast.graph.internal;

import static org.mule.runtime.ast.AllureConstants.LifecycleAndDependencyInjectionFeature.LIFECYCLE_AND_DEPENDENCY_INJECTION;
import static org.mule.runtime.ast.AllureConstants.LifecycleAndDependencyInjectionFeature.LifecyclePhaseStory.LIFECYCLE_PHASE_STORY;

import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.containsInAnyOrder;
import static org.hamcrest.Matchers.is;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.when;

import org.mule.runtime.ast.api.ArtifactAst;
import org.mule.runtime.ast.api.ComponentAst;
import org.mule.runtime.ast.graph.api.ComponentAstDependency;
import org.mule.runtime.ast.test.internal.TestComponentAst;

import java.util.HashSet;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Stream;

import org.jgrapht.Graph;
import org.jgrapht.graph.SimpleDirectedGraph;

import org.junit.Before;
import org.junit.Test;

import io.qameta.allure.Description;
import io.qameta.allure.Feature;
import io.qameta.allure.Issue;
import io.qameta.allure.Story;

@Issue("MULE-19984")
@Feature(LIFECYCLE_AND_DEPENDENCY_INJECTION)
@Story(LIFECYCLE_PHASE_STORY)
public class DefaultArtifactAstDependencyGraphTestCase {

  private DefaultArtifactAstDependencyGraph artifactAstDependencyGraph;
  private ArtifactAst source;
  private Graph<ComponentAst, ComponentAstEdge> graph;
  private Set<ComponentAstDependency> missingDependencies;

  @Before
  public void setUp() throws Exception {
    this.source = mock(ArtifactAst.class);
    this.graph = mock(SimpleDirectedGraph.class);
    this.missingDependencies = new HashSet<>();
    this.artifactAstDependencyGraph = new DefaultArtifactAstDependencyGraph(source, graph, missingDependencies);
  }



  @Test
  @Description("If A depends on B, B depends on C, getRequiredComponents(A) will return a set of A, B, and C.")
  public void transitiveDependencyTest() {
    TestComponentAst componentA = mock(TestComponentAst.class);
    TestComponentAst componentB = mock(TestComponentAst.class);
    TestComponentAst componentC = mock(TestComponentAst.class);

    when(componentA.getComponentId()).thenReturn(Optional.of("componentA_name"));
    when(source.recursiveStream()).thenReturn(Stream.of(componentA));
    when(componentA.toString()).thenReturn("componentA");
    when(componentB.toString()).thenReturn("componentB");
    when(componentC.toString()).thenReturn("componentC");

    ComponentAstEdge edgeFromAToB = mock(ComponentAstEdge.class);
    ComponentAstEdge edgeFromBToC = mock(ComponentAstEdge.class);
    Set<ComponentAstEdge> outgoingEdgesOfA = new HashSet<>();
    Set<ComponentAstEdge> outgoingEdgesOfB = new HashSet<>();
    Set<ComponentAstEdge> edges = new HashSet<>();

    when(edgeFromAToB.getSource()).thenReturn(componentA);
    when(edgeFromAToB.getTarget()).thenReturn(componentB);
    outgoingEdgesOfA.add(edgeFromAToB);

    when(edgeFromBToC.getSource()).thenReturn(componentB);
    when(edgeFromBToC.getTarget()).thenReturn(componentC);
    outgoingEdgesOfB.add(edgeFromBToC);

    edges.add(edgeFromAToB);
    edges.add(edgeFromBToC);

    when(graph.outgoingEdgesOf(componentA)).thenReturn(outgoingEdgesOfA);
    when(graph.outgoingEdgesOf(componentB)).thenReturn(outgoingEdgesOfB);
    when(edgeFromAToB.transitiveOutgoingDependenciesOf(graph)).thenReturn(Stream.of(componentA, componentB, componentC));

    Set<ComponentAst> requiredComponents = artifactAstDependencyGraph.getRequiredComponents("componentA_name");

    assertThat(requiredComponents.size(), is(3));
    assertThat(requiredComponents, containsInAnyOrder(componentC, componentB, componentA));
  }

  @Test
  @Description("If A depends on B, C depends on D, getRequiredComponents(A) will return a set of A and B only")
  public void separatedDependencyTest() {
    TestComponentAst componentA = mock(TestComponentAst.class);
    TestComponentAst componentB = mock(TestComponentAst.class);
    TestComponentAst componentC = mock(TestComponentAst.class);
    TestComponentAst componentD = mock(TestComponentAst.class);

    when(componentA.getComponentId()).thenReturn(Optional.of("componentA_name"));
    when(source.recursiveStream()).thenReturn(Stream.of(componentA));
    when(componentA.toString()).thenReturn("componentA");
    when(componentB.toString()).thenReturn("componentB");
    when(componentC.toString()).thenReturn("componentC");
    when(componentD.toString()).thenReturn("componentD");

    ComponentAstEdge edgeFromAToB = mock(ComponentAstEdge.class);
    ComponentAstEdge edgeFromCToD = mock(ComponentAstEdge.class);
    Set<ComponentAstEdge> outgoingEdgesOfA = new HashSet<>();
    Set<ComponentAstEdge> outgoingEdgesOfC = new HashSet<>();
    Set<ComponentAstEdge> outgoingEdgesOfB = new HashSet<>();

    when(edgeFromAToB.getSource()).thenReturn(componentA);
    when(edgeFromAToB.getTarget()).thenReturn(componentB);
    outgoingEdgesOfA.add(edgeFromAToB);

    when(edgeFromCToD.getSource()).thenReturn(componentC);
    when(edgeFromCToD.getTarget()).thenReturn(componentD);
    outgoingEdgesOfC.add(edgeFromCToD);


    when(graph.outgoingEdgesOf(componentA)).thenReturn(outgoingEdgesOfA);
    when(graph.outgoingEdgesOf(componentB)).thenReturn(outgoingEdgesOfB);
    when(graph.outgoingEdgesOf(componentC)).thenReturn(outgoingEdgesOfC);
    when(edgeFromAToB.transitiveOutgoingDependenciesOf(graph)).thenReturn(Stream.of(componentA, componentB));

    Set<ComponentAst> requiredComponents = artifactAstDependencyGraph.getRequiredComponents("componentA_name");

    assertThat(requiredComponents.size(), is(2));
    assertThat(requiredComponents, containsInAnyOrder(componentA, componentB));
  }
}
