/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.ast.graph.api;

import static org.mule.metadata.api.model.MetadataFormat.JAVA;
import static org.mule.runtime.api.meta.model.parameter.ParameterGroupModel.DEFAULT_GROUP_NAME;
import static org.mule.runtime.ast.api.ComponentGenerationInformation.EMPTY_GENERATION_INFO;
import static org.mule.runtime.ast.api.ComponentMetadataAst.EMPTY_METADATA;
import static org.mule.runtime.ast.api.util.MuleAstUtils.recursiveStreamWithHierarchy;
import static org.mule.runtime.ast.graph.api.ArtifactAstDependencyGraphFactory.generateFor;

import static java.util.Arrays.asList;
import static java.util.Collections.emptyList;
import static java.util.Collections.emptyMap;
import static java.util.Collections.singletonList;
import static java.util.Collections.singletonMap;
import static java.util.Collections.sort;
import static java.util.Optional.of;
import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toSet;

import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.empty;
import static org.hamcrest.Matchers.emptyIterable;
import static org.hamcrest.collection.IsCollectionWithSize.hasSize;
import static org.hamcrest.core.Is.is;
import static org.hamcrest.core.IsIterableContaining.hasItem;
import static org.hamcrest.core.IsNot.not;
import static org.mockito.Mockito.doAnswer;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.spy;
import static org.mockito.Mockito.when;
import static org.mockito.Mockito.withSettings;

import org.mule.metadata.api.builder.BaseTypeBuilder;
import org.mule.metadata.api.builder.ObjectTypeBuilder;
import org.mule.metadata.api.model.ObjectType;
import org.mule.runtime.api.component.ComponentIdentifier;
import org.mule.runtime.api.meta.model.parameter.ParameterGroupModel;
import org.mule.runtime.api.meta.model.parameter.ParameterModel;
import org.mule.runtime.api.meta.model.parameter.ParameterizedModel;
import org.mule.runtime.api.meta.model.stereotype.HasStereotypeModel;
import org.mule.runtime.api.meta.model.stereotype.ImmutableStereotypeModel;
import org.mule.runtime.api.meta.model.stereotype.StereotypeModel;
import org.mule.runtime.api.util.Pair;
import org.mule.runtime.ast.api.ArtifactAst;
import org.mule.runtime.ast.api.ComponentAst;
import org.mule.runtime.ast.api.builder.ComponentAstBuilder;
import org.mule.runtime.ast.internal.DefaultComponentParameterAst;
import org.mule.runtime.ast.internal.builder.LightComponentAstBuilder;
import org.mule.runtime.ast.internal.builder.PropertiesResolver;
import org.mule.runtime.ast.internal.model.ParameterModelUtils;
import org.mule.runtime.ast.test.internal.TestArtifactAst;
import org.mule.runtime.ast.test.internal.TestComponentAst;
import org.mule.runtime.extension.api.declaration.type.annotation.StereotypeTypeAnnotation;

import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.atomic.AtomicReference;

import org.junit.Test;

import io.qameta.allure.Description;
import io.qameta.allure.Issue;

public class ArtifactAstDependencyGraphFactoryTestCase {

  @Test
  public void simpleTopLevelWithInner() {
    final ComponentAst inner = new TestComponentAst();
    final ComponentAst topLevel = new TestComponentAst(inner);

    final ArtifactAstDependencyGraph graph = generateFor(new TestArtifactAst(topLevel));

    final Collection<ComponentAst> topLevelMinimalComponents =
        graph.minimalArtifactFor(topLevel).recursiveStream().collect(toSet());
    assertThat(topLevelMinimalComponents, hasItem(topLevel));
    assertThat(topLevelMinimalComponents, hasItem(inner));
  }

  @Test
  public void simpleTopLevelWithInnerPredicate() {
    final ComponentAst inner = new TestComponentAst();
    final ComponentAst topLevel = new TestComponentAst(inner);

    final ArtifactAstDependencyGraph graph = generateFor(new TestArtifactAst(topLevel));

    final Collection<ComponentAst> topLevelMinimalComponents =
        graph.minimalArtifactFor(topLevel::equals).recursiveStream().collect(toSet());
    assertThat(topLevelMinimalComponents, hasItem(topLevel));
    assertThat(topLevelMinimalComponents, hasItem(inner));
  }

  @Test
  public void unrelatedComponents() {
    final ComponentAst innerA = new TestComponentAst();
    final ComponentAst innerB = new TestComponentAst();

    final ArtifactAstDependencyGraph graph = generateFor(new TestArtifactAst(innerA, innerB));

    final Collection<ComponentAst> innerAMinimalComponents =
        graph.minimalArtifactFor(innerA).recursiveStream().collect(toSet());
    assertThat(innerAMinimalComponents, hasItem(innerA));
    assertThat(innerAMinimalComponents, not(hasItem(innerB)));

    final Collection<ComponentAst> innerBMinimalComponents =
        graph.minimalArtifactFor(innerB).recursiveStream().collect(toSet());
    assertThat(innerBMinimalComponents, hasItem(innerB));
    assertThat(innerBMinimalComponents, not(hasItem(innerA)));
  }

  @Test
  public void unrelatedComponentsPredicate() {
    final ComponentAst innerA = new TestComponentAst();
    final ComponentAst innerB = new TestComponentAst();

    final ArtifactAstDependencyGraph graph = generateFor(new TestArtifactAst(innerA, innerB));

    final Collection<ComponentAst> innerAMinimalComponents =
        graph.minimalArtifactFor(innerA::equals).recursiveStream().collect(toSet());
    assertThat(innerAMinimalComponents, hasItem(innerA));
    assertThat(innerAMinimalComponents, not(hasItem(innerB)));

    final Collection<ComponentAst> innerBMinimalComponents =
        graph.minimalArtifactFor(innerB::equals).recursiveStream().collect(toSet());
    assertThat(innerBMinimalComponents, hasItem(innerB));
    assertThat(innerBMinimalComponents, not(hasItem(innerA)));
  }

  @Test
  public void noAllowedStereotypes() {
    final ParameterizedModel parameterizedModel = mock(ParameterizedModel.class);
    final ParameterModel parameterModel = mock(ParameterModel.class);
    when(parameterModel.getAllowedStereotypes()).thenReturn(emptyList());
    when(parameterModel.getName()).thenReturn("nameParam");

    ParameterGroupModel groupModel = createMockParameterGroup(parameterModel);
    when(parameterizedModel.getParameterGroupModels()).thenReturn(singletonList(groupModel));

    final TestComponentAst innerA = new TestComponentAst(parameterizedModel);
    innerA.setParameters(singletonMap("nameParam", "dependencyId"));
    final TestComponentAst innerB = new TestComponentAst();

    final ArtifactAstDependencyGraph graph = generateFor(new TestArtifactAst(innerA, innerB));

    final Collection<ComponentAst> innerAMinimalComponents =
        graph.minimalArtifactFor(innerA).recursiveStream().collect(toSet());
    assertThat(innerAMinimalComponents, hasItem(innerA));
    assertThat(innerAMinimalComponents, not(hasItem(innerB)));

    final Collection<ComponentAst> innerBMinimalComponents =
        graph.minimalArtifactFor(innerB).recursiveStream().collect(toSet());
    assertThat(innerBMinimalComponents, hasItem(innerB));
    assertThat(innerBMinimalComponents, not(hasItem(innerA)));
  }

  @Test
  public void unmatchingStereotypesUnmatchingName() {
    final ParameterizedModel parameterizedModel = mock(ParameterizedModel.class);
    final ParameterModel parameterModel = mock(ParameterModel.class);
    when(parameterModel.getAllowedStereotypes())
        .thenReturn(asList(new ImmutableStereotypeModel("ST_NAME", "ST_NAMESPACE", null)));
    when(parameterModel.getName()).thenReturn("nameParam");

    ParameterGroupModel groupModel = createMockParameterGroup(parameterModel);
    when(parameterizedModel.getParameterGroupModels()).thenReturn(singletonList(groupModel));

    final TestComponentAst innerA = new TestComponentAst(parameterizedModel);
    innerA.setParameters(singletonMap("nameParam", "dependencyId"));
    final TestComponentAst innerB = new TestComponentAst();
    innerB.setParameters(singletonMap("name", "notMatchingDependencyId"));

    final ArtifactAstDependencyGraph graph = generateFor(new TestArtifactAst(innerA, innerB));

    final Collection<ComponentAst> innerAMinimalComponents =
        graph.minimalArtifactFor(innerA).recursiveStream().collect(toSet());
    assertThat(innerAMinimalComponents, hasItem(innerA));
    assertThat(innerAMinimalComponents, not(hasItem(innerB)));

    final Collection<ComponentAst> innerBMinimalComponents =
        graph.minimalArtifactFor(innerB).recursiveStream().collect(toSet());
    assertThat(innerBMinimalComponents, hasItem(innerB));
    assertThat(innerBMinimalComponents, not(hasItem(innerA)));
  }

  @Test
  public void matchingStereotypesUnmatchingName() {
    final ParameterizedModel parameterizedModel = mock(ParameterizedModel.class);
    final ParameterModel parameterModel = mock(ParameterModel.class);
    when(parameterModel.getAllowedStereotypes())
        .thenReturn(asList(new ImmutableStereotypeModel("ST_NAME", "ST_NAMESPACE", null)));
    when(parameterModel.getName()).thenReturn("nameParam");

    ParameterGroupModel groupModel = createMockParameterGroup(parameterModel);
    when(parameterizedModel.getParameterGroupModels()).thenReturn(singletonList(groupModel));

    final TestComponentAst innerA = new TestComponentAst(parameterizedModel);
    innerA.setParameters(singletonMap("nameParam", "dependencyId"));

    final HasStereotypeModel hasStereotypeModel = mock(HasStereotypeModel.class);
    when(hasStereotypeModel.getStereotype()).thenReturn(new ImmutableStereotypeModel("ST_NAME", "ST_NAMESPACE", null));
    final TestComponentAst innerB = new TestComponentAst(hasStereotypeModel);
    innerB.setParameters(singletonMap("name", "notMatchingDependencyId"));

    final ArtifactAstDependencyGraph graph = generateFor(new TestArtifactAst(innerA, innerB));

    final Collection<ComponentAst> innerAMinimalComponents =
        graph.minimalArtifactFor(innerA).recursiveStream().collect(toSet());
    assertThat(innerAMinimalComponents, hasItem(innerA));
    assertThat(innerAMinimalComponents, not(hasItem(innerB)));

    final Collection<ComponentAst> innerBMinimalComponents =
        graph.minimalArtifactFor(innerB).recursiveStream().collect(toSet());
    assertThat(innerBMinimalComponents, hasItem(innerB));
    assertThat(innerBMinimalComponents, not(hasItem(innerA)));
  }

  @Test
  public void unmatchingStereotypesMatchingName() {
    final ParameterizedModel parameterizedModel = mock(ParameterizedModel.class);
    final ParameterModel parameterModel = mock(ParameterModel.class);
    when(parameterModel.getAllowedStereotypes())
        .thenReturn(asList(new ImmutableStereotypeModel("ST_NAME", "ST_NAMESPACE", null)));
    when(parameterModel.getName()).thenReturn("nameParam");

    ParameterGroupModel groupModel = createMockParameterGroup(parameterModel);
    when(parameterizedModel.getParameterGroupModels()).thenReturn(singletonList(groupModel));

    final TestComponentAst innerA = new TestComponentAst(parameterizedModel);
    innerA.setParameters(singletonMap("nameParam", "dependencyId"));

    final HasStereotypeModel hasStereotypeModel = mock(HasStereotypeModel.class);
    when(hasStereotypeModel.getStereotype()).thenReturn(new ImmutableStereotypeModel("ST_NOT_NAME", "ST_NAMESPACE", null));
    final TestComponentAst innerB = new TestComponentAst(hasStereotypeModel);
    innerB.setParameters(singletonMap("name", "dependencyId"));

    final ArtifactAstDependencyGraph graph = generateFor(new TestArtifactAst(innerA, innerB));

    final Collection<ComponentAst> innerAMinimalComponents =
        graph.minimalArtifactFor(innerA).recursiveStream().collect(toSet());
    assertThat(innerAMinimalComponents, hasItem(innerA));
    assertThat(innerAMinimalComponents, not(hasItem(innerB)));

    final Collection<ComponentAst> innerBMinimalComponents =
        graph.minimalArtifactFor(innerB).recursiveStream().collect(toSet());
    assertThat(innerBMinimalComponents, hasItem(innerB));
    assertThat(innerBMinimalComponents, not(hasItem(innerA)));
  }

  @Test
  @Description("innerA.nameParam is a reference to innerB")
  public void matchingStereotypesMatchingName() {
    final ParameterizedModel parameterizedModel = mock(ParameterizedModel.class);
    final ParameterModel parameterModel = mock(ParameterModel.class);
    when(parameterModel.getAllowedStereotypes())
        .thenReturn(asList(new ImmutableStereotypeModel("ST_NAME", "ST_NAMESPACE", null)));
    when(parameterModel.getName()).thenReturn("nameParam");

    ParameterGroupModel groupModel = createMockParameterGroup(parameterModel);
    when(parameterizedModel.getParameterGroupModels()).thenReturn(singletonList(groupModel));

    final TestComponentAst innerA = new TestComponentAst(parameterizedModel);
    innerA.setParameters(singletonMap("nameParam", "dependencyId"));

    final HasStereotypeModel hasStereotypeModel = mock(HasStereotypeModel.class);
    when(hasStereotypeModel.getStereotype()).thenReturn(new ImmutableStereotypeModel("ST_NAME", "ST_NAMESPACE", null));
    final TestComponentAst innerB = new TestComponentAst(hasStereotypeModel);
    innerB.setParameters(singletonMap("name", "dependencyId"));

    final ArtifactAstDependencyGraph graph = generateFor(new TestArtifactAst(innerA, innerB));

    final Collection<ComponentAst> innerAMinimalComponents =
        graph.minimalArtifactFor(innerA).recursiveStream().collect(toSet());
    assertThat(innerAMinimalComponents, hasItem(innerA));
    assertThat(innerAMinimalComponents, hasItem(innerB));

    final Collection<ComponentAst> innerBMinimalComponents =
        graph.minimalArtifactFor(innerB).recursiveStream().collect(toSet());
    assertThat(innerBMinimalComponents, hasItem(innerB));
    assertThat(innerBMinimalComponents, not(hasItem(innerA)));
  }

  @Test
  @Description("innerA.complexParam.nameParam is a reference to innerB")
  public void referenceFromComplexParam() {
    final ObjectTypeBuilder complexTypeBuilder = BaseTypeBuilder.create(JAVA).objectType();
    complexTypeBuilder.addField()
        .required()
        .key("nameParam")
        .value(BaseTypeBuilder.create(JAVA).stringType().build())
        .with(new StereotypeTypeAnnotation(asList(new ImmutableStereotypeModel("ST_NAME", "ST_NAMESPACE", null))))
        .build();
    final ObjectType complexType = complexTypeBuilder.build();

    final ParameterizedModel parameterizedModel = mock(ParameterizedModel.class);

    final ParameterModel complexParameterModel = mock(ParameterModel.class);
    when(complexParameterModel.getName()).thenReturn("complexParam");
    when(complexParameterModel.getType()).thenReturn(complexType);

    final ParameterGroupModel groupWithComplexParameter = createMockParameterGroup(complexParameterModel);
    when(parameterizedModel.getParameterGroupModels()).thenReturn(singletonList(groupWithComplexParameter));

    final ParameterizedModel complexParameterizedModel = mock(ParameterizedModel.class);
    final ParameterModel parameterModel = mock(ParameterModel.class);
    when(parameterModel.getAllowedStereotypes())
        .thenReturn(asList(new ImmutableStereotypeModel("ST_NAME", "ST_NAMESPACE", null)));
    when(parameterModel.getName()).thenReturn("nameParam");
    when(parameterModel.getType()).thenReturn(BaseTypeBuilder.create(JAVA).stringType().build());

    final ParameterGroupModel parameterGroupModel = createMockParameterGroup(parameterModel);
    when(complexParameterizedModel.getParameterGroupModels()).thenReturn(singletonList(parameterGroupModel));

    final ParameterGroupModel stGroup = mock(ParameterGroupModel.class);
    when(stGroup.getName()).thenReturn(DEFAULT_GROUP_NAME);
    ComponentIdentifier complexParamId = ComponentIdentifier.builder().namespace("mockns")
        .name("complex-param").build();
    ComponentAstBuilder innerAComplexParam = buildOnce(spy(new LightComponentAstBuilder(new ParameterModelUtils())
        .withParameterizedModel(complexParameterizedModel)
        .withParameter(parameterModel, stGroup,
                       new DefaultComponentParameterAst("dependencyId", parameterModel, stGroup, EMPTY_GENERATION_INFO,
                                                        new PropertiesResolver(),
                                                        new ParameterModelUtils()),
                       Optional.empty())))
                           .withIdentifier(complexParamId);

    final ParameterGroupModel complexGroup = mock(ParameterGroupModel.class);
    when(complexGroup.getName()).thenReturn(DEFAULT_GROUP_NAME);
    ComponentAst innerA = new LightComponentAstBuilder(new ParameterModelUtils())
        .withParameterizedModel(parameterizedModel)
        .withParameter(complexParameterModel, complexGroup,
                       new DefaultComponentParameterAst(innerAComplexParam, complexParameterModel, complexGroup,
                                                        EMPTY_METADATA, EMPTY_GENERATION_INFO, new PropertiesResolver(),
                                                        new ParameterModelUtils()),
                       of(complexParamId))
        .withIdentifier(ComponentIdentifier.builder().namespace("mockns").name("inner-a").build())
        .build();

    final HasStereotypeModel hasStereotypeModel = mock(HasStereotypeModel.class);
    when(hasStereotypeModel.getStereotype()).thenReturn(new ImmutableStereotypeModel("ST_NAME", "ST_NAMESPACE", null));
    final TestComponentAst innerB = new TestComponentAst(hasStereotypeModel);
    innerB.setParameters(singletonMap("name", "dependencyId"));

    final ArtifactAstDependencyGraph graph = generateFor(new TestArtifactAst(innerA, innerB));

    final Collection<ComponentAst> innerAMinimalComponents =
        graph.minimalArtifactFor(innerA).recursiveStream().collect(toSet());
    assertThat(innerAMinimalComponents, hasItem(innerA));
    assertThat(innerAMinimalComponents, hasItem(innerAComplexParam.build()));
    assertThat(innerAMinimalComponents, hasItem(innerB));

    final Collection<ComponentAst> innerBMinimalComponents =
        graph.minimalArtifactFor(innerB).recursiveStream().collect(toSet());
    assertThat(innerBMinimalComponents, hasItem(innerB));
    assertThat(innerBMinimalComponents, not(hasItem(innerA)));
  }

  @Test
  public void transitivesHandling() {
    final TestComponentAst inner = new TestComponentAst();
    inner.setParameters(singletonMap("name", "inner"));
    final TestComponentAst intermediateLevel = new TestComponentAst(inner);
    intermediateLevel.setParameters(singletonMap("name", "intermediate"));
    final TestComponentAst topLevel = new TestComponentAst(intermediateLevel);
    topLevel.setParameters(singletonMap("name", "top"));

    final ArtifactAstDependencyGraph graph = generateFor(new TestArtifactAst(topLevel));

    final Collection<ComponentAst> topLevelMinimalComponents =
        graph.minimalArtifactFor(topLevel).recursiveStream().collect(toSet());
    assertThat(topLevelMinimalComponents, hasItem(topLevel));
    assertThat(topLevelMinimalComponents, hasItem(intermediateLevel));
    assertThat(topLevelMinimalComponents, hasItem(inner));

    final Collection<ComponentAst> innerLevelMinimalComponents =
        graph.minimalArtifactFor(inner).recursiveStream().collect(toSet());
    assertThat(innerLevelMinimalComponents, not(hasItem(topLevel)));
    assertThat(innerLevelMinimalComponents, not(hasItem(intermediateLevel)));
    assertThat(innerLevelMinimalComponents, hasItem(inner));
  }

  @Test
  public void transitivesHandlingPredicate() {
    final TestComponentAst inner = new TestComponentAst();
    inner.setParameters(singletonMap("name", "inner"));
    final TestComponentAst intermediateLevel = new TestComponentAst(inner);
    intermediateLevel.setParameters(singletonMap("name", "intermediate"));
    final TestComponentAst topLevel = new TestComponentAst(intermediateLevel);
    topLevel.setParameters(singletonMap("name", "top"));

    final ArtifactAstDependencyGraph graph = generateFor(new TestArtifactAst(topLevel));

    final Collection<ComponentAst> topLevelMinimalComponents =
        graph.minimalArtifactFor(topLevel::equals).recursiveStream().collect(toSet());
    assertThat(topLevelMinimalComponents, hasItem(topLevel));
    assertThat(topLevelMinimalComponents, hasItem(intermediateLevel));
    assertThat(topLevelMinimalComponents, hasItem(inner));

    final Collection<ComponentAst> innerLevelMinimalComponents =
        graph.minimalArtifactFor(inner::equals).recursiveStream().collect(toSet());
    assertThat(innerLevelMinimalComponents, not(hasItem(topLevel)));
    assertThat(innerLevelMinimalComponents, not(hasItem(intermediateLevel)));
    assertThat(innerLevelMinimalComponents, hasItem(inner));
  }

  @Test
  public void missingReported() {
    final ParameterizedModel parameterizedModel = mock(ParameterizedModel.class);
    final ParameterModel parameterModel = mock(ParameterModel.class);
    when(parameterModel.getAllowedStereotypes())
        .thenReturn(asList(new ImmutableStereotypeModel("ST_NAME", "ST_NAMESPACE", null)));
    when(parameterModel.getName()).thenReturn("nameParam");

    ParameterGroupModel groupModel = createMockParameterGroup(parameterModel);
    when(parameterizedModel.getParameterGroupModels()).thenReturn(singletonList(groupModel));

    final TestComponentAst innerA = new TestComponentAst(parameterizedModel);
    innerA.setParameters(singletonMap("nameParam", "dependencyId"));

    final ArtifactAstDependencyGraph graph = generateFor(new TestArtifactAst(innerA));

    assertThat(graph.getMissingDependencies(), hasSize(1));

    final ComponentAstDependency missingDep = graph.getMissingDependencies().iterator().next();
    assertThat(missingDep.getName(), is("dependencyId"));
    assertThat(missingDep.getAllowedStereotypes(), hasSize(1));
    assertThat(missingDep.getAllowedStereotypes(), hasItem(new ImmutableStereotypeModel("ST_NAME", "ST_NAMESPACE", null)));
  }

  @Test
  @Issue("MULE-19865")
  public void inlineParamWithStereotypeNotMarkedAsMissingDependency() {
    final ParameterizedModel parameterizedModel = mock(ParameterizedModel.class);
    final ParameterModel parameterModel = mock(ParameterModel.class);
    when(parameterModel.getAllowedStereotypes())
        .thenReturn(asList(new ImmutableStereotypeModel("ST_NAME", "ST_NAMESPACE", null)));
    when(parameterModel.getName()).thenReturn("nameParam");

    ParameterGroupModel groupModel = createMockParameterGroup(parameterModel);
    when(parameterizedModel.getParameterGroupModels()).thenReturn(singletonList(groupModel));

    final TestComponentAst innerA = new TestComponentAst(parameterizedModel);
    innerA.setComplexParameters(singletonMap("nameParam", new TestComponentAst()));

    final ArtifactAstDependencyGraph graph = generateFor(new TestArtifactAst(innerA));

    assertThat(graph.getMissingDependencies(), emptyIterable());
  }

  @Test
  @Issue("MULE-17730")
  @Description("A missing dependency was not reported if another dependency for the same component is present")
  public void missingReportedButAnotherFound() {
    final ParameterizedModel parameterizedModel = mock(ParameterizedModel.class);
    final ParameterModel parameterModel = mock(ParameterModel.class);
    when(parameterModel.getAllowedStereotypes())
        .thenReturn(asList(new ImmutableStereotypeModel("ST_NAME", "ST_NAMESPACE", null)));
    when(parameterModel.getName()).thenReturn("nameParam");

    ParameterGroupModel groupModel = createMockParameterGroup(parameterModel);
    when(parameterizedModel.getParameterGroupModels()).thenReturn(singletonList(groupModel));

    final TestComponentAst innerA = new TestComponentAst(parameterizedModel,
                                                         // a child component is a dependency
                                                         new TestComponentAst(mock(Object.class)));
    innerA.setParameters(singletonMap("nameParam", "dependencyId"));

    final ArtifactAstDependencyGraph graph = generateFor(new TestArtifactAst(innerA));

    assertThat(graph.getMissingDependencies(), hasSize(1));

    final ComponentAstDependency missingDep = graph.getMissingDependencies().iterator().next();
    assertThat(missingDep.getName(), is("dependencyId"));
    assertThat(missingDep.getAllowedStereotypes(), hasSize(1));
    assertThat(missingDep.getAllowedStereotypes(), hasItem(new ImmutableStereotypeModel("ST_NAME", "ST_NAMESPACE", null)));
  }

  @Test
  @Description("Dependencies missing more than once are only reported once")
  public void missingReportedTwice() {
    final ParameterizedModel parameterizedModel = mock(ParameterizedModel.class);
    final ParameterModel parameterModel = mock(ParameterModel.class);
    when(parameterModel.getAllowedStereotypes())
        .thenReturn(asList(new ImmutableStereotypeModel("ST_NAME", "ST_NAMESPACE", null)));
    when(parameterModel.getName()).thenReturn("nameParam");

    ParameterGroupModel groupModel = createMockParameterGroup(parameterModel);
    when(parameterizedModel.getParameterGroupModels()).thenReturn(singletonList(groupModel));

    final TestComponentAst innerA = new TestComponentAst(parameterizedModel);
    innerA.setParameters(singletonMap("nameParam", "dependencyId"));
    final TestComponentAst innerB = new TestComponentAst(parameterizedModel);
    innerB.setParameters(singletonMap("nameParam", "dependencyId"));

    final ArtifactAstDependencyGraph graph = generateFor(new TestArtifactAst(innerA, innerB));

    assertThat(graph.getMissingDependencies(), hasSize(1));

    final ComponentAstDependency missingDep = graph.getMissingDependencies().iterator().next();
    assertThat(missingDep.getName(), is("dependencyId"));
    assertThat(missingDep.getAllowedStereotypes(), hasSize(1));
    assertThat(missingDep.getAllowedStereotypes(), hasItem(new ImmutableStereotypeModel("ST_NAME", "ST_NAMESPACE", null)));
  }

  @Test
  public void missingDifferentStereotypeReported() {
    final ParameterizedModel parameterizedModel = mock(ParameterizedModel.class);
    final ParameterModel parameterModel = mock(ParameterModel.class);
    when(parameterModel.getAllowedStereotypes())
        .thenReturn(asList(new ImmutableStereotypeModel("ST_NAME", "ST_NAMESPACE", null)));
    when(parameterModel.getName()).thenReturn("nameParam");

    ParameterGroupModel groupModel = createMockParameterGroup(parameterModel);
    when(parameterizedModel.getParameterGroupModels()).thenReturn(singletonList(groupModel));

    final TestComponentAst innerA = new TestComponentAst(parameterizedModel);
    innerA.setParameters(singletonMap("nameParam", "dependencyId"));

    final HasStereotypeModel hasStereotypeModel = mock(HasStereotypeModel.class);
    when(hasStereotypeModel.getStereotype()).thenReturn(new ImmutableStereotypeModel("ST_NOT_NAME", "ST_NAMESPACE", null));
    final TestComponentAst innerB = new TestComponentAst(hasStereotypeModel);
    innerB.setParameters(singletonMap("name", "dependencyId"));

    final ArtifactAstDependencyGraph graph = generateFor(new TestArtifactAst(innerA, innerB));

    assertThat(graph.getMissingDependencies(), hasSize(1));

    final ComponentAstDependency missingDep = graph.getMissingDependencies().iterator().next();
    assertThat(missingDep.getName(), is("dependencyId"));
    assertThat(missingDep.getAllowedStereotypes(), hasSize(1));
    assertThat(missingDep.getAllowedStereotypes(), hasItem(new ImmutableStereotypeModel("ST_NAME", "ST_NAMESPACE", null)));
  }

  @Test
  public void missingSameStereotypeReported() {
    final ParameterizedModel parameterizedModel = mock(ParameterizedModel.class);
    final ParameterModel parameterModel = mock(ParameterModel.class);
    when(parameterModel.getAllowedStereotypes())
        .thenReturn(asList(new ImmutableStereotypeModel("ST_NAME", "ST_NAMESPACE", null)));
    when(parameterModel.getName()).thenReturn("nameParam");

    ParameterGroupModel groupModel = createMockParameterGroup(parameterModel);
    when(parameterizedModel.getParameterGroupModels()).thenReturn(singletonList(groupModel));

    final TestComponentAst innerA = new TestComponentAst(parameterizedModel);
    innerA.setParameters(singletonMap("nameParam", "dependencyId"));

    final HasStereotypeModel hasStereotypeModel = mock(HasStereotypeModel.class);
    when(hasStereotypeModel.getStereotype()).thenReturn(new ImmutableStereotypeModel("ST_NAME", "ST_NAMESPACE", null));
    final TestComponentAst innerB = new TestComponentAst(hasStereotypeModel);
    innerB.setParameters(singletonMap("name", "notMatchingDependencyId"));

    final ArtifactAstDependencyGraph graph = generateFor(new TestArtifactAst(innerA, innerB));

    assertThat(graph.getMissingDependencies(), hasSize(1));

    final ComponentAstDependency missingDep = graph.getMissingDependencies().iterator().next();
    assertThat(missingDep.getName(), is("dependencyId"));
    assertThat(missingDep.getAllowedStereotypes(), hasSize(1));
    assertThat(missingDep.getAllowedStereotypes(), hasItem(new ImmutableStereotypeModel("ST_NAME", "ST_NAMESPACE", null)));
  }

  @Test
  public void referencingParameterNotSet() {
    final ParameterizedModel parameterizedModel = mock(ParameterizedModel.class);
    final ParameterModel parameterModel = mock(ParameterModel.class);
    when(parameterModel.getAllowedStereotypes())
        .thenReturn(asList(new ImmutableStereotypeModel("ST_NAME", "ST_NAMESPACE", null)));
    when(parameterModel.getName()).thenReturn("nameParam");

    ParameterGroupModel groupModel = createMockParameterGroup(parameterModel);
    when(parameterizedModel.getParameterGroupModels()).thenReturn(singletonList(groupModel));

    final TestComponentAst innerA = new TestComponentAst(parameterizedModel);
    // No value set for the parameter with the reference
    innerA.setParameters(emptyMap());

    final HasStereotypeModel hasStereotypeModel = mock(HasStereotypeModel.class);
    when(hasStereotypeModel.getStereotype()).thenReturn(new ImmutableStereotypeModel("ST_NOT_NAME", "ST_NAMESPACE", null));
    final TestComponentAst innerB = new TestComponentAst(hasStereotypeModel);
    innerB.setParameters(singletonMap("name", "dependencyId"));

    final ArtifactAstDependencyGraph graph = generateFor(new TestArtifactAst(innerA, innerB));

    assertThat(graph.getMissingDependencies(), is(empty()));
  }

  @Test
  public void referencingParameterNotSetDefaultValue() {
    final ParameterizedModel parameterizedModel = mock(ParameterizedModel.class);
    final ParameterModel parameterModel = mock(ParameterModel.class);
    when(parameterModel.getAllowedStereotypes())
        .thenReturn(asList(new ImmutableStereotypeModel("ST_NAME", "ST_NAMESPACE", null)));
    when(parameterModel.getName()).thenReturn("nameParam");
    when(parameterModel.getDefaultValue()).thenReturn("dependencyId");

    ParameterGroupModel groupModel = createMockParameterGroup(parameterModel);
    when(parameterizedModel.getAllParameterModels()).thenCallRealMethod();
    when(parameterizedModel.getParameterGroupModels()).thenReturn(singletonList(groupModel));

    final TestComponentAst innerA = new TestComponentAst(parameterizedModel);
    // No value set for the parameter with the reference
    innerA.setParameters(emptyMap());

    final HasStereotypeModel hasStereotypeModel = mock(HasStereotypeModel.class);
    when(hasStereotypeModel.getStereotype()).thenReturn(new ImmutableStereotypeModel("ST_NOT_NAME", "ST_NAMESPACE", null));
    final TestComponentAst innerB = new TestComponentAst(hasStereotypeModel);
    innerB.setParameters(singletonMap("name", "dependencyId"));

    final ArtifactAstDependencyGraph graph = generateFor(new TestArtifactAst(innerA, innerB));

    assertThat(graph.getMissingDependencies(), hasSize(1));
  }

  @Test
  public void transitivesHandlingComparator() {
    final TestComponentAst inner = new TestComponentAst();
    inner.setParameters(singletonMap("name", "inner"));
    final TestComponentAst intermediateLevel = new TestComponentAst(inner);
    intermediateLevel.setParameters(singletonMap("name", "intermediate"));
    final TestComponentAst topLevel = new TestComponentAst(intermediateLevel);
    topLevel.setParameters(singletonMap("name", "top"));

    final ArtifactAstDependencyGraph graph = generateFor(new TestArtifactAst(topLevel));

    final List<TestComponentAst> sorted = asList(inner, intermediateLevel, topLevel);
    final List<TestComponentAst> unorderedA = asList(intermediateLevel, inner, topLevel);
    final List<TestComponentAst> unorderedB = asList(intermediateLevel, topLevel, inner);
    final List<TestComponentAst> unorderedC = asList(inner, topLevel, intermediateLevel);
    final List<TestComponentAst> unorderedD = asList(topLevel, inner, intermediateLevel);
    final List<TestComponentAst> unorderedE = asList(topLevel, intermediateLevel, inner);

    sort(unorderedA, graph.dependencyComparator());
    assertThat(unorderedA, is(sorted));

    sort(unorderedB, graph.dependencyComparator());
    assertThat(unorderedB, is(sorted));

    sort(unorderedC, graph.dependencyComparator());
    assertThat(unorderedC, is(sorted));

    sort(unorderedD, graph.dependencyComparator());
    assertThat(unorderedD, is(sorted));

    sort(unorderedE, graph.dependencyComparator());
    assertThat(unorderedE, is(sorted));
  }

  @Test
  public void transitivesHandlingComparatorNotHandlingCorrectlyTopLevelElements() {
    final TestComponentAst top1 = new TestComponentAst();
    top1.setParameters(singletonMap("name", "top1"));
    final TestComponentAst inner = new TestComponentAst();
    inner.setParameters(singletonMap("name", "inner"));
    final TestComponentAst top2 = new TestComponentAst(top1, inner);
    top2.setParameters(singletonMap("name", "top2"));
    final TestComponentAst top3 = new TestComponentAst(top1);
    top3.setParameters(singletonMap("name", "top3"));

    final ArtifactAstDependencyGraph graph = generateFor(new TestArtifactAst(top1, top2, top3));

    final List<TestComponentAst> sorted = asList(top1, inner, top3, top2);
    final List<TestComponentAst> unordered = asList(top1, top3, top2, inner);

    sort(unordered, graph.dependencyComparator());
    assertThat(unordered, is(sorted));
  }

  @Test
  public void comparatorShouldWorkWithCyclicDependenciesInGraph() {
    final ParameterizedModel parameterizedModel =
        mock(ParameterizedModel.class, withSettings().extraInterfaces(HasStereotypeModel.class));
    ImmutableStereotypeModel stereotypeModel = new ImmutableStereotypeModel("ST_NAME", "ST_NAMESPACE", null);
    when(((HasStereotypeModel) parameterizedModel).getStereotype()).thenReturn(stereotypeModel);
    List<StereotypeModel> stereotypeModelList = asList(stereotypeModel);
    final ParameterModel refParameterModel = mock(ParameterModel.class);
    when(refParameterModel.getAllowedStereotypes()).thenReturn(stereotypeModelList);
    when(refParameterModel.getName()).thenReturn("refParam");

    ParameterGroupModel groupModel = createMockParameterGroup(refParameterModel);
    when(parameterizedModel.getParameterGroupModels()).thenReturn(singletonList(groupModel));

    final TestComponentAst innerA = new TestComponentAst(parameterizedModel);
    Map<String, String> innerAParams = new HashMap<>();
    innerAParams.put("name", "dependencyId1");
    innerAParams.put("refParam", "dependencyId2");
    innerA.setParameters(innerAParams);

    final TestComponentAst innerB = new TestComponentAst(parameterizedModel);
    Map<String, String> innerBParams = new HashMap<>();
    innerBParams.put("name", "dependencyId2");
    innerBParams.put("refParam", "dependencyId1");
    innerB.setParameters(innerBParams);

    final ArtifactAstDependencyGraph graph = generateFor(new TestArtifactAst(innerA, innerB));

    final Collection<ComponentAst> innerAMinimalComponents =
        graph.minimalArtifactFor(innerA).recursiveStream().collect(toSet());
    assertThat(innerAMinimalComponents, hasItem(innerA));
    assertThat(innerAMinimalComponents, hasItem(innerB));

    final Collection<ComponentAst> innerBMinimalComponents =
        graph.minimalArtifactFor(innerB).recursiveStream().collect(toSet());
    assertThat(innerBMinimalComponents, hasItem(innerB));
    assertThat(innerBMinimalComponents, hasItem(innerA));

    final List<TestComponentAst> sorted = asList(innerB, innerA);
    final List<TestComponentAst> unordered = asList(innerA, innerB);

    sort(unordered, graph.dependencyComparator());
    assertThat(unordered, is(sorted));
  }

  @Test
  public void unrelatedComponentsComparator() {
    final ComponentAst innerA = new TestComponentAst();
    final ComponentAst innerB = new TestComponentAst();

    final ArtifactAstDependencyGraph graph = generateFor(new TestArtifactAst(innerA, innerB));

    final List<ComponentAst> unorderedA = asList(innerA, innerB);
    final List<ComponentAst> unorderedB = asList(innerB, innerA);

    sort(unorderedA, graph.dependencyComparator());
    assertThat(unorderedA, hasSize(2));

    sort(unorderedB, graph.dependencyComparator());
    assertThat(unorderedB, hasSize(2));
  }

  @Test
  public void sameComponentsComparator() {
    final ComponentAst innerA = new TestComponentAst();
    final ComponentAst innerB = innerA;

    final ArtifactAstDependencyGraph graph = generateFor(new TestArtifactAst(innerA, innerB));

    final List<ComponentAst> unorderedA = asList(innerA, innerB);
    final List<ComponentAst> unorderedB = asList(innerB, innerA);

    sort(unorderedA, graph.dependencyComparator());
    assertThat(unorderedA, hasSize(2));

    sort(unorderedB, graph.dependencyComparator());
    assertThat(unorderedB, hasSize(2));
  }

  @Test
  public void dependencyCycle() {
    final ImmutableStereotypeModel stereotype1 = new ImmutableStereotypeModel("ST1_NAME", "ST_NAMESPACE", null);
    final ImmutableStereotypeModel stereotype2 = new ImmutableStereotypeModel("ST2_NAME", "ST_NAMESPACE", null);

    final ParameterizedModel parameterizedModel =
        mock(ParameterizedModel.class, withSettings().extraInterfaces(HasStereotypeModel.class));
    final ParameterModel parameterModel = mock(ParameterModel.class);
    when(parameterModel.getAllowedStereotypes()).thenReturn(asList(stereotype1));
    when(parameterModel.getName()).thenReturn("nameParam");

    ParameterGroupModel groupModel = createMockParameterGroup(parameterModel);
    when(parameterizedModel.getParameterGroupModels()).thenReturn(singletonList(groupModel));
    when(((HasStereotypeModel) parameterizedModel).getStereotype()).thenReturn(stereotype2);

    final TestComponentAst innerA = new TestComponentAst(parameterizedModel);
    final Map<String, String> rawParamsA = new HashMap<>();
    rawParamsA.put("name", "dependency2Id");
    rawParamsA.put("nameParam", "dependencyId");
    innerA.setParameters(rawParamsA);

    final HasStereotypeModel hasStereotypeModel =
        mock(HasStereotypeModel.class, withSettings().extraInterfaces(ParameterizedModel.class));
    when(hasStereotypeModel.getStereotype()).thenReturn(stereotype1);
    final ParameterModel parameter2Model = mock(ParameterModel.class);
    when(parameter2Model.getAllowedStereotypes()).thenReturn(asList(stereotype2));
    when(parameter2Model.getName()).thenReturn("nameParam");

    final ParameterGroupModel groupModel2 = createMockParameterGroup(parameter2Model);
    when(((ParameterizedModel) hasStereotypeModel).getParameterGroupModels()).thenReturn(singletonList(groupModel2));

    final TestComponentAst innerB = new TestComponentAst(hasStereotypeModel);
    final Map<String, String> rawParamsB = new HashMap<>();
    rawParamsB.put("name", "dependencyId");
    rawParamsB.put("nameParam", "dependency2Id");
    innerB.setParameters(rawParamsB);

    final ArtifactAstDependencyGraph graph = generateFor(new TestArtifactAst(innerA, innerB));

    final Collection<ComponentAst> innerAMinimalComponents =
        graph.minimalArtifactFor(innerA).recursiveStream().collect(toSet());
    assertThat(innerAMinimalComponents, hasItem(innerA));
    assertThat(innerAMinimalComponents, hasItem(innerB));
  }

  @Test
  @Issue("MULE-19193")
  public void recursiveStreamWithHierarchyOrphansAreNotTopLevel() {
    final ParameterizedModel parameterizedModel = mock(ParameterizedModel.class);

    final TestComponentAst inner = new TestComponentAst(parameterizedModel);
    final TestComponentAst outer = new TestComponentAst(parameterizedModel, inner);

    final ArtifactAstDependencyGraph graph = generateFor(new TestArtifactAst(outer));

    final ArtifactAst minimalArtifact = graph.minimalArtifactFor(inner);
    final List<Pair<ComponentAst, List<ComponentAst>>> withHierarchy =
        recursiveStreamWithHierarchy(minimalArtifact).collect(toList());

    assertThat(withHierarchy, hasSize(1));
    assertThat(withHierarchy.get(0).getSecond(), not(empty()));
  }

  private ComponentAstBuilder buildOnce(final ComponentAstBuilder paramComponent) {
    AtomicReference<ComponentAst> paramComp = new AtomicReference<>();

    doAnswer(inv -> paramComp.updateAndGet(alreadyBuilt -> {
      if (alreadyBuilt != null) {
        return alreadyBuilt;
      } else {
        try {
          return (ComponentAst) inv.callRealMethod();
        } catch (Throwable t) {
          throw new RuntimeException(t);
        }
      }
    })).when(paramComponent).build();

    return paramComponent;
  }

  private static ParameterGroupModel createMockParameterGroup(ParameterModel... parameterModels) {
    ParameterGroupModel groupModel = mock(ParameterGroupModel.class);
    when(groupModel.getName()).thenReturn(DEFAULT_GROUP_NAME);
    when(groupModel.getParameterModels()).thenReturn(asList(parameterModels));
    return groupModel;
  }
}
