/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.ast.test.internal;

import static org.mule.runtime.api.functional.Either.left;
import static org.mule.runtime.api.functional.Either.right;

import static java.util.Arrays.asList;
import static java.util.Collections.emptyList;
import static java.util.Collections.emptyMap;
import static java.util.Collections.singleton;
import static java.util.Optional.empty;
import static java.util.Optional.of;
import static java.util.Optional.ofNullable;
import static java.util.stream.Stream.concat;

import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.when;

import org.mule.metadata.api.model.MetadataType;
import org.mule.runtime.api.component.ComponentIdentifier;
import org.mule.runtime.api.component.TypedComponentIdentifier.ComponentType;
import org.mule.runtime.api.component.location.ComponentLocation;
import org.mule.runtime.api.functional.Either;
import org.mule.runtime.api.meta.model.ExtensionModel;
import org.mule.runtime.api.meta.model.parameter.ParameterGroupModel;
import org.mule.runtime.api.meta.model.parameter.ParameterModel;
import org.mule.runtime.api.meta.model.parameter.ParameterizedModel;
import org.mule.runtime.ast.api.ComponentAst;
import org.mule.runtime.ast.api.ComponentGenerationInformation;
import org.mule.runtime.ast.api.ComponentMetadataAst;
import org.mule.runtime.ast.api.ComponentParameterAst;
import org.mule.runtime.ast.api.ParameterResolutionException;
import org.mule.runtime.ast.api.util.AstTraversalDirection;
import org.mule.runtime.ast.api.util.BaseComponentAst;
import org.mule.runtime.ast.internal.DefaultComponentGenerationInformation;

import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.NoSuchElementException;
import java.util.Optional;
import java.util.Spliterator;
import java.util.stream.Stream;


public class TestComponentAst extends BaseComponentAst {

  private final Object model;
  private final List<ComponentAst> innerComponents;
  private Map<String, String> rawParams = emptyMap();
  private Map<String, ComponentAst> complexParams = emptyMap();

  public TestComponentAst(ComponentAst... innerComponents) {
    this.model = null;
    this.innerComponents = asList(innerComponents);
  }

  public TestComponentAst(Object model, ComponentAst... innerComponents) {
    this.model = model;
    this.innerComponents = asList(innerComponents);
  }

  @Override
  public ComponentIdentifier getIdentifier() {
    return null;
  }

  @Override
  public ComponentType getComponentType() {
    return null;
  }

  @Override
  public ComponentLocation getLocation() {
    return null;
  }

  @Override
  public Map<String, Object> getAnnotations() {
    return emptyMap();
  }

  @Override
  public ComponentMetadataAst getMetadata() {
    return null;
  }

  @Override
  public Optional<String> getComponentId() {
    return of(rawParams.get("name"));
  }

  @Override
  public ExtensionModel getExtensionModel() {
    return mock(ExtensionModel.class);
  }

  @Override
  public <M> Optional<M> getModel(Class<M> modelClass) {
    if (model == null || !modelClass.isAssignableFrom(model.getClass())) {
      return empty();
    } else {
      return (Optional<M>) of(model);
    }
  }

  @Override
  public MetadataType getType() {
    return null;
  }

  @Override
  public ComponentGenerationInformation getGenerationInformation() {
    return DefaultComponentGenerationInformation.builder().build();
  }

  @Override
  public ComponentParameterAst getParameter(String groupName, String paramName) {
    final Optional<ParameterModel> paramModel = getModel(ParameterizedModel.class)
        .flatMap(parameterized -> parameterized.getParameterGroupModels().stream()
            .flatMap(g -> g.getParameterModels().stream())
            .filter(pm -> pm.getName().equals(paramName))
            .findAny());

    final ParameterGroupModel groupModel = mock(ParameterGroupModel.class);
    when(groupModel.getName()).thenReturn(groupName);

    if (rawParams.containsKey(paramName) || paramModel.isPresent()) {
      return new ComponentParameterAst() {

        @Override
        public Either<String, Object> getValue() {
          return right(getRawValue() != null ? getRawValue() : getModel().getDefaultValue());
        }

        @Override
        public <T> Either<String, Either<ParameterResolutionException, T>> getValueOrResolutionError() {
          try {
            return getValue().mapRight(fixedValue -> right((T) fixedValue));
          } catch (ParameterResolutionException e) {
            return right(left(e));
          }
        }

        @Override
        public String getResolvedRawValue() {
          return rawParams.get(paramName);
        }

        @Override
        public String getRawValue() {
          return rawParams.get(paramName);
        }

        @Override
        public ParameterModel getModel() {
          return paramModel.orElse(null);
        }

        @Override
        public ParameterGroupModel getGroupModel() {
          return groupModel;
        }

        @Override
        public Optional<ComponentMetadataAst> getMetadata() {
          return empty();
        }

        @Override
        public ComponentGenerationInformation getGenerationInformation() {
          return DefaultComponentGenerationInformation.EMPTY_GENERATION_INFO;
        }

        @Override
        public boolean isDefaultValue() {
          return !ofNullable(getRawValue()).isPresent();
        }
      };
    } else if (complexParams.containsKey(paramName)) {
      return new ComponentParameterAst() {

        @Override
        public Either<String, Object> getValue() {
          return right(complexParams.get(paramName));
        }

        @Override
        public <T> Either<String, Either<ParameterResolutionException, T>> getValueOrResolutionError() {
          try {
            return getValue().mapRight(fixedValue -> right((T) fixedValue));
          } catch (ParameterResolutionException e) {
            return right(left(e));
          }
        }

        @Override
        public String getResolvedRawValue() {
          return null;
        }

        @Override
        public String getRawValue() {
          return null;
        }

        @Override
        public ParameterModel getModel() {
          return paramModel.orElse(null);
        }

        @Override
        public ParameterGroupModel getGroupModel() {
          return groupModel;
        }

        @Override
        public Optional<ComponentMetadataAst> getMetadata() {
          return empty();
        }

        @Override
        public ComponentGenerationInformation getGenerationInformation() {
          return DefaultComponentGenerationInformation.EMPTY_GENERATION_INFO;
        }

        @Override
        public boolean isDefaultValue() {
          return false;
        }
      };
    } else {
      throw new NoSuchElementException(paramName);
    }
  }

  @Override
  public Collection<ComponentParameterAst> getParameters() {
    return emptyList();
  }

  public void setParameters(Map<String, String> rawParams) {
    this.rawParams = rawParams;
  }

  public void setComplexParameters(Map<String, ComponentAst> complexParams) {
    this.complexParams = complexParams;
  }

  @Override
  public Stream<ComponentAst> recursiveStream(AstTraversalDirection direction) {
    return concat(singleton(this).stream(),
                  innerComponents.stream()
                      .flatMap(cm -> cm.recursiveStream(direction)));
  }

  @Override
  public Spliterator<ComponentAst> recursiveSpliterator(AstTraversalDirection direction) {
    return recursiveStream(direction).spliterator();
  }

  @Override
  public List<ComponentAst> directChildren() {
    return innerComponents;
  }

  @Override
  public String toString() {
    return getComponentId().orElse(super.toString());
  }

}
