/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.ast.internal.serialization;

import static java.nio.charset.StandardCharsets.UTF_8;
import static org.apache.commons.io.IOUtils.toInputStream;
import static org.junit.rules.ExpectedException.none;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.times;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;
import static org.mule.runtime.ast.AllureConstants.ArtifactAstSerialization.AST_SERIALIZATION;

import io.qameta.allure.Description;
import io.qameta.allure.Feature;
import java.io.IOException;
import java.io.InputStream;
import java.util.HashSet;
import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.ExpectedException;
import org.mule.runtime.api.meta.model.ExtensionModel;
import org.mule.runtime.ast.api.serialization.ArtifactAstDeserializer;
import org.mule.runtime.ast.api.serialization.ExtensionModelResolver;

@Feature(AST_SERIALIZATION)
public class DefaultArtifactAstDeserializerTestCase {

  @Rule
  public ExpectedException exception = none();

  private DefaultArtifactAstDeserializer defaultArtifactAstDeserializer;
  private ArtifactAstSerializerMetadataSerializer artifactAstSerializerMetadataSerializer;
  private ArtifactAstSerializerFactory artifactAstSerializerFactory;
  private ArtifactAstDeserializer artifactAstDeserializer;
  private ExtensionModelResolver extensionModelResolver;

  @Before
  public void setUp() {
    artifactAstSerializerFactory = mock(ArtifactAstSerializerFactory.class);
    artifactAstSerializerMetadataSerializer = mock(ArtifactAstSerializerMetadataSerializer.class);
    defaultArtifactAstDeserializer =
        new DefaultArtifactAstDeserializer(artifactAstSerializerFactory, artifactAstSerializerMetadataSerializer);
    artifactAstDeserializer = mock(ArtifactAstDeserializer.class);
    extensionModelResolver = new TestExtensionModelResolver();
  }

  @Test
  @Description("The metadata is read")
  public void testDeserializeCallsMetadataReaderWithParameterArtifactInputStream_WhenDeserializingAnArtifactInputStream()
      throws IOException {
    // Given
    InputStream inputStream = toInputStream("Some artifact ast but serialized", UTF_8);
    ArtifactAstSerializerMetadata artifactAstSerializerMetadata = new ArtifactAstSerializerMetadata("SomeId", "SomeVersion");
    when(artifactAstSerializerMetadataSerializer.readArtifactAstSerializerMetadataFromInputStream(inputStream))
        .thenReturn(artifactAstSerializerMetadata);
    when(artifactAstSerializerFactory.getDeserializer(artifactAstSerializerMetadata.getSerializerId()))
        .thenReturn(artifactAstDeserializer);

    // When
    defaultArtifactAstDeserializer.deserialize(inputStream, extensionModelResolver);

    // Then
    verify(artifactAstSerializerMetadataSerializer, times(1)).readArtifactAstSerializerMetadataFromInputStream(inputStream);
  }

  @Test
  @Description("With the metadata we fetch the deserializer")
  public void testDeserializeCallsSerializerFactoryGetDeserializerWithSerializedMetadataId_WhenDeserializingAnArtifactInputStream()
      throws IOException {
    // Given
    InputStream inputStream = toInputStream("Some artifact ast but serialized", UTF_8);
    ArtifactAstSerializerMetadata artifactAstSerializerMetadata =
        new ArtifactAstSerializerMetadata("SomeOtherId", "SomeOtherVersion");
    when(artifactAstSerializerMetadataSerializer.readArtifactAstSerializerMetadataFromInputStream(inputStream))
        .thenReturn(artifactAstSerializerMetadata);
    when(artifactAstSerializerFactory.getDeserializer(artifactAstSerializerMetadata.getSerializerId()))
        .thenReturn(artifactAstDeserializer);

    // When
    defaultArtifactAstDeserializer.deserialize(inputStream, extensionModelResolver);

    // Then
    verify(artifactAstSerializerFactory, times(1)).getDeserializer(artifactAstSerializerMetadata.getSerializerId());
  }

  @Test
  @Description("With the deserializer we deserialize the input")
  public void testDeserializeCallsResolvedArtifactAstDeserializerDeserializeWithInputStreamAndExtensionModels_WhenDeserializingAnArtifactInputStream()
      throws IOException {
    // Given
    InputStream inputStream = toInputStream("Some artifact ast but serialized", UTF_8);
    ArtifactAstSerializerMetadata artifactAstSerializerMetadata =
        new ArtifactAstSerializerMetadata("YetAnotherId", "YetAnotherVersion");
    when(artifactAstSerializerMetadataSerializer.readArtifactAstSerializerMetadataFromInputStream(inputStream))
        .thenReturn(artifactAstSerializerMetadata);
    when(artifactAstSerializerFactory.getDeserializer(artifactAstSerializerMetadata.getSerializerId()))
        .thenReturn(artifactAstDeserializer);

    // When
    defaultArtifactAstDeserializer.deserialize(inputStream, extensionModelResolver);

    // Then
    verify(artifactAstDeserializer, times(1)).deserialize(inputStream, extensionModelResolver);
  }

  @Test
  public void testDeserializeThrowsNullPointerException_WhenInputStreamIsNull() throws IOException {
    // Given
    InputStream inputStream = null;

    // Then
    exception.expect(NullPointerException.class);
    exception.expectMessage("artifactAstInputStream");

    // When
    defaultArtifactAstDeserializer.deserialize(inputStream, extensionModelResolver);
  }

  @Test
  public void testDeserializeThrowsNullPointerException_WhenExtensionModelSetIsNull() throws IOException {
    // Given
    InputStream inputStream = toInputStream("Some artifact ast but serialized", UTF_8);

    // Then
    exception.expect(NullPointerException.class);
    exception.expectMessage("extensionModelResolver");

    // When
    defaultArtifactAstDeserializer.deserialize(inputStream, null);
  }

  private static class TestExtensionModelResolver implements ExtensionModelResolver {

    @Override
    public ExtensionModel resolve(String name) {
      return null;
    }
  }
}
