/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.ast.internal.serialization;

import static java.lang.String.format;
import static java.util.Arrays.asList;
import static java.util.Objects.requireNonNull;
import static org.apache.commons.lang3.StringUtils.join;

import java.util.List;
import org.mule.runtime.ast.api.serialization.ArtifactAstDeserializer;
import org.mule.runtime.ast.api.serialization.ArtifactAstSerializer;

/**
 * The serializer factory can create a {@link ArtifactAstSerializer} or a {@link ArtifactAstDeserializer} instance in the
 * available ids/formats
 */
public class ArtifactAstSerializerFactory {

  public static final String JSON = "JSON";

  /**
   * This method creates a new instance of an {@link ArtifactAstSerializer} matching the format parameter
   *
   * @param format the format in which the requested serializer serializes the {@link org.mule.runtime.ast.api.ArtifactAst}
   * @return An {@link ArtifactAstSerializer} matching the format
   * @throws IllegalArgumentException if no {@link ArtifactAstSerializer} matching the format is found
   */
  public ArtifactAstSerializer getSerializer(String format) throws IllegalArgumentException {
    requireNonNull(format, "format");

    switch (format) {
      default:
        throw new IllegalArgumentException(format("No serializer named '%s' is available. Available serializers are: ["
            + join(getAvailableSerializers(), ", ") + "]", format));
    }
  }

  /**
   * This method creates a new instance o an {@link ArtifactAstDeserializer} matching the format parameter
   *
   * @param format the format which the requested deserializer should be able to deserialize in order to generate the
   *               {@link org.mule.runtime.ast.api.ArtifactAst}
   * @return An {@link ArtifactAstDeserializer} matching the format
   * @throws IllegalArgumentException if no {@link ArtifactAstDeserializer} matching the format is found
   */
  public ArtifactAstDeserializer getDeserializer(String format) throws IllegalArgumentException {
    requireNonNull(format, "format");

    switch (format) {
      default:
        throw new IllegalArgumentException(format("No deserializer named '%s' is available. Available deserializers are: ["
            + join(getAvailableSerializers(), ", ") + "]", format));
    }
  }

  /**
   * All available serializer formats. An available serializer format should be resolved by the getSerializer(format) and
   * getDeserializer(format) methods
   * 
   * @return a list of serializer formats
   */
  final List<String> getAvailableSerializers() {
    return asList();
  }
}
