/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.ast.internal.serialization;

import static java.util.Objects.requireNonNull;

import java.io.IOException;
import java.io.InputStream;
import org.mule.runtime.api.meta.model.ExtensionModel;
import org.mule.runtime.ast.api.ArtifactAst;
import org.mule.runtime.ast.api.serialization.ArtifactAstDeserializer;
import org.mule.runtime.ast.api.serialization.ArtifactAstSerializer;
import org.mule.runtime.ast.api.serialization.ExtensionModelResolver;

/**
 * Default implementation of an ArtifactAstDeserializer. This deserializer reads the serialization metadata from an
 * {@link InputStream} that was generated by an {@link org.mule.runtime.ast.api.serialization.ArtifactAstSerializer} and with that
 * it delegates the deserialization to the correct implementation of an {@link ArtifactAstDeserializer}
 */
public final class DefaultArtifactAstDeserializer implements ArtifactAstDeserializer {

  private final ArtifactAstSerializerFactory artifactAstSerializerFactory;
  private final ArtifactAstSerializerMetadataSerializer artifactAstSerializerMetadataSerializer;

  public DefaultArtifactAstDeserializer() {
    this(new ArtifactAstSerializerFactory(), new ArtifactAstSerializerMetadataSerializer());
  }

  DefaultArtifactAstDeserializer(ArtifactAstSerializerFactory artifactAstSerializerFactory,
                                 ArtifactAstSerializerMetadataSerializer artifactAstSerializerMetadataSerializer) {
    this.artifactAstSerializerFactory = artifactAstSerializerFactory;
    this.artifactAstSerializerMetadataSerializer = artifactAstSerializerMetadataSerializer;
  }

  /**
   * Given an {@link InputStream} created by serializing an {@link ArtifactAst} with an {@link ArtifactAstSerializer} and the
   * associated {@link ExtensionModel}s, this method deserializes it.
   * <p>
   * This method reads the metadata to determine the format in order to apply the corresponding implementation of
   * {@link ArtifactAstDeserializer}.
   * 
   * @param artifactAstInputStream the input stream generated by an
   *                               {@link org.mule.runtime.ast.api.serialization.ArtifactAstSerializer}
   * @param extensionModelResolver an {@link org.mule.runtime.ast.api.serialization.ExtensionModelResolver} instance
   * @return The artifact
   * @throws IllegalArgumentException if this deserializer cannot deserialize this input stream
   * @throws IOException              if the input stream could not be read
   */
  @Override
  public ArtifactAst deserialize(InputStream artifactAstInputStream, ExtensionModelResolver extensionModelResolver)
      throws IllegalArgumentException, IOException {
    requireNonNull(artifactAstInputStream, "artifactAstInputStream");
    requireNonNull(extensionModelResolver, "extensionModelResolver");

    ArtifactAstSerializerMetadata artifactAstSerializerMetadata =
        artifactAstSerializerMetadataSerializer.readArtifactAstSerializerMetadataFromInputStream(artifactAstInputStream);

    ArtifactAstDeserializer artifactAstDeserializer =
        artifactAstSerializerFactory.getDeserializer(artifactAstSerializerMetadata.getSerializerId());

    return artifactAstDeserializer.deserialize(artifactAstInputStream, extensionModelResolver);

  }

}
