/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.ast.internal.serialization;

import org.mule.runtime.ast.api.ArtifactAst;
import org.mule.runtime.ast.api.serialization.ArtifactAstDeserializer;
import org.mule.runtime.ast.api.serialization.ExtensionModelResolver;
import java.io.IOException;
import java.io.InputStream;
import java.nio.charset.Charset;

import static java.nio.charset.StandardCharsets.UTF_8;

/**
 * Defines a component capable of serializing {@link ArtifactAst}s into/from an {@link InputStream}.
 * <p>
 * A serialized {@link ArtifactAst} will contain the information of the original artifact and some more metadata relevant to the
 * serializer. This metadata will be entirely and exclusively in the first line of the resulting {@link InputStream} and will be
 * composed by an ID corresponding to the serializer used followed by its version and the {@link Charset} name used. This values
 * will be delimited by a pound (#) character.
 * <p>
 * Eg: <code>SERIALIZERID#1.2.3#UTF-8#</code>
 * <p>
 * Once the first line has been read, the remaining payload of the {@link InputStream} (which we will call "the body") will be the
 * serialized artifact itself in the format implemented by the serializer that was used.
 * <p>
 * It should be noted when implementing a deserialize method that when deserializing, the {@link InputStream} should only contain
 * the body without the metadata. This is because for an implementation of the deserialize method to be called, the caller needs
 * to know which one matches the {@link InputStream} and that requires for the metadata line to be consumed.
 */
public interface InternalArtifactAstDeserializer extends ArtifactAstDeserializer {

  /**
   * Given an InputStream created by serializing an ArtifactAst with an ArtifactAstSerializer and the associated extension models,
   * this method deserializes it leveraging the ArtifactAstDeserializerProviders available.
   *
   * @param artifactAstInputStream the input stream generated by an ArtifactAstSerializer
   * @param extensionModelResolver an instance of {@link ExtensionModelResolver}
   * @return The artifact
   * @throws IllegalArgumentException if no provider matched the serialization of the artifact
   * @throws IOException              if the input stream could not be read
   */
  default ArtifactAst deserialize(InputStream artifactAstInputStream, ExtensionModelResolver extensionModelResolver)
      throws IllegalArgumentException, IOException {
    return this.deserialize(artifactAstInputStream, this.getLatestVersion(), UTF_8, extensionModelResolver, null);
  }

  /**
   * Given an InputStream created by serializing an ArtifactAst with an ArtifactAstSerializer and the associated extension models,
   * this method deserializes it leveraging the ArtifactAstDeserializerProviders available.
   *
   * @param artifactAstInputStream the input stream generated by an ArtifactAstSerializer
   * @param extensionModelResolver an instance of {@link ExtensionModelResolver}
   * @param parent                 the parent {@link ArtifactAst} to be set to the resulting deserialized {@link ArtifactAst}
   * @return The artifact
   * @throws IllegalArgumentException if no provider matched the serialization of the artifact
   * @throws IOException              if the input stream could not be read
   */
  default ArtifactAst deserialize(InputStream artifactAstInputStream, ExtensionModelResolver extensionModelResolver,
                                  ArtifactAst parent)
      throws IllegalArgumentException, IOException {
    return this.deserialize(artifactAstInputStream, this.getLatestVersion(), UTF_8, extensionModelResolver, parent);
  }

  String getLatestVersion();

  ArtifactAst deserialize(InputStream artifactAstInputStream, String version, Charset charset,
                          ExtensionModelResolver extensionModelResolver, ArtifactAst parent)
      throws IllegalArgumentException, IOException;
}
