/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.ast.internal.serialization.dto;

import static java.util.Optional.ofNullable;
import static java.util.function.UnaryOperator.identity;

import org.mule.runtime.ast.api.ComponentMetadataAst;
import org.mule.runtime.ast.api.ImportedResource;
import org.mule.runtime.ast.internal.builder.PropertiesResolver;

import java.util.Optional;
import java.util.function.UnaryOperator;

/**
 * This is a serializable version of an {@link ImportedResource}.
 */
public class ImportedResourceDTO implements ImportedResource {

  private final String resourceLocation;
  private final ComponentMetadataAstDTO metadata;
  private final String resolutionFailure;
  private transient PropertiesResolver propertiesResolver;

  public ImportedResourceDTO(String rawResourceLocation, ComponentMetadataAstDTO metadata, String resolutionFailure) {
    this.resourceLocation = rawResourceLocation;
    this.metadata = metadata;
    this.resolutionFailure = resolutionFailure;
  }

  @Override
  public String getRawResourceLocation() {
    return resourceLocation;
  }

  @Override
  public String getResourceLocation() {
    return propertiesResolver.apply(resourceLocation);
  }

  @Override
  public ComponentMetadataAst getMetadata() {
    return metadata;
  }

  @Override
  public Optional<String> getResolutionFailure() {
    return ofNullable(resolutionFailure);
  }

  @Override
  public void updatePropertiesResolver(UnaryOperator<String> propertiesResolver) {
    this.propertiesResolver.setMappingFunction(propertiesResolver);
  }

  public void setPropertiesResolver(PropertiesResolver propertiesResolver) {
    this.propertiesResolver = propertiesResolver;
  }

  public void enrich() {
    PropertiesResolver propertiesResolver = new PropertiesResolver();
    propertiesResolver.setMappingFunction(identity());
    this.setPropertiesResolver(propertiesResolver);
  }
}
