/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.ast.internal.serialization.json;

import org.mule.runtime.api.component.ComponentIdentifier;
import org.mule.runtime.api.component.location.ComponentLocation;
import org.mule.runtime.api.message.ErrorType;
import org.mule.runtime.ast.api.ArtifactAst;
import org.mule.runtime.ast.api.serialization.ExtensionModelResolver;
import org.mule.runtime.ast.internal.serialization.InternalArtifactAstDeserializer;
import org.mule.runtime.ast.internal.serialization.dto.ArtifactAstDTO;
import org.mule.runtime.ast.internal.serialization.dto.ParserAttributesDTO;
import org.mule.runtime.ast.internal.serialization.json.gson.ComponentIdentifierJsonDeserializer;
import org.mule.runtime.ast.internal.serialization.json.gson.ComponentLocationJsonDeserializer;
import org.mule.runtime.ast.internal.serialization.json.gson.ErrorTypeJsonDeserializer;
import org.mule.runtime.ast.internal.serialization.json.gson.ParserAttributesJsonTypeAdapter;
import org.mule.runtime.ast.internal.serialization.json.gson.PostProcessingEnabler;
import org.mule.runtime.ast.internal.serialization.resolver.DefaultGenerationInformationResolver;
import org.mule.runtime.ast.internal.serialization.resolver.GenerationInformationResolver;
import org.mule.runtime.ast.internal.serialization.resolver.NoOpGenerationInformationResolver;

import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.nio.charset.Charset;
import java.util.HashMap;
import java.util.Map;
import org.apache.commons.io.input.BoundedInputStream;

/**
 * Json implementation of an {@link InternalArtifactAstDeserializer}. This deserializer assumes the {@link InputStream} has no
 * header as it should have been consumed by the
 * {@link org.mule.runtime.ast.internal.serialization.DefaultArtifactAstDeserializer} in order to decide to dispatch it here.
 */
public class JsonArtifactAstDeserializer implements InternalArtifactAstDeserializer {

  // Max serialized application size. 10mb = 1024 * 1024 * 10 = 10485760
  private static final long MAX_BYTE_COUNT = 10485760;
  public static final String V_1_0 = "1.0";

  private final Map<String, Gson> versionToGson;

  private final GenerationInformationResolver generationInformationResolver;

  /**
   * This constructor allows a {@link GsonBuilder} parameter.
   * 
   * @param gsonBuilder
   */
  public JsonArtifactAstDeserializer(GsonBuilder gsonBuilder, boolean populateGenerationInformation) {
    versionToGson = new HashMap<>();

    gsonBuilder.registerTypeAdapter(ComponentIdentifier.class, new ComponentIdentifierJsonDeserializer());
    gsonBuilder.registerTypeAdapter(ComponentLocation.class, new ComponentLocationJsonDeserializer());
    gsonBuilder.registerTypeAdapter(ErrorType.class, new ErrorTypeJsonDeserializer());
    gsonBuilder.registerTypeAdapter(ParserAttributesDTO.class, new ParserAttributesJsonTypeAdapter());
    gsonBuilder.registerTypeAdapterFactory(new PostProcessingEnabler());
    versionToGson.put(V_1_0, gsonBuilder.create());

    if (populateGenerationInformation) {
      this.generationInformationResolver = new DefaultGenerationInformationResolver();
    } else {
      this.generationInformationResolver = new NoOpGenerationInformationResolver();
    }

  }

  @Override
  public String getLatestVersion() {
    return V_1_0;
  }

  /**
   * This method takes an input stream of a serialized {@link ArtifactAst} with no metadata header and deserializes and enriches
   * it.
   * 
   * @param artifactAstInputStream the input stream generated by an ArtifactAstSerializer. Max allowed size =
   *                               {@value MAX_BYTE_COUNT} bytes
   * @param version
   * @param extensionModelResolver an instance of {@link ExtensionModelResolver}
   * @return The deserialized {@link ArtifactAst}
   * @throws IllegalArgumentException
   */
  @Override
  public ArtifactAst deserialize(InputStream artifactAstInputStream, String version, Charset charset,
                                 ExtensionModelResolver extensionModelResolver, ArtifactAst parent)
      throws IllegalArgumentException, IOException {
    BoundedInputStream boundedInputStream = new BoundedInputStream(artifactAstInputStream, MAX_BYTE_COUNT);
    ArtifactAstDTO artifactAstDTO =
        versionToGson.get(version).fromJson(new BufferedReader(new InputStreamReader(boundedInputStream, charset)),
                                            ArtifactAstDTO.class);
    artifactAstDTO.setParent(parent);
    artifactAstDTO.enrich(extensionModelResolver, generationInformationResolver);
    return artifactAstDTO;
  }

}
