/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.ast.internal.serialization;

import static java.lang.String.format;
import static java.util.Collections.singletonList;
import static java.util.Objects.requireNonNull;
import static java.util.concurrent.Executors.newSingleThreadExecutor;

import static org.apache.commons.lang3.StringUtils.join;

import org.mule.runtime.api.util.LazyValue;
import org.mule.runtime.ast.api.serialization.ArtifactAstDeserializer;
import org.mule.runtime.ast.api.serialization.ArtifactAstSerializer;
import org.mule.runtime.ast.internal.serialization.json.JsonArtifactAstDeserializer;
import org.mule.runtime.ast.internal.serialization.json.JsonArtifactAstSerializer;

import java.util.List;
import java.util.concurrent.Executor;
import java.util.function.Supplier;

import com.google.gson.GsonBuilder;

/**
 * The serializer factory can create a {@link ArtifactAstSerializer} or a {@link ArtifactAstDeserializer} instance in the
 * available ids/formats
 */
public class ArtifactAstSerializerFactory {

  public static final String JSON = "JSON";
  private final Executor executor;
  private final Supplier<ArtifactAstSerializer> jsonArtifactAstSerializer_1_0_LazyValue;
  private final Supplier<InternalArtifactAstDeserializer> jsonArtifactAstDeserializer_1_0_LazyValue;

  public ArtifactAstSerializerFactory() {
    this(newSingleThreadExecutor(), true);
  }

  public ArtifactAstSerializerFactory(Executor executor, boolean populateGenerationInformation) {
    requireNonNull(executor);
    this.executor = executor;

    jsonArtifactAstSerializer_1_0_LazyValue =
        new LazyValue<>(() -> new JsonArtifactAstSerializer(this.executor));
    jsonArtifactAstDeserializer_1_0_LazyValue =
        new LazyValue<>(() -> new JsonArtifactAstDeserializer(new GsonBuilder(), populateGenerationInformation));
  }

  /**
   * This method creates a new instance of an {@link ArtifactAstSerializer} matching the format parameter
   *
   * @param format            the format in which the requested serializer serializes the
   *                          {@link org.mule.runtime.ast.api.ArtifactAst}
   * @param serializerVersion the desired serializer version
   * @return An {@link ArtifactAstSerializer} matching the format
   * @throws IllegalArgumentException if no {@link ArtifactAstSerializer} matching the format is found
   */
  public ArtifactAstSerializer getSerializer(String format, String serializerVersion)
      throws IllegalArgumentException {
    requireNonNull(format, "format");
    requireNonNull(serializerVersion, "serializerVersion");

    if (JSON.equals(format) && "1.0".equals(serializerVersion)) {
      return jsonArtifactAstSerializer_1_0_LazyValue.get();
    }

    throw new IllegalArgumentException(format("No serializer named '%s' is available. Available serializers are: ["
        + join(getAvailableSerializers(), ", ") + "]", format));
  }

  /**
   * This method creates a new instance o an {@link ArtifactAstDeserializer} matching the format and serializerVersion parameters
   *
   * @param format            the format which the requested deserializer should be able to deserialize in order to generate the
   *                          {@link org.mule.runtime.ast.api.ArtifactAst}
   * @param serializerVersion the desired version of the deserializer
   * @return An {@link ArtifactAstDeserializer} matching the format
   * @throws IllegalArgumentException if no {@link ArtifactAstDeserializer} matching the format is found
   */
  public InternalArtifactAstDeserializer getDeserializer(String format, String serializerVersion)
      throws IllegalArgumentException {
    requireNonNull(format, "format");
    requireNonNull(serializerVersion, "serializerVersion");

    if (JSON.equals(format) && "1.0".equals(serializerVersion)) {
      return jsonArtifactAstDeserializer_1_0_LazyValue.get();
    }

    throw new IllegalArgumentException(format("No deserializer named '%s' version '%s' is available. Available deserializers are: ["
        + join(getAvailableSerializers(), ", ") + "]", format, serializerVersion));
  }

  /**
   * All available serializer formats. An available serializer format should be resolved by the getSerializer(format) and
   * getDeserializer(format) methods
   * 
   * @return a list of serializer formats
   */
  final List<String> getAvailableSerializers() {
    return singletonList(JSON + "[1.0]");
  }
}
