/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.ast;

import static org.mule.runtime.ast.AllureConstants.ArtifactAstSerialization.AST_DTO;
import static org.mule.runtime.ast.AllureConstants.ArtifactAstSerialization.AST_SERIALIZATION;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.containsString;
import static org.hamcrest.Matchers.instanceOf;
import static org.hamcrest.Matchers.is;

import org.mule.runtime.ast.internal.serialization.dto.ParameterValueContainer;
import java.util.ArrayList;
import com.google.gson.Gson;
import io.qameta.allure.Feature;
import io.qameta.allure.Story;
import org.junit.Before;
import org.junit.Test;

@Feature(AST_SERIALIZATION)
@Story(AST_DTO)
public class ParameterValueContainerGsonSerializationTestCase {

  private Gson gson;

  @Before
  public void setUp() throws Exception {
    gson = new Gson();
  }

  @Test
  public void testSerializedParameterValueContainerWithCharIsDeserializedAsChar_WhenSerializingAndDeserializingParameterValueContainer() {
    ParameterValueContainer parameterValueContainer =
        new ParameterValueContainer(null, 'A');
    String json = gson.toJson(parameterValueContainer);

    ParameterValueContainer deserialized = gson.fromJson(json, ParameterValueContainer.class);

    assertThat(deserialized.getContainedValue(), instanceOf(Character.class));
    assertThat(deserialized.getContainedValue(), is('A'));
    assertThat(deserialized.toString(), containsString("character"));
  }

  @Test
  public void testSerializedParameterValueContainerWithBooleanIsDeserializedAsBoolean_WhenSerializingAndDeserializingParameterValueContainer() {
    ParameterValueContainer parameterValueContainer =
        new ParameterValueContainer(null, Boolean.TRUE);
    String json = gson.toJson(parameterValueContainer);

    ParameterValueContainer deserialized = gson.fromJson(json, ParameterValueContainer.class);

    assertThat(deserialized.getContainedValue(), instanceOf(Boolean.class));
    assertThat(deserialized.getContainedValue(), is(Boolean.TRUE));
    assertThat(deserialized.toString(), containsString("aBoolean"));
  }

  @Test
  public void testSerializedParameterValueContainerWithStringIsDeserializedAsString_WhenSerializingAndDeserializingParameterValueContainer() {
    ParameterValueContainer parameterValueContainer =
        new ParameterValueContainer(null, "Some string");
    String json = gson.toJson(parameterValueContainer);

    ParameterValueContainer deserialized = gson.fromJson(json, ParameterValueContainer.class);

    assertThat(deserialized.getContainedValue(), instanceOf(String.class));
    assertThat(deserialized.getContainedValue(), is("Some string"));
    assertThat(deserialized.toString(), containsString("aString"));
  }

  @Test
  public void testSerializedParameterValueContainerWithArrayListIsDeserializedAsArrayList_WhenSerializingAndDeserializingParameterValueContainer() {
    ParameterValueContainer parameterValueContainer =
        new ParameterValueContainer(null, new ArrayList<>());
    String json = gson.toJson(parameterValueContainer);

    ParameterValueContainer deserialized = gson.fromJson(json, ParameterValueContainer.class);

    assertThat(deserialized.getContainedValue(), instanceOf(ArrayList.class));
    assertThat(deserialized.getContainedValue(), is(new ArrayList<>()));
    assertThat(deserialized.toString(), containsString("object"));
  }

  @Test
  public void testSerializedParameterValueContainerWithFloatIsDeserializedAsFloat_WhenSerializingAndDeserializingParameterValueContainer() {
    ParameterValueContainer parameterValueContainer =
        new ParameterValueContainer(null, (float) 13.9);
    String json = gson.toJson(parameterValueContainer);

    ParameterValueContainer deserialized = gson.fromJson(json, ParameterValueContainer.class);

    assertThat(deserialized.getContainedValue(), instanceOf(Float.class));
    assertThat(deserialized.getContainedValue(), is(13.9F));
    assertThat(deserialized.toString(), containsString("aFloat"));
  }

  @Test
  public void testSerializedParameterValueContainerWithIntegerIsDeserializedAsInteger_WhenSerializingAndDeserializingParameterValueContainer() {
    ParameterValueContainer parameterValueContainer =
        new ParameterValueContainer(null, 432);
    String json = gson.toJson(parameterValueContainer);

    ParameterValueContainer deserialized = gson.fromJson(json, ParameterValueContainer.class);

    assertThat(deserialized.getContainedValue(), instanceOf(Integer.class));
    assertThat(deserialized.getContainedValue(), is(432));
    assertThat(deserialized.toString(), containsString("anInteger"));
  }

  @Test
  public void testSerializedParameterValueContainerWithLongIsDeserializedAsLong_WhenSerializingAndDeserializingParameterValueContainer() {
    ParameterValueContainer parameterValueContainer =
        new ParameterValueContainer(null, 78L);
    String json = gson.toJson(parameterValueContainer);

    ParameterValueContainer deserialized = gson.fromJson(json, ParameterValueContainer.class);

    assertThat(deserialized.getContainedValue(), instanceOf(Long.class));
    assertThat(deserialized.getContainedValue(), is(78L));
    assertThat(deserialized.toString(), containsString("aLong"));
  }

  @Test
  public void testSerializedParameterValueContainerWithDoubleIsDeserializedAsDouble_WhenSerializingAndDeserializingParameterValueContainer() {
    ParameterValueContainer parameterValueContainer =
        new ParameterValueContainer(null, 74.635D);
    String json = gson.toJson(parameterValueContainer);

    ParameterValueContainer deserialized = gson.fromJson(json, ParameterValueContainer.class);

    assertThat(deserialized.getContainedValue(), instanceOf(Double.class));
    assertThat(deserialized.getContainedValue(), is(74.635D));
    assertThat(deserialized.toString(), containsString("aDouble"));
  }

  @Test
  public void testSerializedParameterValueContainerWithByteIsDeserializedAsByte_WhenSerializingAndDeserializingParameterValueContainer() {
    ParameterValueContainer parameterValueContainer =
        new ParameterValueContainer(null, Byte.MAX_VALUE);
    String json = gson.toJson(parameterValueContainer);

    ParameterValueContainer deserialized = gson.fromJson(json, ParameterValueContainer.class);

    assertThat(deserialized.getContainedValue(), instanceOf(Byte.class));
    assertThat(deserialized.getContainedValue(), is(Byte.MAX_VALUE));
    assertThat(deserialized.toString(), containsString("aByte"));
  }

  @Test
  public void testSerializedParameterValueContainerWithShortIsDeserializedAsShort_WhenSerializingAndDeserializingParameterValueContainer() {
    ParameterValueContainer parameterValueContainer =
        new ParameterValueContainer(null, Short.MIN_VALUE);
    String json = gson.toJson(parameterValueContainer);

    ParameterValueContainer deserialized = gson.fromJson(json, ParameterValueContainer.class);

    assertThat(deserialized.getContainedValue(), instanceOf(Short.class));
    assertThat(deserialized.getContainedValue(), is(Short.MIN_VALUE));
    assertThat(deserialized.toString(), containsString("aShort"));
  }
}
