/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.ast.internal.serialization.dto;

import static org.mule.runtime.api.component.TypedComponentIdentifier.ComponentType.UNKNOWN;
import static org.mule.runtime.ast.AllureConstants.ArtifactAstSerialization.AST_SERIALIZATION;
import static org.mule.runtime.ast.AllureConstants.ArtifactAstSerialization.AST_SERIALIZATION_ENRICH;
import static org.mule.runtime.ast.internal.serialization.dto.ComponentAstDTOModelType.CONFIGURATION;
import static org.mule.runtime.ast.internal.serialization.dto.ComponentAstDTOModelType.CONNECTION_PROVIDER;
import static org.mule.runtime.ast.internal.serialization.dto.ComponentAstDTOModelType.CONSTRUCT;
import static org.mule.runtime.ast.internal.serialization.dto.ComponentAstDTOModelType.NESTED_COMPONENT;
import static org.mule.runtime.ast.internal.serialization.dto.ComponentAstDTOModelType.OPERATION;
import static org.mule.runtime.ast.internal.serialization.dto.ComponentAstDTOModelType.SOURCE;
import static org.mule.runtime.ast.internal.serialization.dto.ComponentAstDTOModelType.TYPE;

import static java.util.Arrays.asList;
import static java.util.Collections.emptyList;
import static java.util.Collections.singleton;
import static java.util.Collections.singletonList;
import static java.util.Optional.empty;
import static java.util.Optional.of;

import static com.google.common.collect.Lists.newArrayList;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.is;
import static org.hamcrest.Matchers.notNullValue;
import static org.junit.rules.ExpectedException.none;
import static org.mockito.ArgumentMatchers.any;
import static org.mockito.ArgumentMatchers.isNull;
import static org.mockito.ArgumentMatchers.same;
import static org.mockito.Mockito.doAnswer;
import static org.mockito.Mockito.doReturn;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.times;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;

import org.mule.metadata.api.annotation.TypeAliasAnnotation;
import org.mule.metadata.api.annotation.TypeAnnotation;
import org.mule.metadata.api.annotation.TypeIdAnnotation;
import org.mule.metadata.api.model.MetadataFormat;
import org.mule.metadata.api.model.MetadataType;
import org.mule.metadata.api.model.ObjectFieldType;
import org.mule.metadata.api.model.ObjectKeyType;
import org.mule.metadata.api.model.ObjectType;
import org.mule.metadata.api.model.StringType;
import org.mule.metadata.api.model.UnionType;
import org.mule.metadata.api.visitor.MetadataTypeVisitor;
import org.mule.runtime.api.component.ComponentIdentifier;
import org.mule.runtime.api.component.TypedComponentIdentifier.ComponentType;
import org.mule.runtime.api.component.location.ComponentLocation;
import org.mule.runtime.api.meta.model.ComponentModel;
import org.mule.runtime.api.meta.model.ComposableModel;
import org.mule.runtime.api.meta.model.ExtensionModel;
import org.mule.runtime.api.meta.model.SubTypesModel;
import org.mule.runtime.api.meta.model.config.ConfigurationModel;
import org.mule.runtime.api.meta.model.connection.ConnectionProviderModel;
import org.mule.runtime.api.meta.model.construct.ConstructModel;
import org.mule.runtime.api.meta.model.nested.NestedRouteModel;
import org.mule.runtime.api.meta.model.operation.OperationModel;
import org.mule.runtime.api.meta.model.parameter.ParameterGroupModel;
import org.mule.runtime.api.meta.model.parameter.ParameterModel;
import org.mule.runtime.api.meta.model.parameter.ParameterizedModel;
import org.mule.runtime.api.meta.model.source.SourceModel;
import org.mule.runtime.api.meta.model.stereotype.HasStereotypeModel;
import org.mule.runtime.api.meta.model.stereotype.StereotypeModel;
import org.mule.runtime.api.meta.type.TypeCatalog;
import org.mule.runtime.ast.internal.builder.PropertiesResolver;
import org.mule.runtime.ast.internal.model.ExtensionModelHelper;
import org.mule.runtime.ast.internal.serialization.resolver.DefaultGenerationInformationResolver;
import org.mule.runtime.ast.internal.serialization.resolver.GenerationInformationResolver;
import org.mule.runtime.ast.internal.serialization.visitor.DTOMetadataTypeVisitor;
import org.mule.runtime.extension.api.declaration.type.annotation.StereotypeTypeAnnotation;
import org.mule.runtime.extension.api.dsl.syntax.DslElementSyntax;
import org.mule.runtime.extension.api.property.ClassLoaderModelProperty;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import javax.xml.namespace.QName;

import io.qameta.allure.Feature;
import io.qameta.allure.Issue;
import io.qameta.allure.Story;
import org.hamcrest.Matchers;
import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.ExpectedException;

@Feature(AST_SERIALIZATION)
@Story(AST_SERIALIZATION_ENRICH)
public class ComponentAstDTOTestCase {

  @Rule
  public ExpectedException exception = none();

  private GenerationInformationResolver generationInformationResolver;
  private ComponentAstDTO componentAstDTO;
  private ExtensionModelHelper extensionModelHelper;
  private ObjectType metadataType;

  private ComponentAstDTO parentComponentAstDTO;
  private ExtensionModel extensionModel;
  private ComposableModel composableModel;
  private NestedRouteModel nestableModel;

  private ClassLoader applicationClassLoader;
  private TypeCatalog typeCatalog;
  private ClassLoader pluginClassLoader;

  @Before
  public void setUp() {
    generationInformationResolver = mock(DefaultGenerationInformationResolver.class);
    componentAstDTO = mock(ComponentAstDTO.class);
    metadataType = mock(ObjectType.class);
    this.typeCatalog = mock(TypeCatalog.class);
    extensionModelHelper = mock(ExtensionModelHelper.class);
    when(extensionModelHelper.findMetadataType(String.class)).thenReturn(of(mock(StringType.class)));
    when(extensionModelHelper.getTypeCatalog()).thenReturn(typeCatalog);
    extensionModel = mock(ExtensionModel.class);
    composableModel = mock(ComposableModel.class);
    nestableModel = mock(NestedRouteModel.class);

    doReturn(singletonList(nestableModel))
        .when(composableModel)
        .getNestedComponents();
    parentComponentAstDTO = mock(ComponentAstDTO.class);

    ClassLoaderModelProperty classLoaderModelProperty = mock(ClassLoaderModelProperty.class);
    this.pluginClassLoader = mock(ClassLoader.class);
    when(classLoaderModelProperty.getClassLoader()).thenReturn(pluginClassLoader);
    when(extensionModel.getModelProperty(ClassLoaderModelProperty.class)).thenReturn(of(classLoaderModelProperty));

    this.applicationClassLoader = mock(ClassLoader.class);
  }

  @Test
  public void testOnConfigurationResolvesModel_WhenVisitingConfigurationModelWithMatchingModelName() {
    // Given
    String modelName = "config";
    ConfigurationModel configurationModel = mock(ConfigurationModel.class);
    when(configurationModel.getName()).thenReturn(modelName);
    when(extensionModel.getConfigurationModel(modelName)).thenReturn(of(configurationModel));

    ComponentAstDTO componentAstDTO =
        makeComponentAstDTO("config", CONFIGURATION, emptyList(),
                            ComponentIdentifier.builder().namespaceUri("uri").namespace("ns").name("aName").build(),
                            configurationModel, new ArrayList<>(), ComponentType.UNKNOWN.name());

    // When
    componentAstDTO.resolveModelComponent(null, extensionModelHelper, generationInformationResolver);
    // Then
    assertThat(componentAstDTO.getModel(ConfigurationModel.class).get(), is(configurationModel));
    assertThat(componentAstDTO.getModel(ParameterizedModel.class).get(), is(configurationModel));
  }

  @Test
  public void testOnConfigurationDoesNotResolveModel_WhenVisitingConfigurationModelWithNonMatchingModelName() {
    // Given
    String modelName = "config";
    ConfigurationModel configurationModel = mock(ConfigurationModel.class);
    when(configurationModel.getName()).thenReturn(modelName);
    when(extensionModel.getConfigurationModel(modelName)).thenReturn(of(configurationModel));

    ComponentAstDTO componentAstDTO =
        makeComponentAstDTO("other", CONFIGURATION, emptyList(),
                            ComponentIdentifier.builder().namespaceUri("uri").namespace("ns").name("aName").build(),
                            configurationModel, new ArrayList<>(), ComponentType.UNKNOWN.name());

    // When
    componentAstDTO.resolveModelComponent(null, extensionModelHelper, generationInformationResolver);

    // Then
    assertThat(componentAstDTO.getModel(ConfigurationModel.class).isPresent(), is(false));
    assertThat(componentAstDTO.getModel(ParameterizedModel.class).isPresent(), is(false));
  }

  @Test
  public void testOnConnectionProviderResolvesModel_WhenVisitingConnectionProviderModelWithMatchingModelName() {
    // Given
    String modelName = "connection";
    ConnectionProviderModel connectionProviderModel = mock(ConnectionProviderModel.class);
    when(connectionProviderModel.getName()).thenReturn(modelName);
    when(extensionModel.getConnectionProviderModel(modelName)).thenReturn(of(connectionProviderModel));

    ComponentAstDTO componentAstDTO =
        makeComponentAstDTO(modelName, CONNECTION_PROVIDER, emptyList(),
                            ComponentIdentifier.builder().namespaceUri("uri").namespace("ns").name("aName").build(),
                            connectionProviderModel, new ArrayList<>(), ComponentType.UNKNOWN.name());

    // When
    componentAstDTO.resolveModelComponent(null, extensionModelHelper, generationInformationResolver);

    // Then
    assertThat(componentAstDTO.getModel(ConnectionProviderModel.class).get(), is(connectionProviderModel));
    assertThat(componentAstDTO.getModel(ParameterizedModel.class).get(), is(connectionProviderModel));
  }

  @Test
  public void testOnConnectionProviderDoesNotResolvesModel_WhenVisitingConnectionProviderModelWithNonMatchingModelName() {
    // Given
    String modelName = "connection";
    ConnectionProviderModel connectionProviderModel = mock(ConnectionProviderModel.class);
    when(connectionProviderModel.getName()).thenReturn(modelName);
    when(extensionModel.getConnectionProviderModel(modelName)).thenReturn(of(connectionProviderModel));

    ComponentAstDTO componentAstDTO =
        makeComponentAstDTO("another", CONNECTION_PROVIDER, emptyList(),
                            ComponentIdentifier.builder().namespaceUri("uri").namespace("ns").name("aName").build(),
                            connectionProviderModel, new ArrayList<>(), ComponentType.UNKNOWN.name());

    // When
    componentAstDTO.resolveModelComponent(null, extensionModelHelper, generationInformationResolver);

    // Then
    assertThat(componentAstDTO.getModel(ConnectionProviderModel.class).isPresent(), is(false));
    assertThat(componentAstDTO.getModel(ParameterizedModel.class).isPresent(), is(false));
  }

  @Test
  public void testOnOperationResolvesModel_WhenVisitingOperationModelWithMatchingModelName() {
    // Given
    String modelName = "operation";
    OperationModel operationModel = mock(OperationModel.class);
    when(operationModel.getName()).thenReturn(modelName);
    when(extensionModel.getOperationModel(modelName)).thenReturn(of(operationModel));

    ComponentAstDTO componentAstDTO =
        makeComponentAstDTO(modelName, OPERATION, emptyList(),
                            ComponentIdentifier.builder().namespaceUri("uri").namespace("ns").name("aName").build(),
                            operationModel, new ArrayList<>(), ComponentType.UNKNOWN.name());

    // When
    componentAstDTO.resolveModelComponent(null, extensionModelHelper, generationInformationResolver);

    // Then
    assertThat(componentAstDTO.getModel(OperationModel.class).get(), is(operationModel));
    assertThat(componentAstDTO.getModel(ParameterizedModel.class).get(), is(operationModel));
  }

  @Test
  public void testOnOperationDoesNotResolvesModel_WhenVisitingOperationModelWithNonMatchingModelName() {
    // Given
    String modelName = "operation";
    OperationModel operationModel = mock(OperationModel.class);
    when(operationModel.getName()).thenReturn(modelName);
    when(extensionModel.getOperationModel(modelName)).thenReturn(of(operationModel));

    ComponentAstDTO componentAstDTO =
        makeComponentAstDTO("some other", OPERATION, emptyList(),
                            ComponentIdentifier.builder().namespaceUri("uri").namespace("ns").name("aName").build(),
                            operationModel, new ArrayList<>(), ComponentType.UNKNOWN.name());

    // When
    componentAstDTO.resolveModelComponent(null, extensionModelHelper, generationInformationResolver);

    // Then
    assertThat(componentAstDTO.getModel(OperationModel.class).isPresent(), is(false));
    assertThat(componentAstDTO.getModel(ParameterizedModel.class).isPresent(), is(false));
  }

  @Test
  public void testOnSourceResolvesModel_WhenVisitingSourceModelWithMatchingModelName() {
    // Given
    String modelName = "source";
    SourceModel sourceModel = mock(SourceModel.class);
    when(sourceModel.getName()).thenReturn(modelName);
    when(extensionModel.getSourceModel(modelName)).thenReturn(of(sourceModel));

    ComponentAstDTO componentAstDTO =
        makeComponentAstDTO(modelName, SOURCE, emptyList(),
                            ComponentIdentifier.builder().namespaceUri("uri").namespace("ns").name("aName").build(), sourceModel,
                            new ArrayList<>(), ComponentType.UNKNOWN.name());

    // When
    componentAstDTO.resolveModelComponent(null, extensionModelHelper, generationInformationResolver);

    // Then
    assertThat(componentAstDTO.getModel(SourceModel.class).get(), is(sourceModel));
    assertThat(componentAstDTO.getModel(ParameterizedModel.class).get(), is(sourceModel));
  }

  @Test
  public void testOnSourceDoesNotResolvesModel_WhenVisitingSourceModelWithNonMatchingModelName() {
    // Given
    String modelName = "source";
    SourceModel sourceModel = mock(SourceModel.class);
    when(sourceModel.getName()).thenReturn(modelName);
    when(extensionModel.getSourceModel(modelName)).thenReturn(of(sourceModel));

    ComponentAstDTO componentAstDTO =
        makeComponentAstDTO("john", SOURCE, emptyList(),
                            ComponentIdentifier.builder().namespaceUri("uri").namespace("ns").name("aName").build(), sourceModel,
                            new ArrayList<>(), ComponentType.UNKNOWN.name());
    ExtensionModel extensionModel = mock(ExtensionModel.class);
    when(extensionModel.getSourceModel(modelName)).thenReturn(empty());
    componentAstDTO.setExtensionModel(extensionModel);


    // When
    componentAstDTO.resolveModelComponent(null, extensionModelHelper, generationInformationResolver);

    // Then
    assertThat(componentAstDTO.getModel(ParameterizedModel.class).isPresent(), is(false));
  }

  @Test
  public void testOnConstructResolvesModel_WhenVisitingConstructModelWithMatchingModelName() {
    // Given
    String modelName = "source";
    ConstructModel constructModel = mock(ConstructModel.class);
    when(constructModel.getName()).thenReturn(modelName);
    when(extensionModel.getConstructModel(modelName)).thenReturn(of(constructModel));

    ComponentAstDTO componentAstDTO =
        makeComponentAstDTO(modelName, CONSTRUCT, emptyList(),
                            ComponentIdentifier.builder().namespaceUri("uri").namespace("ns").name("aName").build(),
                            constructModel, new ArrayList<>(), ComponentType.UNKNOWN.name());

    // When
    componentAstDTO.resolveModelComponent(null, extensionModelHelper, generationInformationResolver);

    // Then
    assertThat(componentAstDTO.getModel(ConstructModel.class).get(), is(constructModel));
    assertThat(componentAstDTO.getModel(ParameterizedModel.class).get(), is(constructModel));
  }

  @Test
  public void testOnConstructDoesNotResolvesModel_WhenVisitingConstructModelWithNonMatchingModelName() {
    // Given
    String modelName = "source";
    ConstructModel constructModel = mock(ConstructModel.class);
    when(constructModel.getName()).thenReturn(modelName);
    when(extensionModel.getConstructModel(modelName)).thenReturn(of(constructModel));

    ComponentAstDTO componentAstDTO =
        makeComponentAstDTO("Max", CONSTRUCT, emptyList(),
                            ComponentIdentifier.builder().namespaceUri("uri").namespace("ns").name("aName").build(),
                            constructModel, new ArrayList<>(), ComponentType.UNKNOWN.name());

    // When
    componentAstDTO.resolveModelComponent(null, extensionModelHelper, generationInformationResolver);

    // Then
    assertThat(componentAstDTO.getModel(ConstructModel.class).isPresent(), is(false));
    assertThat(componentAstDTO.getModel(ParameterizedModel.class).isPresent(), is(false));
  }

  @Test
  public void testOnNestableElementResolvesModel_WhenVisitingConstructModelWithMatchingModelName() {
    // Given
    String modelName = "source";
    when(nestableModel.getName()).thenReturn(modelName);

    ComponentAstDTO componentAstDTO =
        makeComponentAstDTO(modelName, NESTED_COMPONENT, emptyList(),
                            ComponentIdentifier.builder().namespaceUri("uri").namespace("ns").name("aName").build(),
                            nestableModel, new ArrayList<>(), ComponentType.UNKNOWN.name());

    when(parentComponentAstDTO.getModel(ComposableModel.class)).thenReturn(of(composableModel));

    // When
    componentAstDTO.resolveModelComponent(parentComponentAstDTO, extensionModelHelper, generationInformationResolver);

    // Then
    assertThat(componentAstDTO.getModel(NestedRouteModel.class).get(), is(nestableModel));
  }

  @Test
  public void testOnNestableElementDoesNotResolvesModel_WhenVisitingConstructModelWithNonMatchingModelName() {
    // Given
    String modelName = "source";
    when(nestableModel.getName()).thenReturn(modelName);

    ComponentAstDTO componentAstDTO =
        makeComponentAstDTO("nestable", NESTED_COMPONENT, emptyList(),
                            ComponentIdentifier.builder().namespaceUri("uri").namespace("ns").name("aName").build(),
                            nestableModel, new ArrayList<>(), ComponentType.UNKNOWN.name());

    when(parentComponentAstDTO.getModel(ComposableModel.class)).thenReturn(of(composableModel));

    // When
    componentAstDTO.resolveModelComponent(parentComponentAstDTO, extensionModelHelper, generationInformationResolver);

    // Then
    assertThat(componentAstDTO.getModel(NestedRouteModel.class).isPresent(), is(false));
  }

  @Test
  public void testOnTypeAcceptsADTOMetadataTypeVisitor_WhenWalkingAType() {
    // Given
    TypeCatalog typeCatalog = mock(TypeCatalog.class);
    when(extensionModelHelper.getTypeCatalog()).thenReturn(typeCatalog);
    ParameterizedModel parameterizedModel = mock(ParameterizedModel.class);

    ComponentAstDTO componentAstDTO =
        makeComponentAstDTO("someType", TYPE, emptyList(),
                            ComponentIdentifier.builder().namespaceUri("uri").namespace("ns").name("aName").build(),
                            parameterizedModel, new ArrayList<>(), ComponentType.UNKNOWN.name());

    ObjectType type = mock(ObjectType.class);
    when(typeCatalog.getType("someType")).thenReturn(of(type));

    // When
    componentAstDTO.resolveModelComponent(null, extensionModelHelper, generationInformationResolver);

    // Then
    verify(type, times(1)).accept(any(DTOMetadataTypeVisitor.class));
  }

  @Test
  public void testOnSubTypeAcceptsADTOMetadataTypeVisitor_WhenWalkingAType() {
    // Given
    TypeCatalog typeCatalog = mock(TypeCatalog.class);
    when(extensionModelHelper.getTypeCatalog()).thenReturn(typeCatalog);
    ParameterizedModel parameterizedModel = mock(ParameterizedModel.class);

    ComponentAstDTO componentAstDTO =
        makeComponentAstDTO("someType", TYPE, emptyList(),
                            ComponentIdentifier.builder().namespaceUri("uri").namespace("ns").name("aName").build(),
                            parameterizedModel, new ArrayList<>(), ComponentType.UNKNOWN.name());

    ObjectType baseType = mock(ObjectType.class);
    ObjectType subType = mock(ObjectType.class);
    when(subType.getAnnotation(TypeIdAnnotation.class)).thenReturn(of(new TypeIdAnnotation("someType")));
    doAnswer(inv -> {
      inv.getArgument(0, MetadataTypeVisitor.class).visitObject((ObjectType) inv.getMock());
      return null;
    }).when(subType).accept(any());

    when(typeCatalog.getType("someType")).thenReturn(empty());
    when(extensionModel.getSubTypes()).thenReturn(singleton(new SubTypesModel(baseType, singleton(subType))));
    // When
    componentAstDTO.resolveModelComponent(null, extensionModelHelper, generationInformationResolver);

    // Then
    verify(subType, times(1)).accept(any(DTOMetadataTypeVisitor.class));
  }

  @Test
  public void testSetPropertiesResolverSetsPropertiesResolverOfParameters_WhenSettingPropertiesResolver() {
    // Given
    PropertiesResolver propertiesResolver = new PropertiesResolver();

    ComponentParameterAstDTO componentParameterAstDTO1 = mock(ComponentParameterAstDTO.class);
    ComponentParameterAstDTO componentParameterAstDTO2 = mock(ComponentParameterAstDTO.class);
    ComponentParameterAstDTO componentParameterAstDTO3 = mock(ComponentParameterAstDTO.class);

    List<ComponentParameterAstDTO> parameters =
        newArrayList(componentParameterAstDTO1, componentParameterAstDTO2, componentParameterAstDTO3);
    ParameterizedModel parameterizedModel = mock(ParameterizedModel.class);
    ComponentAstDTO componentAstDTO =
        makeComponentAstDTO(null, null, parameters, null, parameterizedModel, emptyList(), UNKNOWN.toString());

    // When
    componentAstDTO.setPropertiesResolver(propertiesResolver);

    // Then
    verify(componentParameterAstDTO1, times(1)).setPropertiesResolver(propertiesResolver);
    verify(componentParameterAstDTO2, times(1)).setPropertiesResolver(propertiesResolver);
    verify(componentParameterAstDTO3, times(1)).setPropertiesResolver(propertiesResolver);

  }

  @Test
  public void testSetPropertiesResolverSetsPropertiesResolver_WhenSettingPropertiesResolver() {
    // Given
    PropertiesResolver propertiesResolver = new PropertiesResolver();
    ParameterizedModel parameterizedModel = mock(ParameterizedModel.class);
    ComponentAstDTO componentAstDTO =
        makeComponentAstDTO(null, null, emptyList(), null, parameterizedModel, emptyList(), UNKNOWN.toString());

    // When
    componentAstDTO.setPropertiesResolver(propertiesResolver);

    // Then
    assertThat(componentAstDTO.getPropertiesResolver(), is(propertiesResolver));

  }

  @Test
  public void testResolveModelForInfrastructureParametersSetsMetadataTypeAdapterAsParameterizedModel_WhenResolvingInfrastructureParameterModel() {
    // Given
    ParameterizedModel parameterizedModel = mock(ParameterizedModel.class);
    ComponentAstDTO componentAstDTO =
        makeComponentAstDTO("errorMappings", ComponentAstDTOModelType.INFRASTRUCTURE, new ArrayList<>(), null, parameterizedModel,
                            new ArrayList<>(), ComponentType.UNKNOWN.name());
    MetadataType nonStereotypedType = mock(MetadataType.class);
    when(nonStereotypedType.getAnnotation(StereotypeTypeAnnotation.class)).thenReturn(empty());
    componentAstDTO.setType(nonStereotypedType);

    // When
    componentAstDTO.resolveModelComponent(componentAstDTO, extensionModelHelper, generationInformationResolver);

    // Then
    assertThat(componentAstDTO.getModel(ParameterizedModel.class), notNullValue());
  }

  @Test
  @Issue("MULE-19819")
  public void testResolveModelForInfrastructureParametersSetsMetadataTypeAdapterWithStereotypeAsParameterizedModel_WhenResolvingInfrastructureParameterModel() {
    // Given
    ParameterizedModel parameterizedModel = mock(ParameterizedModel.class);
    ComponentAstDTO componentAstDTO =
        makeComponentAstDTO("errorMappings", ComponentAstDTOModelType.INFRASTRUCTURE, new ArrayList<>(), null, parameterizedModel,
                            new ArrayList<>(), ComponentType.UNKNOWN.name());
    MetadataType stereotypedType = mock(MetadataType.class);
    StereotypeModel stereotypeModel = mock(StereotypeModel.class);
    when(stereotypedType.getAnnotation(StereotypeTypeAnnotation.class))
        .thenReturn(of(new StereotypeTypeAnnotation(singletonList(stereotypeModel))));
    componentAstDTO.setType(stereotypedType);

    // When
    componentAstDTO.resolveModelComponent(componentAstDTO, extensionModelHelper, generationInformationResolver);

    // Then
    assertThat(componentAstDTO.getModel(ParameterizedModel.class).get(), Matchers.instanceOf(HasStereotypeModel.class));
    assertThat(((HasStereotypeModel) componentAstDTO.getModel(ParameterizedModel.class).get()).getStereotype(),
               is(stereotypeModel));
  }

  @Test
  public void testResolveModelForInfrastructureParametersResolvesGenerationInformation_WhenResolvingInfrastructureParameterModel() {
    // Given
    ParameterizedModel parameterizedModel = mock(ParameterizedModel.class);
    ComponentAstDTO componentAstDTO =
        makeComponentAstDTO("errorMappings", ComponentAstDTOModelType.INFRASTRUCTURE, new ArrayList<>(), null, parameterizedModel,
                            new ArrayList<>(), ComponentType.UNKNOWN.name());
    ComponentGenerationInformationDTO componentGenerationInformationDTO = new ComponentGenerationInformationDTO(null);
    ExtensionModel extensionModel = mock(ExtensionModel.class);
    componentAstDTO.setExtensionModel(extensionModel);
    when(generationInformationResolver
        .resolveComponentAstGenerationInformation(isNull(), same(componentAstDTO), same(extensionModelHelper)))
            .thenReturn(componentGenerationInformationDTO);

    MetadataType nonStereotypedType = mock(MetadataType.class);
    when(nonStereotypedType.getAnnotation(StereotypeTypeAnnotation.class)).thenReturn(empty());
    componentAstDTO.setType(nonStereotypedType);

    // When
    componentAstDTO.resolveModelComponent(componentAstDTO, extensionModelHelper, generationInformationResolver);

    // Then
    assertThat(componentAstDTO.getGenerationInformation(), is(componentGenerationInformationDTO));
  }

  @Test
  public void testResolveModelForInfrastructureParametersResolvesModelsOfParameters_WhenResolvingInfrastructureParameterModel() {
    prepareResolver();

    ComponentParameterAstDTO componentParameterAstDTO1 = makeComponentParameterAstDTO("name", null);
    ComponentParameterAstDTO componentParameterAstDTO2 = makeComponentParameterAstDTO("name", null);

    ParameterizedModel parameterizedModel = mock(ParameterizedModel.class);
    ComponentAstDTO componentAstDTO = makeComponentAstDTO("errorMappings", ComponentAstDTOModelType.INFRASTRUCTURE,
                                                          asList(componentParameterAstDTO1, componentParameterAstDTO2), null,
                                                          parameterizedModel, new ArrayList<>(),
                                                          ComponentType.UNKNOWN.name());
    componentAstDTO.setType(metadataType);
    TypeAliasAnnotation typeAliasAnnotation = mock(TypeAliasAnnotation.class);
    when(typeAliasAnnotation.getValue()).thenReturn("parameter group name");
    when(metadataType.getAnnotation(TypeAliasAnnotation.class)).thenReturn(of(typeAliasAnnotation));

    // When
    componentAstDTO.resolveModelComponent(componentAstDTO, extensionModelHelper, generationInformationResolver);

    // Then
    assertThat(componentParameterAstDTO1.getModel(), notNullValue());
    assertThat(componentParameterAstDTO1.getGroupModel(), notNullValue());
    assertThat(componentParameterAstDTO2.getModel(), notNullValue());
    assertThat(componentParameterAstDTO2.getGroupModel(), notNullValue());
  }

  @Test
  @Issue("MULE-19806")
  public void generationInformationForUnionType() {
    // Given
    prepareResolver();

    ObjectKeyType key = mock(ObjectKeyType.class);
    when(key.getName()).thenReturn(new QName("unionParam"));

    ObjectType unionObjectType1 = mock(ObjectType.class);
    ObjectType unionObjectType2 = mock(ObjectType.class);

    UnionType unionType = mock(UnionType.class);
    doAnswer(inv -> {
      inv.getArgument(0, MetadataTypeVisitor.class).visitUnion(unionType);
      return null;
    }).when(unionType).accept(any(MetadataTypeVisitor.class));
    when(unionType.getTypes()).thenReturn(asList(unionObjectType1, unionObjectType2));

    ObjectFieldType field = mock(ObjectFieldType.class);
    when(field.getValue()).thenReturn(unionType);
    when(field.getKey()).thenReturn(key);

    when(metadataType.getFields()).thenReturn(singleton(field));

    ComponentParameterAstDTO componentParameterAstDTO1 = makeComponentParameterAstDTO("name", null);

    ComponentAstDTO valueComponentAstDTO = mock(ComponentAstDTO.class);
    ComponentIdentifier valueIdentifier = ComponentIdentifier.builder().name("unionObjectType1").namespace("ns").build();
    when(valueComponentAstDTO.getIdentifier()).thenReturn(valueIdentifier);
    ComponentParameterAstDTO componentParameterAstDTO2 =
        makeComponentParameterAstDTO("unionParam", new ParameterValueContainer(null, valueComponentAstDTO));

    ComponentIdentifier paramIdentifier = ComponentIdentifier.builder().name("unionParam").namespace("ns").build();
    ParameterizedModel parameterizedModel = mock(ParameterizedModel.class);
    ComponentAstDTO componentAstDTO =
        makeComponentAstDTO("errorMappings", ComponentAstDTOModelType.INFRASTRUCTURE,
                            asList(componentParameterAstDTO1, componentParameterAstDTO2), paramIdentifier, parameterizedModel,
                            new ArrayList<>(), ComponentType.UNKNOWN.name());
    componentAstDTO.setType(metadataType);
    TypeAliasAnnotation typeAliasAnnotation = mock(TypeAliasAnnotation.class);
    when(typeAliasAnnotation.getValue()).thenReturn("parameter group name");
    when(metadataType.getAnnotation(TypeAliasAnnotation.class)).thenReturn(of(typeAliasAnnotation));

    DslElementSyntax paramValueSyntax1 = mock(DslElementSyntax.class);
    when(paramValueSyntax1.getElementName()).thenReturn("unionObjectType1");
    when(extensionModelHelper.resolveDslElementModel(unionObjectType1, paramIdentifier.getNamespace()))
        .thenReturn(of(paramValueSyntax1));
    DslElementSyntax paramValueSyntax2 = mock(DslElementSyntax.class);
    when(paramValueSyntax2.getElementName()).thenReturn("unionObjectType2");
    when(extensionModelHelper.resolveDslElementModel(unionObjectType2, paramIdentifier.getNamespace()))
        .thenReturn(of(paramValueSyntax2));

    // When
    componentAstDTO.resolveModelComponent(componentAstDTO, extensionModelHelper, generationInformationResolver);

    // Then
    assertThat(componentParameterAstDTO2.getGenerationInformation().getSyntax().get(), is(paramValueSyntax1));
  }

  @Test
  public void testResolveParameterModelByNameResolvesParameterModel_WhenComponentHasComponentModelWithMatchingParameterModel() {
    // Given
    String parameterModelName = "aParameterModelName";
    ComponentModel componentModel = mock(ComponentModel.class);
    ParameterModel parameterModel = mock(ParameterModel.class);
    when(parameterModel.getName()).thenReturn(parameterModelName);
    when(componentModel.getAllParameterModels()).thenReturn(singletonList(parameterModel));

    ComponentParameterAstDTO componentParameterAstDTO = mock(ComponentParameterAstDTO.class);
    when(componentParameterAstDTO.getModelName()).thenReturn(parameterModelName);

    ComponentAstDTO componentAstDTO =
        makeComponentAstDTO("aModelName", mock(ComponentAstDTOModelType.class), singletonList(componentParameterAstDTO),
                            ComponentIdentifier.builder().namespaceUri("uri").namespace("ns").name("aName").build(),
                            componentModel, new ArrayList<>(), ComponentType.UNKNOWN.name());

    // When
    ParameterModel result = componentAstDTO.resolveParameterModel(componentParameterAstDTO);

    // Then
    assertThat(result, is(parameterModel));
  }

  @Test
  public void testResolveParameterModelByNameResolvesParameterModel_WhenComponentHasConfigurationModelWithMatchingParameterModel() {
    // Given
    String parameterModelName = "aParameterModelName";
    ConfigurationModel configurationModel = mock(ConfigurationModel.class);
    ParameterModel parameterModel = mock(ParameterModel.class);
    when(parameterModel.getName()).thenReturn(parameterModelName);
    when(configurationModel.getAllParameterModels()).thenReturn(singletonList(parameterModel));

    ComponentParameterAstDTO componentParameterAstDTO = mock(ComponentParameterAstDTO.class);
    when(componentParameterAstDTO.getModelName()).thenReturn(parameterModelName);
    ComponentAstDTO componentAstDTO =
        makeComponentAstDTO("aModelName", mock(ComponentAstDTOModelType.class), singletonList(componentParameterAstDTO),
                            ComponentIdentifier.builder().namespaceUri("uri").namespace("ns").name("aName").build(),
                            configurationModel, new ArrayList<>(), ComponentType.UNKNOWN.name());

    // When
    ParameterModel result = componentAstDTO.resolveParameterModel(componentParameterAstDTO);

    // Then
    assertThat(result, is(parameterModel));
  }

  @Test
  public void testResolveParameterModelByNameResolvesParameterModel_WhenComponentHasConnectionProviderModelWithMatchingParameterModel() {
    // Given
    String parameterModelName = "aParameterModelName";
    ConnectionProviderModel connectionProviderModel = mock(ConnectionProviderModel.class);
    ParameterModel parameterModel = mock(ParameterModel.class);
    when(parameterModel.getName()).thenReturn(parameterModelName);
    when(connectionProviderModel.getAllParameterModels()).thenReturn(singletonList(parameterModel));

    ComponentParameterAstDTO componentParameterAstDTO = mock(ComponentParameterAstDTO.class);
    when(componentParameterAstDTO.getModelName()).thenReturn(parameterModelName);
    ComponentAstDTO componentAstDTO =
        makeComponentAstDTO("aModelName", mock(ComponentAstDTOModelType.class), singletonList(componentParameterAstDTO),
                            ComponentIdentifier.builder().namespaceUri("uri").namespace("ns").name("aName").build(),
                            connectionProviderModel, new ArrayList<>(), ComponentType.UNKNOWN.name());

    // When
    ParameterModel result = componentAstDTO.resolveParameterModel(componentParameterAstDTO);

    // Then
    assertThat(result, is(parameterModel));
  }

  @Test
  public void testResolveParameterModelByNameResolvesParameterModel_WhenComponentHasParameterizedModelWithMatchingParameterModel() {
    // Given
    String parameterModelName = "aParameterModelName";
    ParameterizedModel parameterizedModel = mock(ParameterizedModel.class);
    ParameterModel parameterModel = mock(ParameterModel.class);
    when(parameterModel.getName()).thenReturn(parameterModelName);
    when(parameterizedModel.getAllParameterModels()).thenReturn(singletonList(parameterModel));

    ComponentParameterAstDTO componentParameterAstDTO = mock(ComponentParameterAstDTO.class);
    when(componentParameterAstDTO.getModelName()).thenReturn(parameterModelName);
    ComponentAstDTO componentAstDTO =
        makeComponentAstDTO("aModelName", mock(ComponentAstDTOModelType.class), singletonList(componentParameterAstDTO),
                            ComponentIdentifier.builder().namespaceUri("uri").namespace("ns").name("aName").build(),
                            parameterizedModel, new ArrayList<>(), ComponentType.UNKNOWN.name());

    // When
    ParameterModel result = componentAstDTO.resolveParameterModel(componentParameterAstDTO);

    // Then
    assertThat(result, is(parameterModel));
  }

  @Test
  public void testResolveParameterModelByNameDoesNotResolveParameterModel_WhenComponentHasParameterizedModelWithNoMatchingParameterModel() {
    // Given
    String parameterModelName = "aParameterModelName";
    ParameterizedModel parameterizedModel = mock(ParameterizedModel.class);
    ParameterModel parameterModel = mock(ParameterModel.class);
    when(parameterModel.getName()).thenReturn("non matching name");
    when(parameterizedModel.getAllParameterModels()).thenReturn(singletonList(parameterModel));

    ComponentParameterAstDTO componentParameterAstDTO = mock(ComponentParameterAstDTO.class);
    when(componentParameterAstDTO.getModelName()).thenReturn(parameterModelName);
    ComponentAstDTO componentAstDTO =
        makeComponentAstDTO("aModelName", mock(ComponentAstDTOModelType.class), singletonList(componentParameterAstDTO),
                            ComponentIdentifier.builder().namespaceUri("uri").namespace("ns").name("aName").build(),
                            parameterizedModel, new ArrayList<>(), ComponentType.UNKNOWN.name());

    // Then
    exception.expect(IllegalArgumentException.class);
    exception.expectMessage("Could not resolve ParameterModel for [" + parameterModelName + "]");

    // When
    componentAstDTO.resolveParameterModel(componentParameterAstDTO);
  }

  @Test
  public void testResolveParameterGroupModelByNameResolvesParameterGroupModel_WhenComponentHasComponentModelWithMatchingParameterGroupModel() {
    // Given
    ComponentParameterAstDTO componentParameterAstDTO = mock(ComponentParameterAstDTO.class);
    String parameterModelName = "aParameterModelName";
    when(componentParameterAstDTO.getModelName()).thenReturn(parameterModelName);
    String parameterGroupModelName = "aParameterGroupModelName";
    when(componentParameterAstDTO.getGroupModelName()).thenReturn(parameterGroupModelName);
    ComponentModel componentModel = mock(ComponentModel.class);
    ParameterGroupModel parameterGroupModel = mock(ParameterGroupModel.class);
    when(parameterGroupModel.getName()).thenReturn(parameterGroupModelName);
    when(componentModel.getParameterGroupModels()).thenReturn(singletonList(parameterGroupModel));

    ComponentAstDTO componentAstDTO =
        makeComponentAstDTO("aModelName", mock(ComponentAstDTOModelType.class), singletonList(componentParameterAstDTO),
                            ComponentIdentifier.builder().namespaceUri("uri").namespace("ns").name("aName").build(),
                            componentModel, new ArrayList<>(), ComponentType.UNKNOWN.name());

    // When
    ParameterGroupModel result = componentAstDTO.resolveParameterGroupModel(componentParameterAstDTO);

    // Then
    assertThat(result, is(parameterGroupModel));
  }

  @Test
  public void testResolveParameterGroupModelByNameResolvesParameterGroupModel_WhenComponentHasConfigurationModelWithMatchingParameterGroupModel() {
    // Given
    ComponentParameterAstDTO componentParameterAstDTO = mock(ComponentParameterAstDTO.class);
    String parameterModelName = "aParameterModelName";
    when(componentParameterAstDTO.getModelName()).thenReturn(parameterModelName);
    String parameterGroupModelName = "aParameterGroupModelName";
    when(componentParameterAstDTO.getGroupModelName()).thenReturn(parameterGroupModelName);
    ConfigurationModel configurationModel = mock(ConfigurationModel.class);
    ParameterGroupModel parameterGroupModel = mock(ParameterGroupModel.class);
    when(parameterGroupModel.getName()).thenReturn(parameterGroupModelName);
    when(configurationModel.getParameterGroupModels()).thenReturn(singletonList(parameterGroupModel));

    ComponentAstDTO componentAstDTO =
        makeComponentAstDTO("aModelName", mock(ComponentAstDTOModelType.class), singletonList(componentParameterAstDTO),
                            ComponentIdentifier.builder().namespaceUri("uri").namespace("ns").name("aName").build(),
                            configurationModel, new ArrayList<>(), ComponentType.UNKNOWN.name());

    // When
    ParameterGroupModel result = componentAstDTO.resolveParameterGroupModel(componentParameterAstDTO);

    // Then
    assertThat(result, is(parameterGroupModel));
  }

  @Test
  public void testResolveParameterGroupModelByNameResolvesParameterGroupModel_WhenComponentHasConnectionProviderModelWithMatchingParameterGroupModel() {
    // Given
    ComponentParameterAstDTO componentParameterAstDTO = mock(ComponentParameterAstDTO.class);
    String parameterModelName = "aParameterModelName";
    when(componentParameterAstDTO.getModelName()).thenReturn(parameterModelName);
    String parameterGroupModelName = "aParameterGroupModelName";
    when(componentParameterAstDTO.getGroupModelName()).thenReturn(parameterGroupModelName);
    ConnectionProviderModel connectionProviderModel = mock(ConnectionProviderModel.class);
    ParameterGroupModel parameterGroupModel = mock(ParameterGroupModel.class);
    when(parameterGroupModel.getName()).thenReturn(parameterGroupModelName);
    when(connectionProviderModel.getParameterGroupModels()).thenReturn(singletonList(parameterGroupModel));

    ComponentAstDTO componentAstDTO =
        makeComponentAstDTO("aModelName", mock(ComponentAstDTOModelType.class), singletonList(componentParameterAstDTO),
                            ComponentIdentifier.builder().namespaceUri("uri").namespace("ns").name("aName").build(),
                            connectionProviderModel, new ArrayList<>(), ComponentType.UNKNOWN.name());

    // When
    ParameterGroupModel result = componentAstDTO.resolveParameterGroupModel(componentParameterAstDTO);

    // Then
    assertThat(result, is(parameterGroupModel));
  }

  @Test
  public void testResolveParameterGroupModelByNameResolvesParameterGroupModel_WhenComponentHasParameterizedModelWithMatchingParameterGroupModel() {
    // Given
    ComponentParameterAstDTO componentParameterAstDTO = mock(ComponentParameterAstDTO.class);
    String parameterModelName = "aParameterModelName";
    when(componentParameterAstDTO.getModelName()).thenReturn(parameterModelName);
    String parameterGroupModelName = "aParameterGroupModelName";
    when(componentParameterAstDTO.getGroupModelName()).thenReturn(parameterGroupModelName);
    ParameterizedModel connectionProviderModel = mock(ParameterizedModel.class);
    ParameterGroupModel parameterGroupModel = mock(ParameterGroupModel.class);
    when(parameterGroupModel.getName()).thenReturn(parameterGroupModelName);
    when(connectionProviderModel.getParameterGroupModels()).thenReturn(singletonList(parameterGroupModel));

    ComponentAstDTO componentAstDTO =
        makeComponentAstDTO("aModelName", mock(ComponentAstDTOModelType.class), singletonList(componentParameterAstDTO),
                            ComponentIdentifier.builder().namespaceUri("uri").namespace("ns").name("aName").build(),
                            connectionProviderModel, new ArrayList<>(), ComponentType.UNKNOWN.name());

    // When
    ParameterGroupModel result = componentAstDTO.resolveParameterGroupModel(componentParameterAstDTO);

    // Then
    assertThat(result, is(parameterGroupModel));
  }

  @Test
  public void testResolveParameterGroupModelByNameDoesNotResolveParameterGroupModel_WhenComponentHasParameterizedModelWithNoMatchingParameterGroupModel() {
    // Given
    ComponentParameterAstDTO componentParameterAstDTO = mock(ComponentParameterAstDTO.class);
    String parameterModelName = "aParameterModelName";
    when(componentParameterAstDTO.getModelName()).thenReturn(parameterModelName);
    String parameterGroupModelName = "aParameterGroupModelName";
    when(componentParameterAstDTO.getGroupModelName()).thenReturn(parameterGroupModelName);
    ParameterizedModel connectionProviderModel = mock(ParameterizedModel.class);
    ParameterGroupModel parameterGroupModel = mock(ParameterGroupModel.class);
    when(parameterGroupModel.getName()).thenReturn("non matching name");
    when(connectionProviderModel.getParameterGroupModels()).thenReturn(singletonList(parameterGroupModel));

    ComponentAstDTO componentAstDTO =
        makeComponentAstDTO("aModelName", mock(ComponentAstDTOModelType.class), singletonList(componentParameterAstDTO),
                            ComponentIdentifier.builder().namespaceUri("uri").namespace("ns").name("aName").build(),
                            connectionProviderModel, new ArrayList<>(), ComponentType.UNKNOWN.name());

    // Then
    exception.expect(IllegalArgumentException.class);
    exception.expectMessage("Could not resolve ParameterGroupModel [" + parameterGroupModelName + "]");

    // When
    componentAstDTO.resolveParameterGroupModel(componentParameterAstDTO);
  }

  @Test
  public void testResolveTypeReturnsTypeFromTypeCatalog_WhenResolvingATypeThatIsInTheCatalog() {
    // Given
    String modelName = "some.model.Name";
    ObjectType objectType = mock(ObjectType.class);
    Optional<ObjectType> optionalObjectType = of(objectType);
    when(this.typeCatalog.getType(modelName)).thenReturn(optionalObjectType);
    ParameterizedModel parameterizedModel = mock(ParameterizedModel.class);
    ComponentAstDTO componentAstDTO =
        makeComponentAstDTO(modelName, null, null, null, parameterizedModel, null, ComponentType.UNKNOWN.toString());

    // When
    Optional<? extends MetadataType> optionalMetadataType =
        componentAstDTO.resolveType(extensionModelHelper, applicationClassLoader);

    // Then
    assertThat(optionalMetadataType, is(optionalObjectType));
  }

  @Test
  public void testResolveTypeReturnsTypeFromExtensionModelsSubtypes_WhenResolvingATypeThatIsNotInTheCatalogButIsASubtype() {
    // Given
    String modelName = "some.model.Name";
    SubTypesModel subTypesModel = mock(SubTypesModel.class);
    when(this.typeCatalog.getType(modelName)).thenReturn(empty());

    Set<SubTypesModel> subTypesModels = new HashSet<>();
    subTypesModels.add(subTypesModel);

    when(extensionModel.getSubTypes()).thenReturn(subTypesModels);
    ObjectType objectType = new TestObjectType(modelName);

    Set<ObjectType> subtypes = new HashSet<>();
    subtypes.add(objectType);

    when(subTypesModel.getSubTypes()).thenReturn(subtypes);
    ParameterizedModel parameterizedModel = mock(ParameterizedModel.class);
    ComponentAstDTO componentAstDTO =
        makeComponentAstDTO(modelName, null, null, null, parameterizedModel, null, ComponentType.UNKNOWN.toString());
    componentAstDTO.setExtensionModel(this.extensionModel);

    // When
    Optional<? extends MetadataType> optionalMetadataType =
        componentAstDTO.resolveType(extensionModelHelper, applicationClassLoader);

    // Then
    assertThat(optionalMetadataType, is(of(objectType)));
  }

  @Test
  public void testResolveTypeReturnsTypeFromApplicationClassLoader_WhenResolvingATypeThatIsNotInTheCatalogNorIsASubtypeButIsInApplicationClassLoader()
      throws ClassNotFoundException {
    // Given
    String modelName = "some.model.Name";
    when(this.typeCatalog.getType(modelName)).thenReturn(empty());

    Set<SubTypesModel> subTypesModels = new HashSet<>();

    when(extensionModel.getSubTypes()).thenReturn(subTypesModels);
    Class aClass = Class.class;
    when(applicationClassLoader.loadClass(modelName)).thenReturn(aClass);
    MetadataType metadataType = mock(MetadataType.class);
    Optional<MetadataType> typeOptional = of(metadataType);
    when(extensionModelHelper.findMetadataType(aClass)).thenReturn(typeOptional);
    ParameterizedModel parameterizedModel = mock(ParameterizedModel.class);
    ComponentAstDTO componentAstDTO =
        makeComponentAstDTO(modelName, null, null, null, parameterizedModel, null, ComponentType.UNKNOWN.toString());
    componentAstDTO.setExtensionModel(this.extensionModel);

    // When
    Optional<? extends MetadataType> optionalMetadataType =
        componentAstDTO.resolveType(extensionModelHelper, applicationClassLoader);

    // Then
    assertThat(optionalMetadataType, is(typeOptional));
  }

  @Test
  public void testResolveTypeReturnsTypeFromApplicationClassLoader_WhenResolvingATypeThatIsInnerClassInApplicationClassLoader()
      throws ClassNotFoundException {
    // Given
    String modelName = "some.model.Some.Name";
    String inner = "some.model.Some$Name";
    when(this.typeCatalog.getType(modelName)).thenReturn(empty());

    Set<SubTypesModel> subTypesModels = new HashSet<>();

    when(extensionModel.getSubTypes()).thenReturn(subTypesModels);
    Class aClass = Class.class;
    when(applicationClassLoader.loadClass(modelName)).thenThrow(new ClassNotFoundException());
    when(applicationClassLoader.loadClass(inner)).thenReturn(aClass);
    MetadataType metadataType = mock(MetadataType.class);
    Optional<MetadataType> typeOptional = of(metadataType);
    when(extensionModelHelper.findMetadataType(aClass)).thenReturn(typeOptional);
    ParameterizedModel parameterizedModel = mock(ParameterizedModel.class);
    ComponentAstDTO componentAstDTO =
        makeComponentAstDTO(modelName, null, null, null, parameterizedModel, null, ComponentType.UNKNOWN.toString());
    componentAstDTO.setExtensionModel(this.extensionModel);

    // When
    Optional<? extends MetadataType> optionalMetadataType =
        componentAstDTO.resolveType(extensionModelHelper, applicationClassLoader);

    // Then
    assertThat(optionalMetadataType, is(typeOptional));
  }

  @Test
  @Issue("MULE-19864")
  public void testResolveTypeReturnsTypeFromPluginClassLoader_WhenResolvingATypeThatIsInPluginClassLoader()
      throws ClassNotFoundException {
    // Given
    String modelName = "some.model.Some.Name";
    String inner = "some.model.Some$Name";
    when(this.typeCatalog.getType(modelName)).thenReturn(empty());

    Set<SubTypesModel> subTypesModels = new HashSet<>();

    when(extensionModel.getSubTypes()).thenReturn(subTypesModels);
    Class aClass = Class.class;
    when(applicationClassLoader.loadClass(modelName)).thenThrow(new ClassNotFoundException());
    when(applicationClassLoader.loadClass(inner)).thenThrow(new ClassNotFoundException());
    when(pluginClassLoader.loadClass(modelName)).thenReturn(aClass);
    MetadataType metadataType = mock(MetadataType.class);
    Optional<MetadataType> typeOptional = of(metadataType);
    when(extensionModelHelper.findMetadataType(aClass)).thenReturn(typeOptional);
    ParameterizedModel parameterizedModel = mock(ParameterizedModel.class);
    ComponentAstDTO componentAstDTO =
        makeComponentAstDTO(modelName, null, null, null, parameterizedModel, null, ComponentType.UNKNOWN.toString());
    componentAstDTO.setExtensionModel(this.extensionModel);

    // When
    Optional<? extends MetadataType> optionalMetadataType =
        componentAstDTO.resolveType(extensionModelHelper, applicationClassLoader);

    // Then
    assertThat(optionalMetadataType, is(typeOptional));
  }

  @Test
  @Issue("MULE-19864")
  public void testResolveTypeReturnsTypeFromPluginClassLoader_WhenResolvingATypeThatIsInnerClassInPluginClassLoader()
      throws ClassNotFoundException {
    // Given
    String modelName = "some.model.Some.Name";
    String inner = "some.model.Some$Name";
    when(this.typeCatalog.getType(modelName)).thenReturn(empty());

    Set<SubTypesModel> subTypesModels = new HashSet<>();

    when(extensionModel.getSubTypes()).thenReturn(subTypesModels);
    Class aClass = Class.class;
    when(applicationClassLoader.loadClass(modelName)).thenThrow(new ClassNotFoundException());
    when(applicationClassLoader.loadClass(inner)).thenThrow(new ClassNotFoundException());
    when(pluginClassLoader.loadClass(modelName)).thenThrow(new ClassNotFoundException());
    when(pluginClassLoader.loadClass(inner)).thenReturn(aClass);
    MetadataType metadataType = mock(MetadataType.class);
    Optional<MetadataType> typeOptional = of(metadataType);
    when(extensionModelHelper.findMetadataType(aClass)).thenReturn(typeOptional);
    ParameterizedModel parameterizedModel = mock(ParameterizedModel.class);
    ComponentAstDTO componentAstDTO =
        makeComponentAstDTO(modelName, null, null, null, parameterizedModel, null, ComponentType.UNKNOWN.toString());
    componentAstDTO.setExtensionModel(this.extensionModel);

    // When
    Optional<? extends MetadataType> optionalMetadataType =
        componentAstDTO.resolveType(extensionModelHelper, applicationClassLoader);

    // Then
    assertThat(optionalMetadataType, is(typeOptional));
  }

  protected void prepareResolver() {
    ComponentGenerationInformationDTO componentGenerationInformationDTO = new ComponentGenerationInformationDTO(null);
    ExtensionModel extensionModel = mock(ExtensionModel.class);
    when(componentAstDTO.getExtensionModel()).thenReturn(extensionModel);
    when(generationInformationResolver
        .resolveComponentAstGenerationInformation(any(ComponentParameterAstDTO.class), same(componentAstDTO),
                                                  same(extensionModelHelper))).thenReturn(componentGenerationInformationDTO);
  }

  private ComponentAstDTO makeComponentAstDTO(String modelName, ComponentAstDTOModelType modelType,
                                              Collection<ComponentParameterAstDTO> parameters, ComponentIdentifier identifier,
                                              ParameterizedModel componentModel, List<ComponentAstDTO> directChildren,
                                              String componentType) {
    ComponentLocation location = mock(ComponentLocation.class);
    ComponentMetadataAstDTO componentMetadataAstDTO = mock(ComponentMetadataAstDTO.class);
    ComponentAstDTO componentAstDTO =
        new ComponentAstDTO(directChildren, componentType, identifier, location, componentMetadataAstDTO,
                            parameters, "componentId", new ExtensionModelDTO("anExtensionModelName"), modelName, modelType,
                            new HashMap<>());
    componentAstDTO.setParameterizedModel(componentModel);
    componentAstDTO.setExtensionModel(this.extensionModel);
    return componentAstDTO;
  }

  private ComponentParameterAstDTO makeComponentParameterAstDTO(String modelName,
                                                                ParameterValueContainer parameterValueContainer) {
    return new ComponentParameterAstDTO(parameterValueContainer, null, false, null, "parameter group name",
                                        modelName);
  }

  private static class TestObjectType implements ObjectType {

    private final String className;

    private TestObjectType(String className) {
      this.className = className;
    }

    @Override
    public Optional<MetadataType> getOpenRestriction() {
      return empty();
    }

    @Override
    public Collection<ObjectFieldType> getFields() {
      return null;
    }

    @Override
    public boolean isOrdered() {
      return false;
    }

    @Override
    public Optional<ObjectFieldType> getFieldByName(String propertyName) {
      return empty();
    }

    @Override
    public MetadataFormat getMetadataFormat() {
      return null;
    }

    @Override
    public <T extends TypeAnnotation> Optional<T> getAnnotation(Class<T> annotation) {
      if (annotation.equals(TypeIdAnnotation.class)) {
        return (Optional<T>) of(new TypeIdAnnotation(className));
      }

      return empty();
    }

    @Override
    public Set<TypeAnnotation> getAnnotations() {
      return null;
    }

    @Override
    public Optional<String> getDescription() {
      return empty();
    }

    @Override
    public void accept(MetadataTypeVisitor visitor) {
      visitor.visitObject(this);
    }

  }

}
