/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.ast.internal.serialization.dto;

import com.google.common.collect.ImmutableMap;
import io.qameta.allure.Description;
import io.qameta.allure.Feature;
import io.qameta.allure.Issue;
import io.qameta.allure.Story;
import org.junit.Test;
import org.mule.runtime.ast.internal.builder.PropertiesResolver;

import java.util.Map;

import static java.util.Collections.emptyList;
import static java.util.Collections.emptyMap;
import static org.hamcrest.CoreMatchers.is;
import static org.hamcrest.CoreMatchers.nullValue;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.mule.runtime.ast.AllureConstants.ArtifactAstSerialization.AST_SERIALIZATION;
import static org.mule.runtime.ast.AllureConstants.ArtifactAstSerialization.AST_SERIALIZATION_ENRICH;

@Feature(AST_SERIALIZATION)
@Story(AST_SERIALIZATION_ENRICH)
public class ComponentMetadataAstDTOTestCase {

  @Test
  @Description("Property resolver can be set even if document attributes are null")
  public void whenDocAttributesAreNullThenPropertyResolverCanBeSet() {
    final ComponentMetadataAstDTO metadataAst = new ComponentMetadataAstDTO();

    metadataAst.enrich(emptyMap());
    metadataAst.setPropertiesResolver(new PropertiesResolver());
  }

  @Test
  @Description("Get document attributes when null")
  public void whenDocAttributesAreNullThenTheyCanBeRetrieved() {
    final ComponentMetadataAstDTO metadataAst = new ComponentMetadataAstDTO();
    assertThat(metadataAst.getDocAttributes(), is(nullValue()));

    // It is important to also check when there is property resolution involved.
    metadataAst.enrich(emptyMap());
    metadataAst.setPropertiesResolver(new PropertiesResolver());

    assertThat(metadataAst.getDocAttributes(), is(nullValue()));
  }

  @Test
  @Description("Checks that document attributes with properties are resolved")
  @Issue("MULE-20090")
  public void attributesAreResolved() {
    final PropertiesResolver propertiesResolver = new PropertiesResolver();
    propertiesResolver.setMappingFunction(k -> "10");

    final Map<String, String> docAttributes = ImmutableMap.of("paramWithProperty", "${value}");
    final ComponentMetadataAstDTO metadataAst =
        new ComponentMetadataAstDTO(docAttributes, null, null, null, null, emptyList(), null, null, null, null);
    metadataAst.enrich(emptyMap());

    metadataAst.setPropertiesResolver(propertiesResolver);

    assertThat(metadataAst.getDocAttributes().get("paramWithProperty"), is("10"));
  }

  @Test
  @Description("Checks that the property resolver can be hot-swapped")
  public void whenPropertyResolverIsUpdatedThenAttributesAreResolvedWithTheNewResolver() {
    final PropertiesResolver propertiesResolver = new PropertiesResolver();
    propertiesResolver.setMappingFunction(k -> "10");

    final Map<String, String> docAttributes = ImmutableMap.of("paramWithProperty", "${value}");
    final ComponentMetadataAstDTO metadataAst =
        new ComponentMetadataAstDTO(docAttributes, null, null, null, null, emptyList(), null, null, null, null);
    metadataAst.enrich(emptyMap());

    metadataAst.setPropertiesResolver(propertiesResolver);

    // Control test with the original mapping function.
    assertThat(metadataAst.getDocAttributes().get("paramWithProperty"), is("10"));

    // Updates the mapping function and rechecks.
    propertiesResolver.setMappingFunction(k -> "20");
    assertThat(metadataAst.getDocAttributes().get("paramWithProperty"), is("20"));
  }
}
