/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.ast.internal.serialization.json;

import static org.mule.runtime.ast.AllureConstants.ArtifactAstSerialization.AST_JSON_SERIALIZER;
import static org.mule.runtime.ast.AllureConstants.ArtifactAstSerialization.AST_SERIALIZATION;
import static org.mule.runtime.ast.internal.serialization.ArtifactAstSerializerFactory.JSON;
import static org.mule.runtime.ast.internal.serialization.json.JsonArtifactAstSerializer.SERIALIZER_VERSION;

import static java.nio.charset.Charset.defaultCharset;
import static java.nio.charset.StandardCharsets.UTF_8;
import static java.util.concurrent.Executors.newSingleThreadExecutor;

import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.containsString;
import static org.hamcrest.Matchers.is;
import static org.mockito.ArgumentMatchers.any;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.spy;
import static org.mockito.Mockito.times;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;

import org.mule.runtime.api.meta.model.ExtensionModel;
import org.mule.runtime.ast.api.ArtifactAst;
import org.mule.runtime.ast.internal.serialization.dto.ArtifactAstDTO;
import org.mule.runtime.ast.testobjects.TestArtifactAstFactory;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.Writer;
import java.util.Set;
import java.util.concurrent.Executor;

import com.google.gson.Gson;

import org.apache.commons.io.IOUtils;

import org.junit.Before;
import org.junit.Test;

import io.qameta.allure.Description;
import io.qameta.allure.Feature;
import io.qameta.allure.Story;

@Feature(AST_SERIALIZATION)
@Story(AST_JSON_SERIALIZER)
public class JsonArtifactAstSerializerTestCase {

  private JsonArtifactAstSerializer jsonArtifactAstSerializer;

  private TestArtifactAstFactory testArtifactAstFactory;
  private Set<ExtensionModel> extensionModels;
  private Gson gsonMock;

  @Before
  public void setUp() {
    jsonArtifactAstSerializer = new JsonArtifactAstSerializer(newSingleThreadExecutor());
    testArtifactAstFactory = new TestArtifactAstFactory();
    extensionModels = testArtifactAstFactory.extensionModelsSetForTests();
    gsonMock = mock(Gson.class);
  }

  @Test
  @Description("The JsonArtifactAstSerializer adds a header when serializing an artifact that says its a Json and the serializer version used")
  public void testSerializeReturnsAnInputStreamWithAHeaderOfJsonV100WhenSerializingAnArtifact() throws IOException {
    // Given
    ArtifactAst emptyArtifact =
        testArtifactAstFactory.createArtifactFromXmlFile("test-apps/trivial.xml", extensionModels);

    // When
    InputStream result = jsonArtifactAstSerializer.serialize(emptyArtifact);

    // Then
    String headerString = readHeaderString(result);
    String[] headerArray = headerString.trim().split("#");

    assertThat(headerArray[0], is(JSON));
    assertThat(headerArray[1], is(SERIALIZER_VERSION));
  }

  @Test
  public void testSerializeReturnsAnInputStreamOfATrivialAppJson_WhenArtifactIsTrivial() throws IOException {
    // Given
    ArtifactAst emptyArtifact = testArtifactAstFactory.createArtifactFromXmlFile("test-apps/trivial.xml", extensionModels);

    // When
    InputStream result = jsonArtifactAstSerializer.serialize(emptyArtifact);

    // Then
    readHeaderString(result);
    String jsonString = IOUtils.toString(result, defaultCharset());
    assertThat(jsonString, containsString("{\"artifactName\":\"artifact\",\"artifactType\":\"APPLICATION\",\"dependencies\":["));
    assertThat(jsonString, containsString("{\"name\":\"mule\"}"));
    assertThat(jsonString, containsString("{\"name\":\"tls\"}"));
    assertThat(jsonString, containsString("{\"name\":\"http\"}"));
    assertThat(jsonString, containsString("{\"name\":\"wsc\"}"));
    assertThat(jsonString, containsString("{\"name\":\"file\"}"));
    assertThat(jsonString, containsString("{\"name\":\"os\"}"));
    assertThat(jsonString, containsString("{\"name\":\"db\"}"));
    assertThat(jsonString, containsString("],\"topLevelComponentAsts\":[]"));
    assertThat(jsonString,
               containsString("\"errorTypeRepository\":{\"hierarchy\":[{\"identifier\":\"ANY\",\"namespace\":\"MULE\",\"internal\":false,\"children\":[{\"identifier\":\"EXPRESSION\",\"namespace\":\"MULE\",\"internal\":false,\"children\":[]},{\"identifier\":\"TIMEOUT\",\"namespace\":\"MULE\",\"internal\":false,\"children\":[]},{\"identifier\":\"CONNECTIVITY\",\"namespace\":\"MULE\",\"internal\":false,\"children\":[{\"identifier\":\"RETRY_EXHAUSTED\",\"namespace\":\"MULE\",\"internal\":false,\"children\":[]}]},{\"identifier\":\"ROUTING\",\"namespace\":\"MULE\",\"internal\":false,\"children\":[{\"identifier\":\"COMPOSITE_ROUTING\",\"namespace\":\"MULE\",\"internal\":false,\"children\":[]}]},{\"identifier\":\"STREAM_MAXIMUM_SIZE_EXCEEDED\",\"namespace\":\"MULE\",\"internal\":false,\"children\":[]},{\"identifier\":\"SECURITY\",\"namespace\":\"MULE\",\"internal\":false,\"children\":[{\"identifier\":\"SERVER_SECURITY\",\"namespace\":\"MULE\",\"internal\":false,\"children\":[{\"identifier\":\"NOT_PERMITTED\",\"namespace\":\"MULE\",\"internal\":false,\"children\":[]}]},{\"identifier\":\"CLIENT_SECURITY\",\"namespace\":\"MULE\",\"internal\":false,\"children\":[]}]},{\"identifier\":\"VALIDATION\",\"namespace\":\"MULE\",\"internal\":false,\"children\":[{\"identifier\":\"DUPLICATE_MESSAGE\",\"namespace\":\"MULE\",\"internal\":false,\"children\":[]}]},{\"identifier\":\"TRANSFORMATION\",\"namespace\":\"MULE\",\"internal\":false,\"children\":[]},{\"identifier\":\"REDELIVERY_EXHAUSTED\",\"namespace\":\"MULE\",\"internal\":false,\"children\":[]},{\"identifier\":\"SOURCE\",\"namespace\":\"MULE\",\"internal\":true,\"children\":[{\"identifier\":\"SOURCE_RESPONSE\",\"namespace\":\"MULE\",\"internal\":false,\"children\":[{\"identifier\":\"SOURCE_RESPONSE_GENERATE\",\"namespace\":\"MULE\",\"internal\":false,\"children\":[]},{\"identifier\":\"SOURCE_RESPONSE_SEND\",\"namespace\":\"MULE\",\"internal\":false,\"children\":[]}]},{\"identifier\":\"SOURCE_ERROR_RESPONSE_GENERATE\",\"namespace\":\"MULE\",\"internal\":true,\"children\":[]},{\"identifier\":\"SOURCE_ERROR_RESPONSE_SEND\",\"namespace\":\"MULE\",\"internal\":true,\"children\":[]}]},{\"identifier\":\"UNKNOWN\",\"namespace\":\"MULE\",\"internal\":true,\"children\":[]}]},{\"identifier\":\"CRITICAL\",\"namespace\":\"MULE\",\"internal\":true,\"children\":[{\"identifier\":\"FATAL_JVM_ERROR\",\"namespace\":\"MULE\",\"internal\":true,\"children\":[]},{\"identifier\":\"OVERLOAD\",\"namespace\":\"MULE\",\"internal\":true,\"children\":[{\"identifier\":\"FLOW_BACK_PRESSURE\",\"namespace\":\"MULE\",\"internal\":true,\"children\":[]}]}]}]"));
    assertThat(jsonString, containsString("]},\"importedResources\":[]}"));
  }

  @Test
  public void testSerializeCallsGsonWithArtifactAstDTO_WhenSerializingArtifactAst() throws IOException {
    // Given
    this.jsonArtifactAstSerializer = new JsonArtifactAstSerializer(gsonMock, newSingleThreadExecutor());
    ArtifactAst trivialArtifact =
        testArtifactAstFactory.createArtifactFromXmlFile("test-apps/trivial.xml", extensionModels);
    when(gsonMock.toJson(any(ArtifactAstDTO.class)))
        .thenReturn("{\"dependenciesDTOs\":[{\"name\":\"mule\"}],\"topLevelComponentAsts\":[],\"parent\":{}}");

    // When
    InputStream serialize = this.jsonArtifactAstSerializer.serialize(trivialArtifact);
    String string = IOUtils.toString(serialize, UTF_8);

    // Then
    assertThat(string, is("JSON#1.0#UTF-8#\n"));
    verify(gsonMock, times(1)).toJson(any(ArtifactAstDTO.class), any(Writer.class));
  }

  @Test
  public void testSerializeCallsExecutorWithARunnable_WhenSerializingArtifactAst() {
    // Given
    Executor executor = spy(newSingleThreadExecutor());
    this.jsonArtifactAstSerializer = new JsonArtifactAstSerializer(gsonMock, executor);
    ArtifactAst trivialArtifact =
        testArtifactAstFactory.createArtifactFromXmlFile("test-apps/trivial.xml", extensionModels);
    when(gsonMock.toJson(any(ArtifactAstDTO.class)))
        .thenReturn("{\"dependenciesDTOs\":[{\"name\":\"mule\"}],\"topLevelComponentAsts\":[],\"parent\":{}}");

    // When
    this.jsonArtifactAstSerializer.serialize(trivialArtifact);

    // Then
    verify(executor, times(1)).execute(any(Runnable.class));
  }

  // Read the first line only and return it as a string
  private String readHeaderString(InputStream result) throws IOException {
    return new BufferedReader(new InputStreamReader(result)).readLine();
  }

}
