/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.ast.internal.serialization.dto;

import static java.util.Collections.emptyList;
import static java.util.Collections.unmodifiableList;
import static java.util.Optional.ofNullable;
import static java.util.OptionalInt.empty;
import static java.util.OptionalInt.of;
import static java.util.stream.Collectors.toList;

import org.mule.runtime.ast.api.ComponentMetadataAst;
import org.mule.runtime.ast.api.ImportedResource;
import org.mule.runtime.ast.internal.builder.PropertiesResolver;

import java.net.URI;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.OptionalInt;
import java.util.stream.Collectors;

/**
 * This is a serializable form of a {@link ComponentMetadataAst}
 */
public class ComponentMetadataAstDTO implements ComponentMetadataAst {

  private final Map<String, String> docAttributes;
  private final Integer endColumn;
  private final Integer endLine;
  private final String fileName;
  private final URI fileUri;
  private final List<String> importLocationChain;
  private final ParserAttributesDTO parserAttributes;
  private final String sourceCode;
  private final Integer startColumn;
  private final Integer startLine;

  private transient List<ImportedResourceDTO> importChain;
  private transient PropertiesResolver propertiesResolver;

  /**
   * Cached resolution of document attributes. Even though the docAttributes are immutable, the property resolver mapping function
   * is not, so we need to invalidate if it changes.
   */
  private transient Map<String, String> resolvedDocAttributes = null;

  public ComponentMetadataAstDTO(Map<String, String> docAttributes, Integer endColumn, Integer endLine,
                                 String fileName, URI fileUri, List<String> importLocationChain,
                                 Map<String, Object> parserAttributes, String sourceCode,
                                 Integer startColumn, Integer startLine) {
    this.docAttributes = docAttributes;
    this.endColumn = endColumn;
    this.endLine = endLine;
    this.fileName = fileName;
    this.fileUri = fileUri;
    this.importLocationChain = importLocationChain;
    this.parserAttributes = new ParserAttributesDTO(parserAttributes);
    this.sourceCode = sourceCode;
    this.startColumn = startColumn;
    this.startLine = startLine;
  }

  ComponentMetadataAstDTO() {
    this(null, null, null, null, null, emptyList(), null, null, null, null);
  }

  @Override
  public Optional<String> getFileName() {
    return ofNullable(this.fileName);
  }

  @Override
  public Optional<URI> getFileUri() {
    return ofNullable(this.fileUri);
  }

  @Override
  public List<ImportedResource> getImportChain() {
    // TODO MULE-19734 remove this unmodifiableList
    return unmodifiableList(this.importChain);
  }

  @Override
  public OptionalInt getStartLine() {
    return getOptionalInt(this.startLine);
  }

  @Override
  public OptionalInt getStartColumn() {
    return getOptionalInt(this.startColumn);
  }

  @Override
  public OptionalInt getEndLine() {
    return getOptionalInt(this.endLine);
  }

  @Override
  public OptionalInt getEndColumn() {
    return getOptionalInt(this.endColumn);
  }

  @Override
  public Optional<String> getSourceCode() {
    return ofNullable(this.sourceCode);
  }

  @Override
  public Map<String, String> getDocAttributes() {
    if (resolvedDocAttributes == null) {
      resolvedDocAttributes = computeResolvedDocAttributes();
    }
    return resolvedDocAttributes;
  }

  @Override
  public Map<String, Object> getParserAttributes() {
    return this.parserAttributes.get();
  }

  private OptionalInt getOptionalInt(Integer integer) {
    return integer != null ? of(integer) : empty();
  }

  public void enrich(Map<String, ImportedResourceDTO> importResourcesByRawLocation) {
    importChain = importLocationChain.stream()
        .map(importResourcesByRawLocation::get)
        .collect(toList());
  }

  public void setPropertiesResolver(PropertiesResolver propertiesResolver) {
    // Invalidates any cached resolved attributes.
    invalidateResolvedDocAttributes();

    this.propertiesResolver = propertiesResolver;

    // Cascades the resolver to the import chain.
    this.importChain.forEach(imp -> imp.setPropertiesResolver(this.propertiesResolver));

    // If necessary, registers a callback for invalidating the resolved document attributes if the mapping function changes.
    if (docAttributes != null && docAttributes.values().stream().anyMatch(attr -> attr.contains("${"))) {
      this.propertiesResolver.onMappingFunctionChanged(this::invalidateResolvedDocAttributes);
    }
  }

  private Map<String, String> computeResolvedDocAttributes() {
    if (propertiesResolver == null || docAttributes == null) {
      return docAttributes;
    }

    return docAttributes.entrySet()
        .stream()
        .collect(Collectors.toMap(Map.Entry::getKey,
                                  entry -> propertiesResolver.apply(entry.getValue())));
  }

  private void invalidateResolvedDocAttributes() {
    resolvedDocAttributes = null;
  }

}
