/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.ast.internal.serialization;

import static java.nio.charset.Charset.forName;
import static java.nio.charset.StandardCharsets.UTF_8;
import static java.util.Objects.requireNonNull;
import static java.util.stream.Collectors.toList;

import static org.apache.commons.io.IOUtils.toInputStream;
import static org.slf4j.LoggerFactory.getLogger;

import java.io.IOException;
import java.io.InputStream;
import java.io.SequenceInputStream;
import java.nio.charset.Charset;
import java.util.stream.Stream;

import org.slf4j.Logger;

/**
 * This class deals with the adding/reading into/from an input stream the metadata associated to
 * {@link org.mule.runtime.ast.api.ArtifactAst} serialization. This metadata is generated during the artifact's serialization and
 * must be added to the resulting {@link InputStream} so that it can be then read in order to use the correct deserializing
 * implementation when deserializing.
 * <p>
 * The metadata is added to the stream in the form of a header string in the first line of the {@link InputStream}. This header
 * string is composed by the serializer's id and the version, delimited by a pound sign (#). The max allowed length of the header
 * string is {@value MAX_HEADER_LINE_LENGTH} as determined by the {@link #MAX_HEADER_LINE_LENGTH} constant.
 */
public class ArtifactAstSerializerMetadataSerializer {

  private static final Logger LOGGER = getLogger(ArtifactAstSerializerMetadataSerializer.class);

  // The delimiter used here does not depend on the OS. It will be a '\n' regardless to the OS on which it was written.
  public static final char END_HEADER_DELIMITER = '\n';

  // Max header size + possible '/n'
  private static final int MAX_HEADER_LINE_LENGTH = 1001;

  public static final String DELIMITER = "#";

  /**
   * This method reads the metadata from an {@link InputStream} by reading the first line which should be a header string in a
   * serialized {@link org.mule.runtime.ast.api.ArtifactAst}). This header string is composed by the serializer's id and the
   * version, delimited by a pound sign (#).
   * <p>
   * The max allowed length of the header string is {@value MAX_HEADER_LINE_LENGTH} as determined by the
   * {@link #MAX_HEADER_LINE_LENGTH} constant. Since this method only read the first line, the rest of the input stream will be
   * still available for consumption.
   *
   * @param inputStream a serialized {@link org.mule.runtime.ast.api.ArtifactAst} generated by an
   *                    {@link org.mule.runtime.ast.api.serialization.ArtifactAstSerializer}
   * @return a {@link ArtifactAstSerializerMetadata} instance containing the id and version indicated in the input stream's
   *         metadata.
   * @throws IOException If an I/O error occurs
   */
  public ArtifactAstSerializerMetadata readArtifactAstSerializerMetadataFromInputStream(InputStream inputStream)
      throws IOException {
    String headerString = this.getHeaderString(inputStream);

    // The expected format is: ID#VERSION# -> Beginning + (a string with no # followed by a #) * twice
    // I could just as well validate the format of a serializer id or a version. TBD
    String[] split = headerString.split(DELIMITER);

    if (split.length < 3) {
      throw new IllegalArgumentException("The serialized artifact input stream has an invalid header: " + headerString);
    }
    String serializerVersion = split[1];
    if (split.length > 3) {
      LOGGER.warn("Unrecognized extra header fields: '" + Stream.of(split).skip(3).collect(toList())
          + "'. Maybe it was generated by a newer serializer version (" + serializerVersion + ")?");
    }

    return new ArtifactAstSerializerMetadata(split[0], serializerVersion, getCharsetForName(split[2]));
  }

  private String getHeaderString(InputStream inputStream) throws IOException {
    char[] characters = new char[MAX_HEADER_LINE_LENGTH];

    int index = 0;
    int intC;
    while ((intC = inputStream.read()) != -1 && index < MAX_HEADER_LINE_LENGTH) {
      char c = (char) intC;

      if (c == END_HEADER_DELIMITER) {
        break;
      }

      characters[index] = c;
      index++;
    }

    return new String(characters, 0, index);
  }

  private Charset getCharsetForName(String charsetName) {
    try {
      return forName(charsetName);
    } catch (Exception e) {
      throw new IllegalArgumentException("The serialized artifact input stream has an invalid header: "
          + e.getClass().getSimpleName() + ": " + e.getMessage(), e);
    }
  }

  /**
   * This method serializes the {@link ArtifactAstSerializerMetadata} into a header string and adds it as a first line to the
   * provided serialized {@link org.mule.runtime.ast.api.ArtifactAst} which should be an {@link InputStream}
   * 
   * @param inputStream                   the serialized {@link org.mule.runtime.ast.api.ArtifactAst} created with an
   *                                      {@link org.mule.runtime.ast.api.serialization.ArtifactAstSerializer}
   * @param artifactAstSerializerMetadata the {@link ArtifactAstSerializerMetadata} that represents the information associated
   *                                      with the {@link org.mule.runtime.ast.api.serialization.ArtifactAstSerializer} used to
   *                                      generate the given {@link InputStream}
   * @return an {@link InputStream} whose first line is the serialized {@link ArtifactAstSerializerMetadata} and is followed by
   *         the contents of the given serialized {@link org.mule.runtime.ast.api.ArtifactAst}'s {@link InputStream}
   */
  public InputStream addArtifactAstSerializerMetadataToInputStream(InputStream inputStream,
                                                                   ArtifactAstSerializerMetadata artifactAstSerializerMetadata) {
    requireNonNull(inputStream, "inputStream");
    requireNonNull(artifactAstSerializerMetadata, "artifactAstSerializerMetadata");

    String headerString = artifactAstSerializerMetadata.getSerializerId() + DELIMITER
        + artifactAstSerializerMetadata.getSerializerVersion() + DELIMITER
        + artifactAstSerializerMetadata.getCharset().name() + DELIMITER
        + END_HEADER_DELIMITER;
    InputStream headerInputStream = toInputStream(headerString, UTF_8);
    return new SequenceInputStream(headerInputStream, inputStream);
  }
}
