/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.ast.internal.serialization.dto;

import static org.mule.runtime.api.functional.Either.empty;
import static org.mule.runtime.api.functional.Either.left;
import static org.mule.runtime.api.functional.Either.right;
import static org.mule.runtime.api.meta.ExpressionSupport.REQUIRED;
import static org.mule.runtime.api.meta.ExpressionSupport.SUPPORTED;
import static org.mule.runtime.api.meta.model.parameter.ParameterRole.BEHAVIOUR;
import static org.mule.runtime.ast.internal.builder.adapter.MetadataTypeModelAdapter.createSimpleWrapperTypeModelAdapter;
import static org.mule.runtime.extension.api.util.ExtensionMetadataTypeUtils.isMap;
import static org.mule.runtime.internal.dsl.DslConstants.KEY_ATTRIBUTE_NAME;
import static org.mule.runtime.internal.dsl.DslConstants.NAME_ATTRIBUTE_NAME;
import static org.mule.runtime.internal.dsl.DslConstants.VALUE_ATTRIBUTE_NAME;

import static java.util.Collections.emptyList;
import static java.util.Collections.emptySet;
import static java.util.Optional.ofNullable;

import static org.slf4j.LoggerFactory.getLogger;

import org.mule.metadata.api.ClassTypeLoader;
import org.mule.metadata.api.annotation.EnumAnnotation;
import org.mule.metadata.api.model.MetadataType;
import org.mule.metadata.api.model.ObjectType;
import org.mule.metadata.api.model.SimpleType;
import org.mule.runtime.api.functional.Either;
import org.mule.runtime.api.meta.model.ModelProperty;
import org.mule.runtime.api.meta.model.parameter.ParameterGroupModel;
import org.mule.runtime.api.meta.model.parameter.ParameterModel;
import org.mule.runtime.api.meta.model.parameter.ParameterizedModel;
import org.mule.runtime.ast.api.ComponentGenerationInformation;
import org.mule.runtime.ast.api.ComponentMetadataAst;
import org.mule.runtime.ast.api.ComponentParameterAst;
import org.mule.runtime.ast.api.ParameterResolutionException;
import org.mule.runtime.ast.api.serialization.ExtensionModelResolver;
import org.mule.runtime.ast.internal.builder.PropertiesResolver;
import org.mule.runtime.ast.internal.builder.adapter.MetadataTypeModelAdapter;
import org.mule.runtime.ast.internal.model.ExtensionModelHelper;
import org.mule.runtime.ast.internal.model.RawParameterModel;
import org.mule.runtime.ast.internal.serialization.resolver.GenerationInformationResolver;
import org.mule.runtime.ast.internal.serialization.visitor.ArrayMetadataVisitor;
import org.mule.runtime.ast.internal.serialization.visitor.SetComponentAstDTOTypeMetadataTypeVisitor;
import org.mule.runtime.ast.internal.serialization.visitor.SimpleTypeComponentMetadataTypeVisitor;
import org.mule.runtime.ast.internal.serialization.visitor.UnionTypesVisitor;
import org.mule.runtime.extension.api.declaration.type.ExtensionsTypeLoaderFactory;
import org.mule.runtime.extension.api.model.parameter.ImmutableParameterModel;
import org.mule.runtime.internal.dsl.DslConstants;

import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.atomic.AtomicBoolean;

import org.slf4j.Logger;

/**
 * This is a serializable form of a {@link ComponentParameterAst}. A notable difference is that the values are stored as a
 * {@link ParameterValueContainer} in order to prevent type inference problems when serializing.
 */
public class ComponentParameterAstDTO implements ComponentParameterAst {

  private static final Logger LOGGER = getLogger(ComponentParameterAstDTO.class);

  private static final String DEFAULT_EXPRESSION_PREFIX = "#[";
  private static final String DEFAULT_EXPRESSION_SUFFIX = "]";

  private static final Class<? extends ModelProperty> allowsExpressionWithoutMarkersModelPropertyClass;

  static {
    Class<? extends ModelProperty> foundClass = null;
    try {
      foundClass = (Class<? extends ModelProperty>) Class
          .forName("org.mule.runtime.module.extension.api.loader.java.property.AllowsExpressionWithoutMarkersModelProperty");
    } catch (ClassNotFoundException | SecurityException e) {
      // No custom location processing
    }
    allowsExpressionWithoutMarkersModelPropertyClass = foundClass;
  }

  private final String unresolvedValue;
  private final ComponentMetadataAstDTO metadata;
  private final String groupModelName;
  private final String modelName;

  private ParameterValueContainer value;
  private final boolean defaultValue;
  private ComponentGenerationInformationDTO generationInformation;

  private transient String resolvedRawValue;
  private transient ParameterModel model;
  private transient ParameterGroupModel groupModel;
  private transient PropertiesResolver propertiesResolver;

  public ComponentParameterAstDTO(ParameterValueContainer value, String unresolvedValue, boolean defaultValue,
                                  ComponentMetadataAstDTO metadata, String groupModelName, String modelName) {
    this.value = value;
    this.unresolvedValue = unresolvedValue;
    this.defaultValue = defaultValue;
    this.metadata = metadata;
    this.groupModelName = groupModelName;
    this.modelName = modelName;
  }

  @Override
  public ParameterModel getModel() {
    return model;
  }

  @Override
  public ParameterGroupModel getGroupModel() {
    return groupModel;
  }

  @Override
  public <T> Either<String, T> getValue() {

    if (value != null && value.isFixedValue()) {
      if (value.getContainedValue() instanceof String) {
        this.resolvedRawValue = (String) value.getContainedValue();
      }

      return rightValue(value.getContainedValue());
    }

    if (this.value != null && value.isExpression()) {
      return left(value.getExpression());
    }

    // Value is null
    if (unresolvedValue != null) {
      if (isAnExpression()) {
        if (propertiesResolver != null) {
          this.resolvedRawValue = this.propertiesResolver.apply(unresolvedValue);
        } else {
          this.resolvedRawValue = unresolvedValue;
        }

        this.value = new ParameterValueContainer(extractExpression(resolvedRawValue), null);
        return left(value.getExpression());
      } else {
        // need to account for this being called before the propertiesResolver is set, for enriching the importChains
        if (propertiesResolver == null) {
          return empty();
        }

        // Right value with properties inside
        this.resolvedRawValue = this.propertiesResolver.apply(unresolvedValue);
        this.value = new ParameterValueContainer(null, resolvedRawValue);
      }

      return rightValue(value.getContainedValue());
    }

    // Value and raw value were null
    if (this.model == null) {
      throw new IllegalStateException("Cannot resolve value for parameter without model");
    }

    final Object defaultValue = this.model.getDefaultValue();

    if (defaultValue == null) {
      this.value = null;
      return empty();
    }

    // Has default value
    if (defaultValue instanceof String) {
      this.resolvedRawValue = this.propertiesResolver.apply(defaultValue.toString());
      this.value = new ParameterValueContainer(null, resolvedRawValue);
      return rightValue(value.getContainedValue());
    }

    if (this.model.getType().getAnnotation(EnumAnnotation.class).isPresent()) {
      this.resolvedRawValue = ((Enum<?>) defaultValue).name();
      this.value = new ParameterValueContainer(null, resolvedRawValue);
      return rightValue(value.getContainedValue());
    }

    this.value = new ParameterValueContainer(null, defaultValue);
    return rightValue(defaultValue);
  }

  @Override
  public <T> Either<String, Either<ParameterResolutionException, T>> getValueOrResolutionError() {
    try {
      return getValue().mapRight(fixedValue -> right((T) fixedValue));
    } catch (ParameterResolutionException e) {
      return right(left(e));
    }
  }

  private boolean isAnExpression() {
    return this.model != null
        && (SUPPORTED.equals(this.model.getExpressionSupport()) || REQUIRED.equals(this.model.getExpressionSupport()))
        && isExpression(unresolvedValue);
  }

  private boolean isExpression(Object value) {
    if (value instanceof String) {
      String trim = ((String) value).trim();

      if (trim.startsWith(DEFAULT_EXPRESSION_PREFIX) && trim.endsWith(DEFAULT_EXPRESSION_SUFFIX)) {
        return true;
      }

      return allowsExpressionWithoutMarkersModelPropertyClass != null
          && getModel().getModelProperty(allowsExpressionWithoutMarkersModelPropertyClass).isPresent();
    } else {
      return false;
    }
  }

  /**
   * Parse the given value and remove expression markers if it is considered as an expression.
   *
   * @param expression Value to parse
   * @return a String containing the expression without markers or null if the value is not an expression.
   */
  public String extractExpression(String expression) {
    String trimmedText = expression.trim();

    if (trimmedText.startsWith(DEFAULT_EXPRESSION_PREFIX) && trimmedText.endsWith(DEFAULT_EXPRESSION_SUFFIX)) {
      return trimmedText.substring(DEFAULT_EXPRESSION_PREFIX.length(), trimmedText.length() - DEFAULT_EXPRESSION_SUFFIX.length());
    } else {
      return trimmedText;
    }
  }

  private <T> Either<String, T> rightValue(Object value) {
    return (Either<String, T>) right(String.class, value);
  }

  @Override
  public String getRawValue() {
    return unresolvedValue;
  }

  @Override
  public String getResolvedRawValue() {
    if (this.resolvedRawValue == null) {
      // Try to init values
      this.getValue();
    }

    if (this.resolvedRawValue == null) {
      this.resolvedRawValue = unresolvedValue;
    }

    return resolvedRawValue;
  }

  @Override
  public Optional<ComponentMetadataAst> getMetadata() {
    return ofNullable(metadata);
  }

  @Override
  public ComponentGenerationInformation getGenerationInformation() {
    return generationInformation;
  }

  @Override
  public boolean isDefaultValue() {
    return defaultValue;
  }

  public void setModel(ParameterModel parameterModel, ParameterGroupModel parameterGroupModel) {
    this.model = parameterModel;
    this.groupModel = parameterGroupModel;
  }

  public void setGenerationInformation(ComponentGenerationInformationDTO generationInformation) {
    this.generationInformation = generationInformation;
  }

  public String getGroupModelName() {
    return groupModelName;
  }

  public String getModelName() {
    return modelName;
  }

  /**
   * Setting a properties resolver should not only impact this instance. It should cascade to possible
   * {@link org.mule.runtime.ast.api.ComponentAst}s that might be the value of this parameter.
   *
   * @param propertiesResolver
   */
  public void setPropertiesResolver(PropertiesResolver propertiesResolver) {
    this.propertiesResolver = propertiesResolver;

    if (this.value != null && this.value.isFixedValue() && this.isValueAListOfComponentAstDTOs()) {
      List<ComponentAstDTO> componentAstDTOS = (List<ComponentAstDTO>) this.value.getContainedValue();
      componentAstDTOS.forEach(componentAstDTO -> componentAstDTO.setPropertiesResolver(propertiesResolver));
    }

    if (this.value != null && this.value.isFixedValue() && this.isValueAComponentAstDTO()) {
      ComponentAstDTO componentAstDTO = (ComponentAstDTO) this.value.getContainedValue();
      componentAstDTO.setPropertiesResolver(propertiesResolver);
    }

    // Listen for any update on property resolution triggered by calling ArtifactAst.updatePropertiesResolver
    propertiesResolver.onMappingFunctionChanged(() -> {
      if (this.unresolvedValue != null && this.unresolvedValue.contains("${")) {
        this.resolvedRawValue = null;
        this.value = null;
      }
    });
  }

  public PropertiesResolver getPropertiesResolver() {
    return propertiesResolver;
  }

  @Override
  public String toString() {
    return "ComponentParameterAstDTO{" + "groupModelName='" + groupModelName + "', modelName='" + modelName + "'" + ", value='"
        + value + ", unresolvedValue='" + unresolvedValue + "'}";
  }

  private void resolveMapEntriesElementsModels(ComponentAstDTO parentComponentAstDTO, ExtensionModelHelper extensionModelHelper,
                                               ExtensionModelResolver extensionModelResolver,
                                               GenerationInformationResolver generationInformationResolver) {
    List<ComponentAstDTO> componentAstDTOList = getValueAsComponentAstDTOList();

    componentAstDTOList.forEach(componentAstDTO -> {
      componentAstDTO.resolveMapEntryComponentModels(parentComponentAstDTO, this, extensionModelHelper,
                                                     extensionModelResolver, generationInformationResolver);

      // In this case, a direct children of a map entry would be a complex value
      componentAstDTO.directChildrenStream().map(childComponentAst -> (ComponentAstDTO) childComponentAst)
          .forEach(childComponentAstDTO -> childComponentAstDTO.resolveModelsRecursively(componentAstDTO, extensionModelHelper,
                                                                                         extensionModelResolver,
                                                                                         generationInformationResolver));
    });
  }

  public void resolveListElementModels(ExtensionModelHelper extensionModelHelper, ExtensionModelResolver extensionModelResolver,
                                       GenerationInformationResolver generationInformationResolver) {
    this.model.getType().accept(new ArrayMetadataVisitor(new SimpleTypeComponentMetadataTypeVisitor(this,
                                                                                                    extensionModelHelper,
                                                                                                    generationInformationResolver,
                                                                                                    extensionModelResolver)));
  }

  /**
   * If a {@link org.mule.runtime.ast.internal.serialization.dto.ComponentParameterAstDTO} represents a list of strings or other
   * scalar types then its value will be a list of {@link ComponentAstDTO} that have parameters named
   * '{@value DslConstants#VALUE_ATTRIBUTE_NAME}'
   *
   * @return whether it is a list of simple values or not
   */
  private boolean isListOfSimpleTypeValuesAsComponent() {
    return this.isValueAListOfComponentAstDTOs()
        && this.getValueAsComponentAstDTOList().stream().allMatch(componentAstDTO -> componentAstDTO.isSimpleTypeComponent());
  }

  public ParameterModel resolveParameterModelFromOwnerMetadataTypeModelAdapter(MetadataTypeModelAdapter metadataTypeModelAdapter) {
    return metadataTypeModelAdapter.getAllParameterModels().stream()
        .filter(parameterModel -> parameterModel.getName().equals(getModelName())).findFirst()
        .orElse(new RawParameterModel(modelName));
  }

  void resolveModelParameter(ComponentAstDTO ownerComponentAstDTO, ExtensionModelHelper extensionModelHelper,
                             ExtensionModelResolver extensionModelResolver,
                             GenerationInformationResolver generationInformationResolver) {
    LOGGER.debug("Enrichment: resolveModelParameter({}, {})", ownerComponentAstDTO, this);

    this.model = ownerComponentAstDTO.resolveParameterModel(this);
    this.groupModel = ownerComponentAstDTO.resolveParameterGroupModel(this);

    LOGGER.debug("Enrichment: resolveModelParameter, resolved groupModel and model: {}, {}", groupModel, model);

    AtomicBoolean paramProcessed = new AtomicBoolean();

    // This is needed for reconnection strategies
    if (model.getType() != null) {
      model.getType().accept(new UnionTypesVisitor(this, extensionModelHelper, paramProcessed));
    }

    if (ownerComponentAstDTO.isWrapped(this, extensionModelHelper)) {
      LOGGER.debug("Enrichment: resolveModelParameter, isWrapped {}", ownerComponentAstDTO, this);
      this.resolveParameterValueModel(ownerComponentAstDTO, extensionModelHelper, extensionModelResolver,
                                      generationInformationResolver);

      if (!paramProcessed.get()) {
        this.generationInformation = generationInformationResolver
            .resolveWrappedComponentParameterAstGenerationInformation(this, ownerComponentAstDTO, extensionModelHelper);
        LOGGER.debug("Enrichment: resolveModelParameter, generationInformation: {}",
                     this.generationInformation);
      }
    } else {
      if (!paramProcessed.get()) {
        this.generationInformation = generationInformationResolver
            .resolveRegularComponentParameterAstGenerationInformation(this, ownerComponentAstDTO, extensionModelHelper);
        LOGGER.debug("Enrichment: resolveModelParameter, generationInformation: {}",
                     this.generationInformation);
      }

      this.resolveParameterValueModel(ownerComponentAstDTO, extensionModelHelper, extensionModelResolver,
                                      generationInformationResolver);
    }

  }

  public boolean isValueAComponentAstDTO() {
    return this.getValue() != null && this.getValue().isRight() && getValue().getRight() instanceof ComponentAstDTO;

  }

  /**
   * A value in a parameter is represented with an {@link Either} and it can be a left or a right value of it. A right value might
   * be a list of {@link ComponentAstDTO}s
   *
   *
   * @return Whether it is a {@link List} of {@link ComponentAstDTO}s or not
   */
  public boolean isValueAListOfComponentAstDTOs() {
    return this.getValue() != null
        && this.getValue().isRight()
        && (this.getValue().getRight() instanceof List
            && !((List) this.getValue().getRight()).isEmpty()
            && ((List) this.getValue().getRight()).get(0) instanceof ComponentAstDTO);

  }

  private void resolveParameterValueModel(ComponentAstDTO parentComponentAstDTO, ExtensionModelHelper extensionModelHelper,
                                          ExtensionModelResolver extensionModelResolver,
                                          GenerationInformationResolver generationInformationResolver) {
    if (this.isValueAComponentAstDTO()) {
      ComponentAstDTO valueComponentAstDTO = this.getValueAsComponentAstDTO();
      LOGGER.debug("Enrichment: resolveParameterValueModel, value is ComponentAstDTO, parameter {} value {}", this,
                   valueComponentAstDTO);
      model.getType().accept(new SetComponentAstDTOTypeMetadataTypeVisitor(valueComponentAstDTO));
      valueComponentAstDTO.resolveModelsRecursively(parentComponentAstDTO, extensionModelHelper, extensionModelResolver,
                                                    generationInformationResolver);
    } else if (this.isListOfSimpleTypeValuesAsComponent()) {
      LOGGER.debug("Enrichment: resolveParameterValueModel, value is list of Simple ComponentAstDTO, parameter {}", this);
      this.resolveListElementModels(extensionModelHelper, extensionModelResolver, generationInformationResolver);
    } else if (this.isValueAMap()) {
      LOGGER.debug("Enrichment: resolveParameterValueModel, value is list of Map, parameter {}", this);
      this.resolveMapEntriesElementsModels(parentComponentAstDTO, extensionModelHelper, extensionModelResolver,
                                           generationInformationResolver);
    } else if (this.isValueAListOfComponentAstDTOs()) {
      LOGGER.debug("Enrichment: resolveParameterValueModel, value is list of ComponentAstDTOs, parameter {}", this);
      this.getValueAsComponentAstDTOList().forEach(componentParamValueItem -> componentParamValueItem
          .resolveModelsRecursively(parentComponentAstDTO, extensionModelHelper, extensionModelResolver,
                                    generationInformationResolver));
    }
  }

  void resolveMapEntryAttribute(ParameterModel ownerParameterModel,
                                ComponentGenerationInformation ownerParameterGenerationInformation,
                                ExtensionModelHelper extensionModelHelper, ExtensionModelResolver extensionModelResolver,
                                GenerationInformationResolver generationInformationResolver, ComponentAstDTO componentAstDTO) {
    switch (modelName) {
      case KEY_ATTRIBUTE_NAME:
        ClassTypeLoader typeLoader = ExtensionsTypeLoaderFactory.getDefault().createTypeLoader();

        this.model = new ImmutableParameterModel(KEY_ATTRIBUTE_NAME, "", typeLoader.load(String.class), false, true, false, false,
                                                 SUPPORTED,
                                                 null, BEHAVIOUR, null, null, null, null, emptyList(), emptySet());
        this.groupModel = componentAstDTO.resolveParameterGroupModel(this);

        this.generationInformation = generationInformationResolver
            .resolveGenerationInformationThroughParent(KEY_ATTRIBUTE_NAME, componentAstDTO, ownerParameterModel,
                                                       ownerParameterGenerationInformation);
        break;
      case VALUE_ATTRIBUTE_NAME:

        this.model =
            new ImmutableParameterModel(VALUE_ATTRIBUTE_NAME, "", this.resolveMapEntryValueType(ownerParameterModel), false, true,
                                        false, false,
                                        SUPPORTED, null, BEHAVIOUR, null, null, null, null, emptyList(), emptySet());
        this.groupModel = componentAstDTO.resolveParameterGroupModel(this);

        this.generationInformation = generationInformationResolver
            .resolveGenerationInformationThroughParent(VALUE_ATTRIBUTE_NAME, componentAstDTO, ownerParameterModel,
                                                       ownerParameterGenerationInformation);

        this.resolveParameterValueModel(componentAstDTO, extensionModelHelper, extensionModelResolver,
                                        generationInformationResolver);
        break;
      case NAME_ATTRIBUTE_NAME:
        this.model = new RawParameterModel(getModelName());
        this.groupModel = componentAstDTO.resolveParameterGroupModel(this);

        this.generationInformation = generationInformationResolver
            .resolveComponentParameterAstGenerationInformation(this, componentAstDTO, extensionModelHelper);
        break;
    }
  }

  private MetadataType resolveMapEntryValueType(ParameterModel ownerParameterModel) {
    if (ownerParameterModel.getType() instanceof ObjectType
        && ((ObjectType) ownerParameterModel.getType()).getOpenRestriction().isPresent()) {
      return ((ObjectType) ownerParameterModel.getType()).getOpenRestriction().get();
    }

    return ownerParameterModel.getType();
  }

  private ComponentAstDTO getValueAsComponentAstDTO() {
    return (ComponentAstDTO) getValue().getRight();
  }

  private List<ComponentAstDTO> getValueAsComponentAstDTOList() {
    return (List<ComponentAstDTO>) getValue().getRight();
  }

  private boolean isValueAMap() {
    return this.isValueAListOfComponentAstDTOs() && isMap(model.getType());
  }

  void resolveInfrastructureParameter(ComponentAstDTO ownerComponentAstDTO, ExtensionModelHelper extensionModelHelper,
                                      GenerationInformationResolver generationInformationResolver) {
    LOGGER.debug("Enrichment: resolveInfrastructureParameter: {} - {}", ownerComponentAstDTO, this);
    ParameterizedModel parameterizedModel = ownerComponentAstDTO.getModel(ParameterizedModel.class).get();

    this.model = this.resolveParameterModelFromOwnerMetadataTypeModelAdapter((MetadataTypeModelAdapter) parameterizedModel);
    this.groupModel = parameterizedModel.getParameterGroupModels().stream()
        .filter(parameterModel -> parameterModel.getName().equals(getGroupModelName())).findFirst()
        .orElse(null);

    AtomicBoolean paramProcessed = new AtomicBoolean();

    // This is needed for reconnection strategies
    this.model.getType().accept(new UnionTypesVisitor(this, extensionModelHelper, paramProcessed));

    if (!paramProcessed.get()) {
      this.generationInformation =
          generationInformationResolver.crateComponentGenerationInformationFromParent(ownerComponentAstDTO, this);
    }
  }

  void enrichWithImportedResources(Map<String, ImportedResourceDTO> importResourcesByRawLocation) {
    LOGGER.debug("Enrichment: enrichWithImportedResources: {}", this);
    if (metadata != null) {
      metadata.enrich(importResourcesByRawLocation);
    }

    if (this.isValueAComponentAstDTO()) {
      this.getValueAsComponentAstDTO().enrichWithImportedResources(importResourcesByRawLocation);
      return;
    }

    if (this.isValueAListOfComponentAstDTOs()) {
      this.getValueAsComponentAstDTOList()
          .forEach(componentAstDTO -> componentAstDTO.enrichWithImportedResources(importResourcesByRawLocation));
      return;
    }
  }

  void resolveSimpleTypeComponentParameter(ComponentAstDTO ownerComponentAstDTO, SimpleType simpleType,
                                           ExtensionModelHelper extensionModelHelper,
                                           GenerationInformationResolver generationInformationResolver) {
    this.model =
        resolveParameterModelFromOwnerMetadataTypeModelAdapter(createSimpleWrapperTypeModelAdapter(simpleType,
                                                                                                   extensionModelHelper));
    this.groupModel = ownerComponentAstDTO.resolveParameterGroupModel(this);
    this.generationInformation = generationInformationResolver
        .resolveComponentParameterAstGenerationInformation(this, ownerComponentAstDTO, extensionModelHelper);
  }
}
