/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.ast.internal.serialization.dto.factory;

import static java.util.Optional.ofNullable;

import org.mule.runtime.api.exception.MuleRuntimeException;
import org.mule.runtime.api.meta.NamedObject;
import org.mule.runtime.ast.api.ComponentParameterAst;
import org.mule.runtime.ast.internal.serialization.dto.ComponentParameterAstDTO;
import org.mule.runtime.ast.internal.serialization.dto.ParameterValueContainer;

public class ComponentParameterAstDTOFactory {

  private final AstDTOFactoryProvider astDTOFactoryProvider;
  private final ParameterValueContainerFactory parameterValueContainerFactory;

  public ComponentParameterAstDTOFactory(AstDTOFactoryProvider astDTOFactoryProvider) {
    this.astDTOFactoryProvider = astDTOFactoryProvider;
    this.parameterValueContainerFactory = new ParameterValueContainerFactory(astDTOFactoryProvider);
  }

  public ComponentParameterAstDTO from(ComponentParameterAst componentParameterAst) {
    return new ComponentParameterAstDTO(getParameterValueContainer(componentParameterAst), componentParameterAst.getRawValue(),
                                        getDefaultValue(componentParameterAst),
                                        componentParameterAst.getMetadata()
                                            .map(metadata -> astDTOFactoryProvider.getComponentMetadataAstDTOFactory()
                                                .from(metadata))
                                            .orElse(null),
                                        getGroupModelName(componentParameterAst), componentParameterAst.getModel().getName());
  }

  private String getGroupModelName(ComponentParameterAst componentParameterAst) {
    return ofNullable(componentParameterAst.getGroupModel()).map(NamedObject::getName).orElse(null);
  }

  /**
   * Gets whether a parameter's value is the default value
   * 
   * @param componentParameterAst a parameter
   * @return whether it's value is the default value
   */
  private boolean getDefaultValue(ComponentParameterAst componentParameterAst) {
    try {
      return componentParameterAst.isDefaultValue();
    } catch (MuleRuntimeException e) {
      // This happens when the value requires property resolution and there is no PropertyResolver yet.
      return false;
    }
  }

  /**
   * Makes a {@link ParameterValueContainer} with the {@link ComponentParameterAst}'s value
   * 
   * @param componentParameterAst a parameter
   * @return The {@link ParameterValueContainer} with the parameter's value or null if it needs property resolution and it could
   *         not be resolved
   */
  private ParameterValueContainer getParameterValueContainer(ComponentParameterAst componentParameterAst) {
    try {
      return parameterValueContainerFactory.from(componentParameterAst);
    } catch (MuleRuntimeException e) {
      // This happens when the value requires property resolution and there is no PropertyResolver yet.
      return null;
    }
  }

}
