/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.ast.api.serialization;

import org.mule.api.annotation.NoImplement;
import org.mule.runtime.api.meta.model.ExtensionModel;
import org.mule.runtime.ast.api.ArtifactAst;
import java.io.IOException;
import java.io.InputStream;

/**
 * Defines a component capable of deserializing an {@link InputStream} created using an {@link ArtifactAstSerializer} into an
 * {@link ArtifactAst}.
 * <p>
 * The content of the {@link InputStream} might have been generated in different ways, depending on the
 * {@link ArtifactAstSerializer} used, each requiring a specific deserialization process to be determined by the implementation.
 * If the deserializer cannot handle this input stream, an {@link IllegalArgumentException} will be thrown.
 * <p>
 * {@link ExtensionModel} related information must be provided by the client code. This means that in order to deserialize, such
 * information is required and must be presented as an input of the deserialization method.
 *
 * @since 1.0
 */
@NoImplement
public interface ArtifactAstDeserializer {

  /**
   * Given an {@link InputStream} created by serializing an {@link ArtifactAst} with an {@link ArtifactAstSerializer} and an
   * {@link ExtensionModelResolver} able to provide the required {@link ExtensionModel}s, this method deserializes the
   * {@link InputStream} into an {@link ArtifactAst}.
   * 
   * @param artifactAstInputStream the input stream generated by an {@link ArtifactAstSerializer}
   * @param extensionModelResolver an {@link ExtensionModelResolver} able to provide the {@link ExtensionModel}s required by this
   *                               serialized {@link ArtifactAst}
   * @return The artifact
   * @throws IllegalArgumentException if this deserializer cannot perform a deserialization with the given {@link InputStream}
   *                                  and/or {@link ExtensionModelResolver}
   * @throws IOException              if the input stream could not be read
   */
  ArtifactAst deserialize(InputStream artifactAstInputStream, ExtensionModelResolver extensionModelResolver)
      throws IllegalArgumentException, IOException;

  /**
   * Given an {@link InputStream} created by serializing an {@link ArtifactAst} with an {@link ArtifactAstSerializer} and an
   * {@link ExtensionModelResolver} able to provide the required {@link ExtensionModel}s, this method deserializes the
   * {@link InputStream} into an {@link ArtifactAst}.
   * <p>
   * This method accepts a parent {@link ArtifactAst} as an argument. An {@link ArtifactAst} might have a parent that requires its
   * own {@link ExtensionModel}s in order to be enrich, and furthermore, might be the parent of multiple {@link ArtifactAst}s.
   * That is why the parent needs to be resolved before its children passing it as a parameter with this method.
   *
   * @param artifactAstInputStream the input stream generated by an {@link ArtifactAstSerializer}
   * @param extensionModelResolver an {@link ExtensionModelResolver} able to provide the {@link ExtensionModel}s required by this
   *                               serialized {@link ArtifactAst}
   * @param parent                 the parent {@link ArtifactAst} to be set to the resulting deserialized {@link ArtifactAst}
   * @return The artifact
   * @throws IllegalArgumentException if this deserializer cannot perform a deserialization with the given {@link InputStream}
   *                                  and/or {@link ExtensionModelResolver}
   * @throws IOException              if the input stream could not be read
   */
  ArtifactAst deserialize(InputStream artifactAstInputStream, ExtensionModelResolver extensionModelResolver, ArtifactAst parent)
      throws IllegalArgumentException, IOException;

}
