/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.ast.internal.serialization.dto;

import static java.util.Collections.unmodifiableCollection;
import static java.util.Collections.unmodifiableList;
import static java.util.Collections.unmodifiableSet;
import static java.util.Optional.ofNullable;
import static java.util.function.UnaryOperator.identity;
import static java.util.stream.Collectors.toMap;
import static java.util.stream.Collectors.toSet;

import static org.slf4j.LoggerFactory.getLogger;

import org.mule.runtime.api.exception.ErrorTypeRepository;
import org.mule.runtime.api.meta.model.ExtensionModel;
import org.mule.runtime.ast.api.ArtifactAst;
import org.mule.runtime.ast.api.ArtifactType;
import org.mule.runtime.ast.api.ComponentAst;
import org.mule.runtime.ast.api.ImportedResource;
import org.mule.runtime.ast.api.serialization.ExtensionModelResolver;
import org.mule.runtime.ast.api.util.BaseArtifactAst;
import org.mule.runtime.ast.internal.builder.PropertiesResolver;
import org.mule.runtime.ast.internal.model.ExtensionModelHelper;
import org.mule.runtime.ast.internal.serialization.resolver.GenerationInformationResolver;

import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.function.UnaryOperator;
import org.slf4j.Logger;

/**
 * This class represent an {@link ArtifactAst} but in a serializable fashion. All the data that cannot be serialized can be added
 * after deserialization using {@link #enrich(ExtensionModelResolver, GenerationInformationResolver)}.
 */
public class ArtifactAstDTO extends BaseArtifactAst implements ArtifactAst {

  private static final Logger LOGGER = getLogger(ArtifactAstDTO.class);

  private final String artifactName;
  private final ArtifactType artifactType;

  private final Set<ExtensionModelDTO> dependencies;
  private final List<ComponentAstDTO> topLevelComponentAsts;
  private final ErrorTypeRepositoryDTO errorTypeRepository;
  private final Collection<ImportedResourceDTO> importedResources;

  private transient ArtifactAst parent;
  private transient Set<ExtensionModel> dependenciesTransient = new HashSet<>();
  private transient PropertiesResolver propertiesResolver;

  public ArtifactAstDTO(String artifactName, ArtifactType artifactType,
                        Set<ExtensionModelDTO> dependencies, List<ComponentAstDTO> topLevelComponentAsts,
                        ErrorTypeRepositoryDTO errorTypeRepository, Collection<ImportedResourceDTO> importedResources) {
    this.artifactName = artifactName;
    this.artifactType = artifactType;
    this.dependencies = dependencies;
    this.topLevelComponentAsts = topLevelComponentAsts;
    this.errorTypeRepository = errorTypeRepository;
    this.importedResources = importedResources;
  }

  @Override
  public String getArtifactName() {
    return artifactName;
  }

  @Override
  public ArtifactType getArtifactType() {
    return artifactType;
  }

  @Override
  public Set<ExtensionModel> dependencies() {
    return unmodifiableSet(dependenciesTransient);
  }

  @Override
  public Optional<ArtifactAst> getParent() {
    return ofNullable(parent);
  }

  @Override
  public List<ComponentAst> topLevelComponents() {
    // TODO MULE-19734 remove this unmodifiableList
    return unmodifiableList(this.topLevelComponentAsts);
  }

  @Override
  public void updatePropertiesResolver(UnaryOperator<String> newPropertiesResolver) {
    propertiesResolver.setMappingFunction(newPropertiesResolver);
  }

  @Override
  public ErrorTypeRepository getErrorTypeRepository() {
    return errorTypeRepository;
  }

  @Override
  public Collection<ImportedResource> getImportedResources() {
    // TODO MULE-19734 remove this unmodifiableList
    return unmodifiableCollection(this.importedResources);
  }

  public PropertiesResolver getPropertiesResolver() {
    return propertiesResolver;
  }

  /**
   * Setting a {@link PropertiesResolver} to this DTO will cascade down all child components in order to eventually impact all
   * parameters. The {@link ComponentParameterAstDTO} should listen for changes in this properties resolver triggered by calling
   * {@link #updatePropertiesResolver(UnaryOperator)}.
   *
   * @param propertiesResolver
   */
  public void setPropertiesResolver(PropertiesResolver propertiesResolver) {
    this.propertiesResolver = propertiesResolver;
    this.topLevelComponentAsts.forEach(componentAstDTO -> componentAstDTO.setPropertiesResolver(propertiesResolver));
    this.importedResources.forEach(importedResource -> importedResource.setPropertiesResolver(propertiesResolver));
  }

  public void setParent(ArtifactAst parent) {
    this.parent = parent;
  }

  /**
   * An {@link ArtifactAstDTO} is a serialized version of an {@link ArtifactAst}. During serialization, some info such as the
   * {@link ExtensionModel} is lost. In order to use this DTO interchangeably as an {@link ArtifactAst}, that information needs to
   * be reloaded. This is done through this method (with the help of Resolvers)
   *
   * @param extensionModelResolver
   * @param generationInformationResolver
   */
  public void enrich(ExtensionModelResolver extensionModelResolver, GenerationInformationResolver generationInformationResolver) {
    LOGGER.debug("Enriching importResources...");
    Map<String, ImportedResourceDTO> importResourcesByRawLocation =
        this.importedResources.stream().collect(toMap(ImportedResourceDTO::getRawResourceLocation, identity()));

    this.topLevelComponentAsts
        .forEach(componentAst -> componentAst.enrichWithImportedResources(importResourcesByRawLocation));

    PropertiesResolver propertiesResolver = new PropertiesResolver();
    propertiesResolver.setMappingFunction(identity());
    this.setPropertiesResolver(propertiesResolver);

    LOGGER.debug("Enrichment: resolving dependencies...");
    this.dependenciesTransient =
        this.dependencies.stream().map(dto -> extensionModelResolver.resolve(dto.getName())).collect(toSet());

    ExtensionModelHelper extensionModelHelper = new ExtensionModelHelper(this.dependenciesTransient);

    LOGGER.debug("Enriching models...");
    this.topLevelComponentAsts.stream()
        .filter(topLevelComponent -> topLevelComponent.getExtensionModelDTO() != null)
        .forEach(topLevelComponent -> topLevelComponent
            .resolveModelsRecursively(null, extensionModelHelper, extensionModelResolver, generationInformationResolver));
  }

}
