/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.ast.internal.serialization.dto;

import static java.lang.Boolean.getBoolean;
import static java.util.Optional.empty;
import static java.util.Optional.of;
import static java.util.Optional.ofNullable;
import static java.util.function.UnaryOperator.identity;
import static java.util.stream.Collectors.toList;
import static java.util.stream.Stream.concat;
import static org.mule.runtime.api.util.MuleSystemProperties.MULE_LAX_ERROR_TYPES;
import static org.mule.runtime.api.util.MultiMap.toMultiMap;
import static org.mule.runtime.ast.internal.error.ErrorTypeBuilder.ANY_IDENTIFIER;
import static org.mule.runtime.ast.internal.error.ErrorTypeBuilder.CORE_NAMESPACE_NAME;
import static org.mule.runtime.ast.internal.error.ErrorTypeBuilder.CRITICAL_IDENTIFIER;
import static org.mule.runtime.ast.internal.error.ErrorTypeBuilder.builder;
import static org.slf4j.LoggerFactory.getLogger;

import org.mule.runtime.api.component.ComponentIdentifier;
import org.mule.runtime.api.exception.ErrorTypeRepository;
import org.mule.runtime.api.message.ErrorType;
import org.mule.runtime.api.util.LazyValue;
import org.mule.runtime.api.util.MultiMap;
import org.mule.runtime.ast.internal.serialization.json.gson.PostProcessingEnabler.PostProcessable;

import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.function.Predicate;
import java.util.function.Supplier;
import java.util.stream.Stream;

import org.slf4j.Logger;

/**
 * This is a serializable version of an {@link ErrorTypeRepository}. It is designed to behave as any possible serialized instance,
 * whether it is a default or a composite one.
 */
public class ErrorTypeRepositoryDTO implements ErrorTypeRepository, PostProcessable {

  private static final Logger LOGGER = getLogger(ErrorTypeRepositoryDTO.class);

  private static final ComponentIdentifier ANY_ERROR_IDENTIFIER = ComponentIdentifier.builder()
      .namespace(CORE_NAMESPACE_NAME)
      .name(ANY_IDENTIFIER)
      .build();
  private static final ComponentIdentifier SOURCE_ERROR_IDENTIFIER = ComponentIdentifier.builder()
      .namespace(CORE_NAMESPACE_NAME)
      .name("SOURCE")
      .build();
  private static final ComponentIdentifier SOURCE_RESPONSE_ERROR_IDENTIFIER = ComponentIdentifier.builder()
      .namespace(CORE_NAMESPACE_NAME)
      .name("SOURCE_RESPONSE")
      .build();
  private static final ComponentIdentifier CRITICAL_ERROR_IDENTIFIER = ComponentIdentifier.builder()
      .namespace(CORE_NAMESPACE_NAME)
      .name(CRITICAL_IDENTIFIER)
      .build();

  private final List<ErrorTypeHierarchyItemDTO> hierarchy;

  private transient Supplier<Map<ComponentIdentifier, ErrorType>> errorTypes;
  private transient Supplier<Map<ComponentIdentifier, ErrorType>> internalErrorTypes;

  private transient ErrorType anyErrorType;
  private transient ErrorType sourceErrorType;
  private transient ErrorType sourceResponseErrorType;
  private transient ErrorType criticalErrorType;

  @Override
  public void postProcess() {
    initErrorTypesCollections();
  }

  protected void initErrorTypesCollections() {
    errorTypes = new LazyValue<>(() -> initializeErrorTypeSet(false));
    internalErrorTypes = new LazyValue<>(() -> initializeErrorTypeSet(true));
  }

  protected Map<ComponentIdentifier, ErrorType> initializeErrorTypeSet(boolean internal) {
    Map<ComponentIdentifier, ErrorType> types = new HashMap<>();
    initializeErrorTypeSetForChildrenRecursively(null, getHierarchy(), types, internal);
    return types;
  }

  protected void initializeErrorTypeSetForChildrenRecursively(ErrorType parentErrorType,
                                                              List<ErrorTypeHierarchyItemDTO> hierarchyItemChildren,
                                                              Map<ComponentIdentifier, ErrorType> types,
                                                              boolean internal) {
    for (ErrorTypeHierarchyItemDTO hierarchyChildItem : hierarchyItemChildren) {
      String namespace = hierarchyChildItem.getNamespace().toUpperCase();

      ErrorType hierarchyItemErrorType = builder()
          .namespace(namespace)
          .identifier(hierarchyChildItem.getIdentifier())
          .parentErrorType(parentErrorType)
          .build();

      if (hierarchyChildItem.isInternal() == internal) {
        ComponentIdentifier errorTypeIdentifier = ComponentIdentifier.builder()
            .namespace(namespace)
            .name(hierarchyChildItem.getIdentifier()).build();

        types.put(errorTypeIdentifier,
                  hierarchyItemErrorType);

        if (ANY_ERROR_IDENTIFIER.equals(errorTypeIdentifier)) {
          anyErrorType = hierarchyItemErrorType;
        } else if (SOURCE_ERROR_IDENTIFIER.equals(errorTypeIdentifier)) {
          sourceErrorType = hierarchyItemErrorType;
        } else if (SOURCE_RESPONSE_ERROR_IDENTIFIER.equals(errorTypeIdentifier)) {
          sourceResponseErrorType = hierarchyItemErrorType;
        } else if (CRITICAL_ERROR_IDENTIFIER.equals(errorTypeIdentifier)) {
          criticalErrorType = hierarchyItemErrorType;
        }
      }

      initializeErrorTypeSetForChildrenRecursively(hierarchyItemErrorType, hierarchyChildItem.getChildren(), types, internal);
    }
  }

  public ErrorTypeRepositoryDTO(ErrorType anyErrorType, ErrorType sourceErrorType, ErrorType sourceResponseErrorType,
                                ErrorType criticalErrorType, Set<ErrorType> errorTypes, Set<ErrorType> internalErrorTypes) {

    MultiMap<ErrorType, ErrorType> parentChildren = nonRootErrorTypesStream(errorTypes, internalErrorTypes)
        .collect(toMultiMap(ErrorType::getParentErrorType, identity()));

    this.hierarchy = rootErrorTypesStream(errorTypes, internalErrorTypes)
        .map(errorType -> createHierarchyItemRecursively(errorType, internalErrorTypes::contains, parentChildren))
        .collect(toList());

    initErrorTypesCollections();

    this.anyErrorType = anyErrorType;
    this.sourceErrorType = sourceErrorType;
    this.sourceResponseErrorType = sourceResponseErrorType;
    this.criticalErrorType = criticalErrorType;
  }

  protected Stream<ErrorType> nonRootErrorTypesStream(Set<ErrorType> errorTypes, Set<ErrorType> internalErrorTypes) {
    return concat(errorTypes.stream(), internalErrorTypes.stream())
        .filter(errorType -> errorType.getParentErrorType() != null);
  }

  protected Stream<ErrorType> rootErrorTypesStream(Set<ErrorType> errorTypes, Set<ErrorType> internalErrorTypes) {
    return concat(errorTypes.stream(), internalErrorTypes.stream())
        .filter(errorType -> errorType.getParentErrorType() == null);
  }

  private ErrorTypeHierarchyItemDTO createHierarchyItemRecursively(ErrorType errorType, Predicate<ErrorType> internal,
                                                                   MultiMap<ErrorType, ErrorType> parentChildren) {
    return new ErrorTypeHierarchyItemDTO(errorType.getIdentifier(), errorType.getNamespace(), internal.test(errorType),
                                         parentChildren.getAll(errorType).stream()
                                             .map(childErrorType -> createHierarchyItemRecursively(childErrorType,
                                                                                                   internal,
                                                                                                   parentChildren))
                                             .collect(toList()));
  }

  @Override
  public ErrorType addErrorType(ComponentIdentifier componentIdentifier, ErrorType errorType) {
    throw new UnsupportedOperationException("This error repository is immutable");
  }

  @Override
  public ErrorType addInternalErrorType(ComponentIdentifier componentIdentifier, ErrorType errorType) {
    throw new UnsupportedOperationException("This error repository is immutable");
  }

  @Override
  public Optional<ErrorType> lookupErrorType(ComponentIdentifier componentIdentifier) {
    Optional<ErrorType> optional = doLookupErrorType(errorTypes.get(), componentIdentifier);

    if (optional.isPresent()) {
      return optional;
    }

    if (getBoolean(MULE_LAX_ERROR_TYPES)) {
      LOGGER.warn("Could not find error '{}'.", componentIdentifier);

      // Create one
      return of(builder().namespace(componentIdentifier.getNamespace().toUpperCase())
          .identifier(componentIdentifier.getName())
          .parentErrorType(getAnyErrorType()).build());
    }

    return empty();
  }

  protected Optional<ErrorType> doLookupErrorType(Map<ComponentIdentifier, ErrorType> errorTypes,
                                                  ComponentIdentifier componentIdentifier) {
    return ofNullable(errorTypes.get(componentIdentifier));
  }

  @Override
  public Optional<ErrorType> getErrorType(ComponentIdentifier componentIdentifier) {
    Optional<ErrorType> errorType = doLookupErrorType(errorTypes.get(), componentIdentifier);
    if (!errorType.isPresent()) {
      errorType = doLookupErrorType(internalErrorTypes.get(), componentIdentifier);
    }
    return errorType;
  }

  @Override
  public Collection<String> getErrorNamespaces() {
    return concat(getErrorTypes().stream(), getInternalErrorTypes().stream())
        .map(errorType -> errorType.getNamespace().toUpperCase())
        .distinct()
        .collect(toList());
  }

  @Override
  public ErrorType getAnyErrorType() {
    return anyErrorType;
  }

  @Override
  public ErrorType getSourceErrorType() {
    return sourceErrorType;
  }

  @Override
  public ErrorType getSourceResponseErrorType() {
    return sourceResponseErrorType;
  }

  @Override
  public ErrorType getCriticalErrorType() {
    return criticalErrorType;
  }

  @Override
  public Set<ErrorType> getErrorTypes() {
    return new HashSet<>(errorTypes.get().values());
  }

  @Override
  public Set<ErrorType> getInternalErrorTypes() {
    return new HashSet<>(internalErrorTypes.get().values());
  }

  public List<ErrorTypeHierarchyItemDTO> getHierarchy() {
    return hierarchy;
  }
}
