/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.ast;

import static org.mule.runtime.api.util.MuleSystemProperties.SYSTEM_PROPERTY_PREFIX;
import static org.mule.runtime.ast.AllureConstants.ArtifactAstSerialization.AST_SERIALIZATION;
import static org.mule.runtime.ast.AllureConstants.ArtifactAstSerialization.AST_SERIALIZATION_END_TO_END;
import static org.mule.runtime.ast.api.DependencyResolutionMode.MINIMAL;
import static org.mule.runtime.ast.internal.serialization.ArtifactAstSerializerFactory.JSON;

import static java.lang.System.clearProperty;
import static java.lang.System.setProperty;
import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toSet;

import static com.google.common.collect.Streams.forEachPair;
import static org.apache.commons.io.FileUtils.copyInputStreamToFile;
import static org.hamcrest.CoreMatchers.equalTo;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.allOf;
import static org.hamcrest.Matchers.containsString;
import static org.hamcrest.Matchers.empty;
import static org.hamcrest.Matchers.endsWith;
import static org.hamcrest.Matchers.is;
import static org.hamcrest.Matchers.not;
import static org.hamcrest.Matchers.notNullValue;
import static org.hamcrest.Matchers.startsWith;
import static org.hamcrest.collection.IsCollectionWithSize.hasSize;
import static org.junit.Assert.assertThrows;

import org.mule.runtime.api.component.location.LocationPart;
import org.mule.runtime.api.meta.model.ExtensionModel;
import org.mule.runtime.api.meta.model.parameter.ParameterModel;
import org.mule.runtime.ast.api.ArtifactAst;
import org.mule.runtime.ast.api.ComponentAst;
import org.mule.runtime.ast.api.ComponentMetadataAst;
import org.mule.runtime.ast.api.ComponentParameterAst;
import org.mule.runtime.ast.api.DependencyResolutionMode;
import org.mule.runtime.ast.api.ImportedResource;
import org.mule.runtime.ast.api.serialization.ArtifactAstDeserializer;
import org.mule.runtime.ast.api.serialization.ArtifactAstSerializer;
import org.mule.runtime.ast.api.serialization.ArtifactAstSerializerProvider;
import org.mule.runtime.ast.api.serialization.ExtensionModelResolver;
import org.mule.runtime.ast.testobjects.TestArtifactAstFactory;
import org.mule.runtime.ast.testobjects.TestExtensionModelResolver;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.util.List;
import java.util.Optional;
import java.util.OptionalInt;
import java.util.Set;

import org.apache.commons.io.IOUtils;

import org.junit.After;
import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.TemporaryFolder;

import io.qameta.allure.Feature;
import io.qameta.allure.Issue;
import io.qameta.allure.Story;

@Feature(AST_SERIALIZATION)
@Story(AST_SERIALIZATION_END_TO_END)
public class ArtifactAstSerializationTestCase {

  @Rule
  public TemporaryFolder folder = new TemporaryFolder();

  private ArtifactAstSerializerProvider artifactAstSerializerProvider;
  private TestArtifactAstFactory testArtifactAstFactory;
  private ExtensionModelResolver extensionModelResolver;
  private Set<ExtensionModel> extensionModels;

  @Before
  public void setUp() throws Exception {
    artifactAstSerializerProvider = new ArtifactAstSerializerProvider();
    testArtifactAstFactory = new TestArtifactAstFactory();
    extensionModels = testArtifactAstFactory.extensionModelsSetForTests();
    extensionModelResolver = new TestExtensionModelResolver(extensionModels);
  }

  @After
  public void tearDown() {
    clearProperty(SYSTEM_PROPERTY_PREFIX + DependencyResolutionMode.class.getName());
  }

  @Test
  public void testSerializationOfTrivialAppJsonImplementation() throws IOException {
    ArtifactAstSerializer artifactAstSerializer = artifactAstSerializerProvider.getSerializer(JSON, "1.0");

    ArtifactAst trivialArtifactAst = testArtifactAstFactory.createArtifactFromXmlFile("test-apps/trivial.xml", extensionModels);

    InputStream serializedArtifactAst = artifactAstSerializer.serialize(trivialArtifactAst);

    String jsonString = IOUtils.toString(new InputStreamReader(serializedArtifactAst));
    assertThat(jsonString, containsString("JSON#1.0"));
  }

  @Test
  public void testSerializationOfSimpleHttpAppJsonImplementation() throws IOException {
    ArtifactAstSerializer artifactAstSerializer = artifactAstSerializerProvider.getSerializer(JSON, "1.0");

    ArtifactAst simpleHttpArtifactAst =
        testArtifactAstFactory.createArtifactFromXmlFile("test-apps/http-listener-with-logger.xml", extensionModels);

    InputStream serializedArtifactAst = artifactAstSerializer.serialize(simpleHttpArtifactAst);

    String jsonString = IOUtils.toString(new InputStreamReader(serializedArtifactAst));
    assertThat(jsonString, containsString("JSON#1.0"));
    assertThat(jsonString, containsString("/test"));
    assertThat(jsonString, containsString("HTTP_Listener_config"));
    assertThat(jsonString, containsString("8081"));
    assertThat(jsonString, containsString("INFO"));
  }

  @Test
  public void testSerializationOfImportedSimpleHttpAppJsonImplementation() throws IOException {
    ArtifactAstSerializer artifactAstSerializer = artifactAstSerializerProvider.getSerializer(JSON, "1.0");

    ArtifactAst simpleHttpArtifactAst =
        testArtifactAstFactory.createArtifactFromXmlFile("test-apps/imports-http-listener-with-logger.xml", extensionModels);

    InputStream serializedArtifactAst = artifactAstSerializer.serialize(simpleHttpArtifactAst);

    String jsonString = IOUtils.toString(new InputStreamReader(serializedArtifactAst));
    assertThat(jsonString, containsString("JSON#1.0"));
    assertThat(jsonString, containsString("/test"));
    assertThat(jsonString, containsString("HTTP_Listener_config"));
    assertThat(jsonString, containsString("8081"));
    assertThat(jsonString, containsString("INFO"));
  }

  @Test
  public void testSerializationOfUnresolvableImportJsonImplementation() throws IOException {
    ArtifactAstSerializer artifactAstSerializer = artifactAstSerializerProvider.getSerializer(JSON, "1.0");

    ArtifactAst simpleHttpArtifactAst =
        testArtifactAstFactory.createArtifactFromXmlFile("test-apps/failing-imports.xml", extensionModels);

    InputStream serializedArtifactAst = artifactAstSerializer.serialize(simpleHttpArtifactAst);

    String jsonString = IOUtils.toString(new InputStreamReader(serializedArtifactAst));
    assertThat(jsonString, containsString("JSON#1.0"));
    assertThat(jsonString, containsString("Could not find imported resource \\u0027unresolvable.xml\\u0027"));
  }

  @Test
  public void testSerializationDeserializationOfTrivialAppJsonImplementation() throws IOException {
    ArtifactAstSerializer artifactAstSerializer = artifactAstSerializerProvider.getSerializer(JSON, "1.0");

    ArtifactAst trivialArtifactAst = testArtifactAstFactory.createArtifactFromXmlFile("test-apps/trivial.xml", extensionModels);

    InputStream serializedArtifactAst = artifactAstSerializer.serialize(trivialArtifactAst);

    ArtifactAst artifactAst =
        artifactAstSerializerProvider.getDeserializer().deserialize(serializedArtifactAst, extensionModelResolver);

    assertThat(artifactAst.topLevelComponentsStream().count(), is(trivialArtifactAst.topLevelComponentsStream().count()));

    assertThat(artifactAst.dependencies().size(),
               is(trivialArtifactAst.dependencies().stream().map(ExtensionModel::getName).collect(toSet()).size()));

    assertThat(artifactAst.getErrorTypeRepository().getErrorTypes().size(),
               is(trivialArtifactAst.getErrorTypeRepository().getErrorTypes().size()));
  }

  @Test
  public void testSerializationDeserializationOfSimpleHttpAppJsonImplementation() throws IOException {
    ArtifactAstSerializer artifactAstSerializer = artifactAstSerializerProvider.getSerializer(JSON, "1.0");

    ArtifactAst simpleHttpArtifactAst =
        testArtifactAstFactory.createArtifactFromXmlFile("test-apps/http-listener-with-logger.xml", extensionModels);

    InputStream serializedArtifactAst = artifactAstSerializer.serialize(simpleHttpArtifactAst);

    ArtifactAstDeserializer deserializer = artifactAstSerializerProvider.getDeserializer();

    ArtifactAst artifactAst = deserializer.deserialize(serializedArtifactAst, extensionModelResolver);

    assertThat(artifactAst.topLevelComponentsStream().count(), is(simpleHttpArtifactAst.topLevelComponentsStream().count()));

    assertThat(artifactAst.dependencies().size(),
               is(simpleHttpArtifactAst.dependencies().stream().map(ExtensionModel::getName).collect(toSet()).size()));

    List<ComponentParameterAst> componentParameterAstStream = artifactAst.topLevelComponentsStream()
        .flatMap(componentAst -> componentAst.getParameters().stream()).collect(toList());
    assertThat(componentParameterAstStream, not(empty()));

    // These assertions require the extension model to be properly formed.
    // A http extension model with a configuration model for the listener's config. This configuration model must have a
    // connection provider model for the listeners-connection and a source model for the http:listener message source
    // A mule extension model is also necessary for the flow construct and the logger operation.
    // Every one of these models needs to have its parameters defined as parameter models inside parameter groups.
    componentParameterAstStream
        .forEach(componentParameterAst -> assertThat(componentParameterAst.getModel(), notNullValue(ParameterModel.class)));
  }

  @Test
  public void testSerializationDeserializationOfSimpleHttpAppJsonImplementationKeepsComponentMetadataAstEquivalent()
      throws IOException {
    ArtifactAstSerializer artifactAstSerializer = artifactAstSerializerProvider.getSerializer(JSON, "1.0");

    ArtifactAst simpleHttpArtifactAst =
        testArtifactAstFactory.createArtifactFromXmlFile("test-apps/http-listener-with-logger.xml", extensionModels);

    InputStream serializedArtifactAst = artifactAstSerializer.serialize(simpleHttpArtifactAst);

    ArtifactAstDeserializer deserializer = artifactAstSerializerProvider.getDeserializer();

    ArtifactAst artifactAst = deserializer.deserialize(serializedArtifactAst, extensionModelResolver);

    assertThat(artifactAst.topLevelComponentsStream().count(), is(simpleHttpArtifactAst.topLevelComponentsStream().count()));

    // iterates all components recursively and asserts their metadata is equal
    forEachPair(artifactAst.recursiveStream().map(ComponentAst::getMetadata),
                simpleHttpArtifactAst.recursiveStream().map(ComponentAst::getMetadata),
                this::assertComponentMetadataAstEqual);
  }

  @Test
  @Issue("MULE-19775")
  public void testSerializationToFileAndDeserializationOfSimpleHttpAppJsonImplementationKeepsComponentMetadataAstEquivalent()
      throws IOException {
    ArtifactAstSerializer artifactAstSerializer = artifactAstSerializerProvider.getSerializer(JSON, "1.0");

    ArtifactAst simpleHttpArtifactAst =
        testArtifactAstFactory.createArtifactFromXmlFile("test-apps/http-listener-with-logger.xml", extensionModels);

    InputStream serializedArtifactAst = artifactAstSerializer.serialize(simpleHttpArtifactAst);

    File file = folder.newFile("http-listener-artifact.ast");
    copyInputStreamToFile(serializedArtifactAst, file);

    ArtifactAstDeserializer deserializer = artifactAstSerializerProvider.getDeserializer();

    InputStream targetStream = new FileInputStream(file);

    ArtifactAst artifactAst = deserializer.deserialize(targetStream, extensionModelResolver);

    assertThat(artifactAst.topLevelComponentsStream().count(), is(simpleHttpArtifactAst.topLevelComponentsStream().count()));

    // iterates all components recursively and asserts their metadata is equal
    forEachPair(artifactAst.recursiveStream().map(ComponentAst::getMetadata),
                simpleHttpArtifactAst.recursiveStream().map(ComponentAst::getMetadata),
                this::assertComponentMetadataAstEqual);
  }

  private void assertComponentMetadataAstEqual(ComponentMetadataAst actual, ComponentMetadataAst expected) {
    assertThat(actual.getDocAttributes(), equalTo(expected.getDocAttributes()));
    assertThat(actual.getParserAttributes(), equalTo(expected.getParserAttributes()));
    assertThat(actual.getFileName(), equalTo(expected.getFileName()));
    assertThat(actual.getFileUri(), equalTo(expected.getFileUri()));
    assertThat(actual.getSourceCode(), equalTo(expected.getSourceCode()));
    assertThat(actual.getStartColumn(), equalTo(expected.getStartColumn()));
    assertThat(actual.getStartLine(), equalTo(expected.getStartLine()));
    assertThat(actual.getEndColumn(), equalTo(expected.getEndColumn()));
    assertThat(actual.getEndLine(), equalTo(expected.getEndLine()));
  }


  @Test
  public void testSerializationDeserializationOfImportChain() throws IOException {
    ArtifactAstSerializer artifactAstSerializer = artifactAstSerializerProvider.getSerializer(JSON, "1.0");

    ArtifactAst simpleHttpArtifactAst =
        testArtifactAstFactory.createArtifactFromXmlFile("test-apps/imports-http-listener-with-logger.xml", extensionModels);

    InputStream serializedArtifactAst = artifactAstSerializer.serialize(simpleHttpArtifactAst);

    ArtifactAstDeserializer deserializer = artifactAstSerializerProvider.getDeserializer();

    ArtifactAst artifactAst = deserializer.deserialize(serializedArtifactAst, extensionModelResolver);

    assertThat(artifactAst.topLevelComponentsStream().count(), is(simpleHttpArtifactAst.topLevelComponentsStream().count()));

    assertThat(artifactAst.dependencies().size(),
               is(simpleHttpArtifactAst.dependencies().stream().map(ExtensionModel::getName).collect(toSet()).size()));

    List<ComponentAst> componentAstStream = artifactAst.topLevelComponents();

    componentAstStream
        .forEach(componentAst -> {
          List<ImportedResource> componentImportChain = componentAst.getMetadata().getImportChain();
          assertThat(componentImportChain, hasSize(1));
          assertThat(componentImportChain.get(0).getResourceLocation(), is("test-apps/http-listener-with-logger.xml"));
          assertThat(componentImportChain.get(0).getMetadata().getFileName().get(),
                     is("test-apps/imports-http-listener-with-logger.xml"));
          assertThat(componentImportChain.get(0).getMetadata().getFileUri().get().toString(),
                     allOf(startsWith("file:/"), endsWith("test-apps/imports-http-listener-with-logger.xml")));
        });
  }

  @Test
  public void testSerializationDeserializationOfAppWithDescriptions() throws IOException {
    ArtifactAstSerializer artifactAstSerializer = artifactAstSerializerProvider.getSerializer(JSON, "1.0");

    ArtifactAst trivialArtifactAst =
        testArtifactAstFactory.createArtifactFromXmlFile("test-apps/with-descriptions.xml", extensionModels);

    InputStream serializedArtifactAst = artifactAstSerializer.serialize(trivialArtifactAst);

    ArtifactAst artifactAst =
        artifactAstSerializerProvider.getDeserializer().deserialize(serializedArtifactAst, extensionModelResolver);

    assertThat(artifactAst.topLevelComponentsStream().count(), is(trivialArtifactAst.topLevelComponentsStream().count()));

    assertThat(artifactAst.dependencies().size(),
               is(trivialArtifactAst.dependencies().stream().map(ExtensionModel::getName).collect(toSet()).size()));

    assertThat(artifactAst.getErrorTypeRepository().getErrorTypes().size(),
               is(trivialArtifactAst.getErrorTypeRepository().getErrorTypes().size()));
  }

  @Test
  @Issue("MULE-19818")
  public void testSerializationDeserializationOfLocationFileCoordinates() throws IOException {
    ArtifactAstSerializer artifactAstSerializer = artifactAstSerializerProvider.getSerializer(JSON, "1.0");

    ArtifactAst simpleHttpArtifactAst =
        testArtifactAstFactory.createArtifactFromXmlFile("test-apps/imports-http-listener-with-logger.xml", extensionModels);

    InputStream serializedArtifactAst = artifactAstSerializer.serialize(simpleHttpArtifactAst);

    ArtifactAstDeserializer deserializer = artifactAstSerializerProvider.getDeserializer();

    ArtifactAst artifactAst = deserializer.deserialize(serializedArtifactAst, extensionModelResolver);

    ComponentAst httpListenerConfig = artifactAst.topLevelComponents().get(0);
    assertThat(httpListenerConfig.getLocation().getLine().getAsInt(), is(6));
    assertThat(httpListenerConfig.getLocation().getColumn().getAsInt(), is(5));
    ComponentAst listenerConnection = httpListenerConfig.directChildren().get(0);
    assertThat(listenerConnection.getLocation().getLine().getAsInt(), is(7));
    assertThat(listenerConnection.getLocation().getColumn().getAsInt(), is(9));
    ComponentAst flow = artifactAst.topLevelComponents().get(1);
    assertThat(flow.getLocation().getLine().getAsInt(), is(9));
    assertThat(flow.getLocation().getColumn().getAsInt(), is(5));
    ComponentAst listenerSource = flow.directChildren().get(0);
    assertThat(listenerSource.getLocation().getLine().getAsInt(), is(10));
    assertThat(listenerSource.getLocation().getColumn().getAsInt(), is(9));
    ComponentAst logger = flow.directChildren().get(1);
    assertThat(logger.getLocation().getLine().getAsInt(), is(11));
    assertThat(logger.getLocation().getColumn().getAsInt(), is(9));
  }

  @Test
  @Issue("MULE-19826")
  public void testProcessorsPartPath() throws IOException {
    ArtifactAstSerializer artifactAstSerializer = artifactAstSerializerProvider.getSerializer(JSON, "1.0");

    ArtifactAst simpleHttpArtifactAst =
        testArtifactAstFactory.createArtifactFromXmlFile("test-apps/imports-http-listener-with-logger.xml", extensionModels);

    InputStream serializedArtifactAst = artifactAstSerializer.serialize(simpleHttpArtifactAst);

    ArtifactAstDeserializer deserializer = artifactAstSerializerProvider.getDeserializer();

    ArtifactAst artifactAst = deserializer.deserialize(serializedArtifactAst, extensionModelResolver);

    ComponentAst flow = artifactAst.topLevelComponents().get(1);
    ComponentAst logger = flow.directChildren().get(1);

    LocationPart processorsLocationPart = logger.getLocation().getParts().get(1);
    assertThat(processorsLocationPart.getPartPath(), is("processors"));
    assertThat(processorsLocationPart.getPartIdentifier(), is(Optional.empty()));
    assertThat(processorsLocationPart.getFileName(), is(Optional.empty()));
    assertThat(processorsLocationPart.getLine(), is(OptionalInt.empty()));
    assertThat(processorsLocationPart.getColumn(), is(OptionalInt.empty()));
  }

  @Test
  @Issue("MULE-19850")
  public void serializeWithCustomRependencyResolutionModeFails() {
    setProperty(SYSTEM_PROPERTY_PREFIX + DependencyResolutionMode.class.getName(),
                MINIMAL.name());

    ArtifactAstSerializer artifactAstSerializer = artifactAstSerializerProvider.getSerializer(JSON, "1.0");

    ArtifactAst trivialArtifactAst = testArtifactAstFactory.createArtifactFromXmlFile("test-apps/trivial.xml", extensionModels);

    assertThrows("Artifact AST serialization with `DependencyResolutionMode != COMPILE` would have limited usability when deserialized.",
                 IllegalStateException.class,
                 () -> artifactAstSerializer.serialize(trivialArtifactAst));

  }
}
