/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.ast.internal.serialization.dto.factory;

import static org.mule.runtime.ast.AllureConstants.ArtifactAstSerialization.AST_DTO;
import static org.mule.runtime.ast.AllureConstants.ArtifactAstSerialization.AST_SERIALIZATION;

import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.hasSize;
import static org.hamcrest.Matchers.is;
import static org.mockito.ArgumentMatchers.any;
import static org.mockito.Mockito.inOrder;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.times;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;

import org.mule.runtime.api.meta.model.ExtensionModel;
import org.mule.runtime.ast.api.ArtifactAst;
import org.mule.runtime.ast.api.ComponentAst;
import org.mule.runtime.ast.api.ComponentMetadataAst;
import org.mule.runtime.ast.api.ComponentParameterAst;
import org.mule.runtime.ast.internal.serialization.dto.ComponentAstDTO;
import org.mule.runtime.ast.internal.serialization.dto.ComponentMetadataAstDTO;
import org.mule.runtime.ast.internal.serialization.dto.ComponentParameterAstDTO;
import org.mule.runtime.ast.internal.serialization.dto.ExtensionModelDTO;
import org.mule.runtime.ast.testobjects.TestArtifactAstFactory;

import java.util.Set;

import org.junit.Before;
import org.junit.Test;

import org.mockito.InOrder;

import io.qameta.allure.Feature;
import io.qameta.allure.Story;

@Feature(AST_SERIALIZATION)
@Story(AST_DTO)
public class ComponentAstDTOFactoryTestCase {

  private TestArtifactAstFactory testArtifactAstFactory;
  private AstDTOFactoryProvider astDTOFactoryProvider;
  private ExtensionModelDTOFactory extensionModelDTOFactory;
  private ComponentMetadataAstDTOFactory componentMetadataAstDTOFactory;
  private ComponentParameterAstDTOFactory componentParameterAstDTOFactory;
  private ExtensionModelDTO extensionModelDTO;
  private ComponentMetadataAstDTO componentMetadataAstDTO;
  private ComponentParameterAstDTO componentParameterAstDTO;
  private Set<ExtensionModel> extensionModels;

  @Before
  public void setUp() throws Exception {
    testArtifactAstFactory = new TestArtifactAstFactory();

    // Extension model builder
    extensionModelDTOFactory = mock(ExtensionModelDTOFactory.class);
    extensionModelDTO = new ExtensionModelDTO("Some name");
    when(extensionModelDTOFactory.from(any(ExtensionModel.class))).thenReturn(extensionModelDTO);

    componentMetadataAstDTOFactory = mock(ComponentMetadataAstDTOFactory.class);
    componentMetadataAstDTO = new ComponentMetadataAstDTO(null, null, null, null, null, null, null, null, null, null);
    when(componentMetadataAstDTOFactory.from(any(ComponentMetadataAst.class))).thenReturn(componentMetadataAstDTO);

    componentParameterAstDTOFactory = mock(ComponentParameterAstDTOFactory.class);
    componentParameterAstDTO = new ComponentParameterAstDTO(null, null, false, null, null, null);
    when(componentParameterAstDTOFactory.from(any(ComponentParameterAst.class))).thenReturn(componentParameterAstDTO);

    // Local implementation that overrides builder getters to return the mocks initialized above
    astDTOFactoryProvider = new TestAstDTOFactoryProvider();
  }

  @Test
  public void testMappingSimpleHttpAppConfigComponent() {
    // Given
    extensionModels = testArtifactAstFactory.extensionModelsSetForTests();
    ArtifactAst simpleHttpArtifact =
        testArtifactAstFactory.createArtifactFromXmlFile("test-apps/http-listener-with-logger.xml", extensionModels);

    // There are 2 top level components. The http config and one flow
    assertThat(simpleHttpArtifact.topLevelComponentsStream().count(), is(2L));

    ComponentAst config = simpleHttpArtifact.topLevelComponentsStream()
        .filter(componentAst -> "listener-config".equals(componentAst.getIdentifier().getName())).findFirst().get();

    // When
    ComponentAstDTO configDTO = astDTOFactoryProvider.getComponentAstDTOFactory().from(config);

    // Then
    // The config is accurately mapped to a DTO
    assertDTOIsAnalogToComponentAST(config, configDTO);
    verify(extensionModelDTOFactory, times(2)).from(any(ExtensionModel.class));
  }

  @Test
  public void testMappingSimpleHttpAppFlowComponent() {
    // Given
    extensionModels = testArtifactAstFactory.extensionModelsSetForTests();
    ArtifactAst simpleHttpArtifact =
        testArtifactAstFactory.createArtifactFromXmlFile("test-apps/http-listener-with-logger.xml", extensionModels);

    // There are 2 top level components. The http config and one flow
    assertThat(simpleHttpArtifact.topLevelComponentsStream().count(), is(2L));

    ComponentAst flow = simpleHttpArtifact.topLevelComponentsStream()
        .filter(componentAst -> "flow".equals(componentAst.getIdentifier().getName())).findFirst().get();

    // When
    ComponentAstDTO flowDTO = astDTOFactoryProvider.getComponentAstDTOFactory().from(flow);

    // Then
    // The flow is accurately mapped to a DTO
    assertDTOIsAnalogToComponentAST(flow, flowDTO);

    assertThat(flowDTO.directChildrenStream().count(), is(2L));

    flow.directChildrenStream().forEach(componentAst -> {
      ComponentAstDTO componentAstDTO = astDTOFactoryProvider.getComponentAstDTOFactory().from(componentAst);
      assertDTOIsAnalogToComponentAST(componentAst, componentAstDTO);
    });

    verify(extensionModelDTOFactory, times(5)).from(any(ExtensionModel.class));
  }

  private void assertDTOIsAnalogToComponentAST(ComponentAst componentAst, ComponentAstDTO componentAstDTO) {
    assertThat(componentAstDTO.getIdentifier(), is(componentAst.getIdentifier()));
    assertThat(componentAstDTO.getLocation(), is(componentAst.getLocation()));

    assertThat(componentAstDTO.getComponentId(), is(componentAst.getComponentId()));
    assertThat(componentAstDTO.getComponentType(), is(componentAst.getComponentType()));

    if (componentAst.getExtensionModel() != null) {
      assertThat(componentAstDTO.getExtensionModelDTO(), is(extensionModelDTO));
    }

    if (componentAst.getMetadata() != null) {
      InOrder order = inOrder(componentMetadataAstDTOFactory, componentMetadataAstDTOFactory);
      order.verify(componentMetadataAstDTOFactory, times(1)).from(componentAst.getMetadata());
      assertThat(componentAstDTO.getMetadata(), is(componentMetadataAstDTO));
    }

    componentAst.getParameters().forEach(param -> {
      InOrder order = inOrder(componentParameterAstDTOFactory, componentParameterAstDTOFactory);
      order.verify(componentParameterAstDTOFactory, times(1)).from(param);
    });

    if (!componentAst.getParameters().isEmpty()) {
      assertThat(componentAstDTO.getParameters().size(), is(componentAst.getParameters().size()));
      // The builder is mocked to return the same DTO every time
      componentAstDTO.getParameters().forEach(paramDTO -> assertThat(paramDTO, is(componentParameterAstDTO)));
    }

    assertThat(componentAstDTO.getParameters(), hasSize(componentAst.getParameters().size()));

    assertThat(componentAstDTO.getType(), is(componentAst.getType()));

    if (componentAst.getAnnotations() != null) {
      assertThat(componentAstDTO.getAnnotations(), is(componentAst.getAnnotations()));
    }

  }

  private class TestAstDTOFactoryProvider extends AstDTOFactoryProvider {

    @Override
    public ExtensionModelDTOFactory getExtensionModelDTOFactory() {
      return extensionModelDTOFactory;
    }

    @Override
    public ComponentMetadataAstDTOFactory getComponentMetadataAstDTOFactory() {
      return componentMetadataAstDTOFactory;
    }

    @Override
    public ComponentParameterAstDTOFactory getComponentParameterAstDTOFactory() {
      return componentParameterAstDTOFactory;
    }
  }
}
