/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.ast.internal.serialization.json;

import static org.mule.runtime.ast.AllureConstants.ArtifactAstSerialization.AST_JSON_DESERIALIZER;
import static org.mule.runtime.ast.AllureConstants.ArtifactAstSerialization.AST_SERIALIZATION;

import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.is;
import static org.mockito.ArgumentMatchers.any;
import static org.mockito.ArgumentMatchers.eq;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.times;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;

import org.mule.runtime.ast.api.ArtifactAst;
import org.mule.runtime.ast.api.serialization.ExtensionModelResolver;
import org.mule.runtime.ast.internal.serialization.dto.ArtifactAstDTO;
import org.mule.runtime.ast.internal.serialization.resolver.GenerationInformationResolver;

import java.io.IOException;
import java.io.InputStream;
import java.io.Reader;

import com.google.gson.Gson;
import com.google.gson.GsonBuilder;

import org.junit.Before;
import org.junit.Test;

import io.qameta.allure.Feature;
import io.qameta.allure.Issue;
import io.qameta.allure.Story;

@Feature(AST_SERIALIZATION)
@Story(AST_JSON_DESERIALIZER)
public class JsonArtifactAstDeserializerTestCase {

  private JsonArtifactAstDeserializer jsonArtifactAstDeserializer;
  private Gson gson;
  private ArtifactAstDTO artifactAstDTO;

  @Before
  public void setUp() throws Exception {
    GsonBuilder gsonBuilder = mock(GsonBuilder.class);
    gson = mock(Gson.class);
    when(gsonBuilder.create()).thenReturn(gson);
    artifactAstDTO = mock(ArtifactAstDTO.class);
    when(gson.fromJson(any(Reader.class), eq(ArtifactAstDTO.class))).thenReturn(artifactAstDTO);
    jsonArtifactAstDeserializer = new JsonArtifactAstDeserializer(gsonBuilder, true);
  }

  @Test
  public void testDeserializeCallGsonFromJsonForArtifactAstDTOClass_WhenDeserializingAnInputStream() throws IOException {
    // Given
    InputStream artifactAstInputStream = mock(InputStream.class);
    ExtensionModelResolver extensionModelResolver = mock(ExtensionModelResolver.class);

    // When
    jsonArtifactAstDeserializer.deserialize(artifactAstInputStream, extensionModelResolver);

    // Then
    verify(gson, times(1)).fromJson(any(Reader.class), eq(ArtifactAstDTO.class));
  }

  @Test
  public void testDeserializeCallGsonFromJsonResultsEnrich_WhenDeserializingAnInputStream() throws IOException {
    // Given
    InputStream artifactAstInputStream = mock(InputStream.class);
    ExtensionModelResolver extensionModelResolver = mock(ExtensionModelResolver.class);

    // When
    jsonArtifactAstDeserializer.deserialize(artifactAstInputStream, extensionModelResolver);

    // Then
    verify(artifactAstDTO, times(1)).enrich(any(ExtensionModelResolver.class), any(GenerationInformationResolver.class));
  }

  @Test
  @Issue("MULE-19834")
  public void testDeserializeCallsResultingArtifactSetParentWithInputParent_WhenDeserializingAnInputStreamWithParent()
      throws IOException {
    // Given
    InputStream artifactAstInputStream = mock(InputStream.class);
    ExtensionModelResolver extensionModelResolver = mock(ExtensionModelResolver.class);
    ArtifactAst parent = mock(ArtifactAst.class);

    // When
    jsonArtifactAstDeserializer.deserialize(artifactAstInputStream, extensionModelResolver, parent);

    // Then
    verify(artifactAstDTO, times(1)).setParent(parent);
  }

  @Test
  public void testDeserializeReturnsEnrichedArtifactAst_WhenDeserializingAnInputStream() throws IOException {
    // Given
    InputStream artifactAstInputStream = mock(InputStream.class);
    ExtensionModelResolver extensionModelResolver = mock(ExtensionModelResolver.class);

    // When
    ArtifactAst artifactAst = jsonArtifactAstDeserializer.deserialize(artifactAstInputStream, extensionModelResolver);

    // Then
    assertThat(artifactAst, is(artifactAstDTO));
  }

}
