/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.ast.internal.serialization.json.gson;

import static org.mule.runtime.ast.AllureConstants.ArtifactAstSerialization.AST_JSON_DESERIALIZER;
import static org.mule.runtime.ast.AllureConstants.ArtifactAstSerialization.AST_JSON_SERIALIZER;
import static org.mule.runtime.ast.AllureConstants.ArtifactAstSerialization.AST_SERIALIZATION;
import static org.mule.runtime.ast.AllureConstants.ArtifactAstSerialization.AST_SERIALIZATION_VERSIONING;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.is;
import static org.hamcrest.Matchers.nullValue;

import com.google.gson.Gson;
import io.qameta.allure.Description;
import io.qameta.allure.Feature;
import io.qameta.allure.Stories;
import io.qameta.allure.Story;
import org.junit.Test;

@Feature(AST_SERIALIZATION)
@Stories({
    @Story(AST_JSON_SERIALIZER),
    @Story(AST_JSON_DESERIALIZER),
    @Story(AST_SERIALIZATION_VERSIONING)
})
public class GsonChangesToleranceTestCase {

  @Test
  @Description("Gson tolerates older Jsons with missing attributes using different new object to deserialize")
  public void testNewVersionHasMoreAttributesAndSerializedIsOldAndDeserializerNew() {
    // Given
    Gson gson = new Gson();
    String jsonString = gson.toJson(new OldVersion("one", "two"));

    // When
    NewAttributesVersion newAttributesVersion = gson.fromJson(jsonString, NewAttributesVersion.class);

    // Then
    assertThat(newAttributesVersion.getOldAttribute1(), is("one"));
    assertThat(newAttributesVersion.getOldAttribute2(), is("two"));
    assertThat(newAttributesVersion.getNewAttribute(), nullValue());
  }

  @Test
  @Description("Gson tolerates older Jsons with missing attributes using extended old object to deserialize")
  public void testExtendedOlVersionHasMoreAttributesAndSerializedIsOldAndDeserializerExtended() {
    // Given
    Gson gson = new Gson();
    String jsonString = gson.toJson(new OldVersion("one", "two"));

    // When
    ExtendedOldVersion newVersion = gson.fromJson(jsonString, ExtendedOldVersion.class);

    // Then
    assertThat(newVersion.getOldAttribute1(), is("one"));
    assertThat(newVersion.getOldAttribute2(), is("two"));
    assertThat(newVersion.getExtendedAttribute(), nullValue());
  }

  @Test
  @Description("Gson tolerates inner objects of an older version with less attributes")
  public void testNewVersionHasMoreAttributesAndSerializedIsOldInsideAnotherObjectAndDeserializerNew() {
    // Given
    Gson gson = new Gson();
    String jsonString = gson.toJson(new HighOrderObjectOld(new OldVersion("one", "two")));

    // When
    HighOrderObjectNew highOrderObjectNew = gson.fromJson(jsonString, HighOrderObjectNew.class);

    // Then
    assertThat(highOrderObjectNew.getInner().getOldAttribute1(), is("one"));
    assertThat(highOrderObjectNew.getInner().getOldAttribute2(), is("two"));
    assertThat(highOrderObjectNew.getInner().getNewAttribute(), nullValue());
  }

  @Test
  @Description("Gson tolerates jsons of newer versions with extra attributes")
  public void testNewVersionHasMoreAttributesAndSerializedIsNewAndDeserializerOld() {
    // Given
    Gson gson = new Gson();
    String jsonString = gson.toJson(new NewAttributesVersion("one", "two", "three"));

    // When
    OldVersion oldVersion = gson.fromJson(jsonString, OldVersion.class);

    // Then
    assertThat(oldVersion.getOldAttribute1(), is("one"));
    assertThat(oldVersion.getOldAttribute2(), is("two"));
  }

  @Test
  @Description("Gson tolerates jsons of newer versions extended from the old with extra attributes")
  public void testExtendedVersionHasMoreAttributesAndSerializedIsExtendedOldAndDeserializerOld() {
    // Given
    Gson gson = new Gson();
    String jsonString = gson.toJson(new ExtendedOldVersion("one", "two", "three"));

    // When
    OldVersion oldVersion = gson.fromJson(jsonString, OldVersion.class);

    // Then
    assertThat(oldVersion.getOldAttribute1(), is("one"));
    assertThat(oldVersion.getOldAttribute2(), is("two"));
  }

  @Test
  @Description("Gson tolerates inner objects of a newer version with more attributes")
  public void testNewVersionHasMoreAttributesAndSerializedIsNewInsideAnotherObjectAndDeserializerOld() {
    // Given
    Gson gson = new Gson();
    String jsonString = gson.toJson(new HighOrderObjectNew(new NewAttributesVersion("one", "two", "three")));

    // When
    HighOrderObjectOld highOrderObjectOld = gson.fromJson(jsonString, HighOrderObjectOld.class);

    // Then
    assertThat(highOrderObjectOld.getInner().getOldAttribute1(), is("one"));
    assertThat(highOrderObjectOld.getInner().getOldAttribute2(), is("two"));
  }

  @Test
  @Description("Gson tolerates type differences in equally named attributes if content can be converted")
  public void testIncompatibleVersionChangedAttributeTypeAndSerializedIsOldAndDeserializerIncompatible() {
    // Given
    Gson gson = new Gson();
    String jsonString = gson.toJson(new OldVersion("one", "2"));

    // When
    StringIsNowIntVersion stringIsNowIntVersion = gson.fromJson(jsonString, StringIsNowIntVersion.class);

    // Then
    assertThat(stringIsNowIntVersion.getOldAttribute1(), is("one"));
    assertThat(stringIsNowIntVersion.getOldAttribute2(), is(2));
  }

  private class OldVersion {

    protected String oldAttribute1;
    protected String oldAttribute2;

    public OldVersion(String oldAttribute1, String oldAttribute2) {
      this.oldAttribute1 = oldAttribute1;
      this.oldAttribute2 = oldAttribute2;
    }

    public String getOldAttribute1() {
      return oldAttribute1;
    }

    public String getOldAttribute2() {
      return oldAttribute2;
    }
  }

  private class StringIsNowIntVersion {

    protected String oldAttribute1;
    protected Integer oldAttribute2;

    public StringIsNowIntVersion(String oldAttribute1, Integer oldAttribute2) {
      this.oldAttribute1 = oldAttribute1;
      this.oldAttribute2 = oldAttribute2;
    }

    public String getOldAttribute1() {
      return oldAttribute1;
    }

    public Integer getOldAttribute2() {
      return oldAttribute2;
    }
  }

  private class NewAttributesVersion {

    private String oldAttribute1;
    private String oldAttribute2;
    private String newAttribute;

    public NewAttributesVersion(String oldAttribute1, String oldAttribute2, String newAttribute) {
      this.oldAttribute1 = oldAttribute1;
      this.oldAttribute2 = oldAttribute2;
      this.newAttribute = newAttribute;
    }

    public String getOldAttribute1() {
      return oldAttribute1;
    }

    public String getOldAttribute2() {
      return oldAttribute2;
    }

    public String getNewAttribute() {
      return newAttribute;
    }
  }

  private class ExtendedOldVersion extends OldVersion {

    private String extendedAttribute;

    public ExtendedOldVersion(String oldAttribute1, String oldAttribute2, String newAttribute) {
      super(oldAttribute1, oldAttribute2);
      this.extendedAttribute = newAttribute;
    }

    public String getExtendedAttribute() {
      return extendedAttribute;
    }
  }

  private class HighOrderObjectOld {

    private OldVersion inner;

    public HighOrderObjectOld(OldVersion oldVersion) {
      this.inner = oldVersion;
    }

    public OldVersion getInner() {
      return inner;
    }
  }

  private class HighOrderObjectNew {

    private NewAttributesVersion inner;

    public HighOrderObjectNew(NewAttributesVersion newAttributesVersion) {
      this.inner = newAttributesVersion;
    }

    public NewAttributesVersion getInner() {
      return inner;
    }
  }
}
