/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.ast.internal.serialization.resolver;

import static org.mule.runtime.api.component.TypedComponentIdentifier.ComponentType.OPERATION;
import static org.mule.runtime.api.component.TypedComponentIdentifier.ComponentType.UNKNOWN;
import static org.mule.runtime.ast.AllureConstants.ArtifactAstSerialization.AST_SERIALIZATION;
import static org.mule.runtime.ast.AllureConstants.ArtifactAstSerialization.AST_SERIALIZATION_ENRICH;

import static java.util.Arrays.asList;
import static java.util.Optional.of;

import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.is;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;

import org.mule.metadata.api.model.MetadataType;
import org.mule.runtime.api.component.ComponentIdentifier;
import org.mule.runtime.api.component.location.ComponentLocation;
import org.mule.runtime.api.meta.model.parameter.ParameterGroupModel;
import org.mule.runtime.api.meta.model.parameter.ParameterModel;
import org.mule.runtime.api.meta.model.parameter.ParameterizedModel;
import org.mule.runtime.ast.internal.model.ExtensionModelHelper;
import org.mule.runtime.ast.internal.serialization.dto.ComponentAstDTO;
import org.mule.runtime.ast.internal.serialization.dto.ComponentGenerationInformationDTO;
import org.mule.runtime.ast.internal.serialization.dto.ComponentParameterAstDTO;
import org.mule.runtime.ast.internal.serialization.dto.ParameterValueContainer;
import org.mule.runtime.extension.api.dsl.syntax.DslElementSyntax;

import org.junit.Before;
import org.junit.Test;

import io.qameta.allure.Feature;
import io.qameta.allure.Issue;
import io.qameta.allure.Story;

@Feature(AST_SERIALIZATION)
@Story(AST_SERIALIZATION_ENRICH)
public class GenerationInformationResolverTestCase {

  private GenerationInformationResolver generationInformationResolver;
  private ExtensionModelHelper extensionModelHelper;

  private ParameterizedModel parameterizedModel;
  private ParameterGroupModel paramGroupModel;
  private ParameterModel paramModel;

  @Before
  public void setUp() throws Exception {
    generationInformationResolver = new DefaultGenerationInformationResolver();

    extensionModelHelper = mock(ExtensionModelHelper.class);

    paramModel = mock(ParameterModel.class);
    when(paramModel.getName()).thenReturn("param");

    paramGroupModel = mock(ParameterGroupModel.class);
    when(paramGroupModel.getName()).thenReturn("General");

    parameterizedModel = mock(ParameterizedModel.class);
    when(parameterizedModel.getAllParameterModels()).thenReturn(asList(paramModel));
    when(parameterizedModel.getParameterGroupModels()).thenReturn(asList(paramGroupModel));
  }

  @Test
  @Issue("MULE-19823")
  public void dslSyntaxForSubTypeParameterFromAnotherExtension() {
    ComponentIdentifier paramValueIdentifier = mock(ComponentIdentifier.class);
    when(paramValueIdentifier.getNamespace()).thenReturn("anotherExtension");
    MetadataType paramValueType = mock(MetadataType.class);
    ComponentAstDTO paramValueComponentAstDTO =
        new ComponentAstDTO(null, UNKNOWN.name(), paramValueIdentifier, mock(ComponentLocation.class),
                            null, null, null, null, null, null, null);
    paramValueComponentAstDTO.setType(paramValueType);

    ComponentParameterAstDTO componentParameterAstDTO =
        new ComponentParameterAstDTO(new ParameterValueContainer(null, paramValueComponentAstDTO),
                                     null, false, null, "General", "param");
    componentParameterAstDTO.setModel(paramModel, paramGroupModel);

    ComponentIdentifier paramOwnerIdentifier = mock(ComponentIdentifier.class);
    ComponentAstDTO componentAstDTO =
        new ComponentAstDTO(null, OPERATION.name(), paramOwnerIdentifier, mock(ComponentLocation.class),
                            null, null, null, null, null, null, null);
    componentAstDTO.setParameterizedModel(parameterizedModel);
    componentAstDTO.setGenerationInformation(new ComponentGenerationInformationDTO(null));

    DslElementSyntax paramSyntax = mock(DslElementSyntax.class);
    when(paramSyntax.isWrapped()).thenReturn(true);
    when(extensionModelHelper.resolveDslElementModel(paramModel, paramOwnerIdentifier))
        .thenReturn(paramSyntax);

    DslElementSyntax paramValueSyntax = mock(DslElementSyntax.class);
    when(paramValueSyntax.isWrapped()).thenReturn(true);
    when(extensionModelHelper.resolveDslElementModel(paramValueType, "anotherExtension"))
        .thenReturn(of(paramValueSyntax));

    ComponentGenerationInformationDTO componentGenerationInformationDTO = generationInformationResolver
        .resolveComponentParameterAstGenerationInformation(componentParameterAstDTO, componentAstDTO, extensionModelHelper);

    verify(extensionModelHelper)
        .resolveDslElementModel(paramValueComponentAstDTO.getType(),
                                paramValueComponentAstDTO.getIdentifier().getNamespace());

    assertThat(componentGenerationInformationDTO.getSyntax().get(), is(paramValueSyntax));
  }

  @Test
  @Issue("MULE-19827")
  public void dslSyntaxAvoidHelperResolutionIfAvailableFromParent() {
    DslElementSyntax ownerSyntax = mock(DslElementSyntax.class);
    DslElementSyntax paramSyntax = mock(DslElementSyntax.class);
    when(ownerSyntax.getChild("param")).thenReturn(of(paramSyntax));

    ComponentIdentifier paramOwnerIdentifier = mock(ComponentIdentifier.class);
    ComponentAstDTO componentAstDTO =
        new ComponentAstDTO(null, OPERATION.name(), paramOwnerIdentifier, mock(ComponentLocation.class),
                            null, null, null, null, null, null, null);
    componentAstDTO.setParameterizedModel(parameterizedModel);
    componentAstDTO.setGenerationInformation(new ComponentGenerationInformationDTO(ownerSyntax));

    ComponentParameterAstDTO componentParameterAstDTO =
        new ComponentParameterAstDTO(new ParameterValueContainer(null, "hello"),
                                     null, false, null, "General", "param");
    componentParameterAstDTO.setModel(paramModel, paramGroupModel);

    ComponentGenerationInformationDTO componentGenerationInformationDTO = generationInformationResolver
        .resolveComponentParameterAstGenerationInformation(componentParameterAstDTO, componentAstDTO, extensionModelHelper);

    assertThat(componentGenerationInformationDTO.getSyntax().get(), is(paramSyntax));
  }
}
