/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.ast.internal.serialization.dto.factory;

import static org.mule.metadata.api.utils.MetadataTypeUtils.getTypeId;
import static org.mule.runtime.ast.internal.serialization.dto.ComponentAstDTOModelType.CONFIGURATION;
import static org.mule.runtime.ast.internal.serialization.dto.ComponentAstDTOModelType.CONNECTION_PROVIDER;
import static org.mule.runtime.ast.internal.serialization.dto.ComponentAstDTOModelType.CONSTRUCT;
import static org.mule.runtime.ast.internal.serialization.dto.ComponentAstDTOModelType.INFRASTRUCTURE;
import static org.mule.runtime.ast.internal.serialization.dto.ComponentAstDTOModelType.NESTED_COMPONENT;
import static org.mule.runtime.ast.internal.serialization.dto.ComponentAstDTOModelType.NESTED_ROUTE;
import static org.mule.runtime.ast.internal.serialization.dto.ComponentAstDTOModelType.OPERATION;
import static org.mule.runtime.ast.internal.serialization.dto.ComponentAstDTOModelType.SOURCE;
import static org.mule.runtime.ast.internal.serialization.dto.ComponentAstDTOModelType.TYPE;
import static org.mule.runtime.extension.api.util.ExtensionMetadataTypeUtils.isInfrastructure;

import static java.util.Collections.unmodifiableList;
import static java.util.Collections.unmodifiableMap;
import static java.util.stream.Collectors.toList;

import org.mule.runtime.api.meta.NamedObject;
import org.mule.runtime.api.meta.model.config.ConfigurationModel;
import org.mule.runtime.api.meta.model.connection.ConnectionProviderModel;
import org.mule.runtime.api.meta.model.construct.ConstructModel;
import org.mule.runtime.api.meta.model.nested.NestableElementModel;
import org.mule.runtime.api.meta.model.nested.NestedComponentModel;
import org.mule.runtime.api.meta.model.nested.NestedRouteModel;
import org.mule.runtime.api.meta.model.operation.OperationModel;
import org.mule.runtime.api.meta.model.source.SourceModel;
import org.mule.runtime.ast.api.ComponentAst;
import org.mule.runtime.ast.api.MetadataTypeAdapter;
import org.mule.runtime.ast.internal.serialization.dto.ComponentAstDTO;
import org.mule.runtime.ast.internal.serialization.dto.ComponentAstDTOModelType;

import java.util.Optional;

public class ComponentAstDTOFactory {

  private final AstDTOFactoryProvider astDTOFactoryProvider;

  public ComponentAstDTOFactory(AstDTOFactoryProvider astDTOFactoryProvider) {
    this.astDTOFactoryProvider = astDTOFactoryProvider;
  }

  public ComponentAstDTO from(ComponentAst componentAst) {
    return new ComponentAstDTO(
                               unmodifiableList(componentAst.directChildrenStream()
                                   .map(child -> astDTOFactoryProvider.getComponentAstDTOFactory().from(child))
                                   .collect(toList())),
                               componentAst.getComponentType().toString(), componentAst.getIdentifier(),
                               componentAst.getLocation(),
                               componentAst.getMetadata() != null
                                   ? astDTOFactoryProvider.getComponentMetadataAstDTOFactory().from(componentAst.getMetadata())
                                   : null,
                               componentAst.getParameters().stream()
                                   .map(param -> astDTOFactoryProvider.getComponentParameterAstDTOFactory().from(param))
                                   .collect(toList()),
                               componentAst.getComponentId().orElse(null),
                               componentAst.getExtensionModel() != null
                                   ? astDTOFactoryProvider.getExtensionModelDTOFactory().from(componentAst.getExtensionModel())
                                   : null,
                               getModelName(componentAst),
                               getModelType(componentAst),
                               componentAst.getAnnotations() != null
                                   ? unmodifiableMap(componentAst.getAnnotations())
                                   : null);
  }

  private String getModelName(ComponentAst componentAst) {
    return componentAst.getModel(MetadataTypeAdapter.class)
        .flatMap(mt -> getTypeId(mt.getType()))
        .orElseGet(() -> componentAst.getModel(NamedObject.class)
            .map(x -> {
              try {
                return x.getName();
              } catch (Exception e) {
                // This might happen if the model of this component is a MetadataModelTypeAdapter such as in an entry of a map.
                return componentAst.getIdentifier().getName();
              }
            })
            .orElse(null));
  }

  private ComponentAstDTOModelType getModelType(ComponentAst componentAst) {
    Optional<MetadataTypeAdapter> metadataTypeAdapter = componentAst.getModel(MetadataTypeAdapter.class);
    if (metadataTypeAdapter.isPresent()) {
      if (isInfrastructure(metadataTypeAdapter.get().getType())) {
        return INFRASTRUCTURE;
      } else {
        return TYPE;
      }
    } else if (componentAst.getModel(ConfigurationModel.class).isPresent()) {
      return CONFIGURATION;
    } else if (componentAst.getModel(ConnectionProviderModel.class).isPresent()) {
      return CONNECTION_PROVIDER;
    } else if (componentAst.getModel(OperationModel.class).isPresent()) {
      return OPERATION;
    } else if (componentAst.getModel(SourceModel.class).isPresent()) {
      return SOURCE;
    } else if (componentAst.getModel(ConstructModel.class).isPresent()) {
      return CONSTRUCT;
    } else if (componentAst.getModel(NestableElementModel.class).isPresent()
        || componentAst.getModel(NestedComponentModel.class).isPresent()) {
      return NESTED_COMPONENT;
    } else if (componentAst.getModel(NestedRouteModel.class).isPresent()) {
      return NESTED_ROUTE;
    } else {
      return null;
    }
  }
}
