/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.ast.internal.serialization.json.gson;

import static org.mule.runtime.ast.api.error.ErrorTypeRepositoryProvider.getCoreErrorTypeRepo;
import static org.mule.runtime.ast.internal.error.ErrorTypeBuilder.ANY_IDENTIFIER;
import static org.mule.runtime.ast.internal.error.ErrorTypeBuilder.CORE_NAMESPACE_NAME;
import static org.mule.runtime.ast.internal.error.ErrorTypeBuilder.CRITICAL_IDENTIFIER;
import static org.mule.runtime.ast.internal.error.ErrorTypeBuilder.builder;

import org.mule.runtime.api.component.ComponentIdentifier;
import org.mule.runtime.api.exception.ErrorTypeRepository;
import org.mule.runtime.api.message.ErrorType;
import java.lang.reflect.Type;
import java.util.Optional;
import com.google.gson.JsonDeserializationContext;
import com.google.gson.JsonDeserializer;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.google.gson.JsonParseException;

/**
 * This class implements a Gson {@link JsonDeserializer} and it can be registered to Gson as a Type Adapter in order to
 * deserialize {@link ErrorType}s
 */
public class ErrorTypeJsonDeserializer implements JsonDeserializer<ErrorType> {

  private final ErrorTypeRepository coreErrorTypeRepo = getCoreErrorTypeRepo();

  @Override
  public ErrorType deserialize(JsonElement jsonElement, Type type, JsonDeserializationContext jsonDeserializationContext)
      throws JsonParseException {
    JsonObject errorTypeJsonObject = jsonElement.getAsJsonObject();
    return this.getErrorType(
                             type,
                             jsonDeserializationContext,
                             errorTypeJsonObject.get("identifier").getAsString(),
                             errorTypeJsonObject.get("namespace").getAsString(),
                             errorTypeJsonObject.get("parentErrorType"));

  }

  private ErrorType getErrorType(Type type, JsonDeserializationContext jsonDeserializationContext, String identifier,
                                 String namespace, JsonElement parentErrorTypeJsonElement) {
    Optional<ErrorType> errorTypeOptional =
        coreErrorTypeRepo.lookupErrorType(ComponentIdentifier.builder().name(identifier).namespace(namespace).build());

    return errorTypeOptional.orElseGet(() -> this.buildErrorType(type, jsonDeserializationContext, identifier, namespace,
                                                                 parentErrorTypeJsonElement));
  }

  private ErrorType buildErrorType(Type type, JsonDeserializationContext jsonDeserializationContext, String identifier,
                                   String namespace, JsonElement parentErrorTypeJsonElement) {
    if (!isOrphan(identifier, namespace)) {
      return buildErrorTypeWithParent(type, jsonDeserializationContext, identifier, namespace, parentErrorTypeJsonElement);
    }

    return builder().identifier(identifier).namespace(namespace).build();
  }

  private ErrorType buildErrorTypeWithParent(Type type, JsonDeserializationContext jsonDeserializationContext, String identifier,
                                             String namespace, JsonElement parentErrorTypeJsonElement) {
    ErrorType parentErrorType = this.deserialize(parentErrorTypeJsonElement, type, jsonDeserializationContext);
    return builder().identifier(identifier).namespace(namespace).parentErrorType(parentErrorType).build();
  }

  private boolean isOrphan(String identifier, String namespace) {
    return (identifier.equals(ANY_IDENTIFIER) || identifier.equals(CRITICAL_IDENTIFIER))
        && namespace.equals(CORE_NAMESPACE_NAME);
  }
}
