/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.ast.internal.serialization;

import org.mule.runtime.ast.api.ArtifactAst;
import org.mule.runtime.ast.api.ComponentGenerationInformation;
import org.mule.runtime.ast.api.serialization.ArtifactAstSerializer;

import java.util.List;
import java.util.concurrent.Executor;

/**
 * Represents a serialization/deserialization format for an {@link ArtifactAst}.
 *
 * @since 1.3, 1.2, 1.1.4
 */
public interface ArtifactAstSerializerFormat {

  /**
   * @return A {@link String} that identifies this format.
   */
  String getId();

  /**
   * Creates a serializer for this format.
   *
   * @param version  The version of the serializer.
   * @param executor An {@link Executor} to use for the serialization work.
   * @return The serializer.
   */
  ArtifactAstSerializer createSerializer(String version, Executor executor);

  /**
   * Creates a deserializer for this format.
   *
   * @param version                       The version of the serializer.
   * @param populateGenerationInformation Whether the deserializer must populate the {@link ComponentGenerationInformation}.
   * @return The deserializer.
   */
  InternalArtifactAstDeserializer createDeserializer(String version, boolean populateGenerationInformation);

  /**
   * @return The available versions of this format.
   */
  List<String> getAvailableVersions();

  /**
   * @return The latest available version of this format.
   */
  String getLatestVersion();
}
