/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.ast.test.internal.serialization.dto.factory;

import static java.util.Arrays.asList;
import static java.util.stream.Collectors.toList;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.containsInAnyOrder;
import static org.hamcrest.Matchers.containsString;
import static org.hamcrest.Matchers.instanceOf;
import static org.hamcrest.Matchers.is;
import static org.junit.Assert.assertThrows;

import static org.mule.runtime.api.component.ComponentIdentifier.builder;
import static org.mule.runtime.ast.AllureConstants.ArtifactAstSerialization.AST_DTO;
import static org.mule.runtime.ast.AllureConstants.ArtifactAstSerialization.AST_SERIALIZATION;
import static org.mule.runtime.ast.api.error.ErrorTypeRepositoryProvider.getCoreErrorTypeRepo;

import org.mule.runtime.api.component.ComponentIdentifier;
import org.mule.runtime.api.exception.ErrorTypeRepository;
import org.mule.runtime.api.message.ErrorType;
import org.mule.runtime.ast.internal.error.CompositeErrorTypeRepository;
import org.mule.runtime.ast.internal.error.DefaultErrorTypeRepository;
import org.mule.runtime.ast.internal.serialization.dto.ErrorTypeRepositoryDTO;
import org.mule.runtime.ast.internal.serialization.dto.factory.ErrorTypeRepositoryDTOFactory;

import java.util.ArrayList;
import java.util.Optional;

import io.qameta.allure.Issue;
import org.junit.Before;
import org.junit.Test;

import io.qameta.allure.Feature;
import io.qameta.allure.Story;

@Feature(AST_SERIALIZATION)
@Story(AST_DTO)
public class ErrorTypeRepositoryDTOFactoryTestCase {

  private static final ErrorTypeRepository CORE_ERROR_TYPE_REPOSITORY = getCoreErrorTypeRepo();

  private ErrorTypeRepository coreErrorTypeRepo;
  private ErrorTypeRepository errorTypeRepository;
  private ErrorTypeRepositoryDTOFactory errorTypeRepositoryDTOFactory;

  @Before
  public void setUp() throws Exception {
    errorTypeRepositoryDTOFactory = new ErrorTypeRepositoryDTOFactory();
    coreErrorTypeRepo = CORE_ERROR_TYPE_REPOSITORY;
    errorTypeRepository = new DefaultErrorTypeRepository();
  }

  @Test
  public void testBuildReturnsErrorTypeRepositoryDTOWithOneErrorTypeDTO_WhenBuildingFromAnErrorTypeRepositoryWithOneErrorType() {
    // Given
    addErrorType("boom", "chaka", errorTypeRepository);

    // When
    ErrorTypeRepositoryDTO errorTypeRepositoryDTO = toErrorTypeRepoDto();

    // Then
    assertThat(errorTypeRepositoryDTO.getErrorTypes()
        .stream()
        .filter(errorType -> !errorType.getNamespace().equals("MULE"))
        .count(),
               is(1L));
  }

  @Test
  public void testBuildReturnsErrorTypeRepositoryDTOWithOneErrorTypeDTOJuan_WhenBuildingFromAnErrorTypeRepositoryWithOneErrorTypeJuan() {
    // Given
    addErrorType("Juan", "Jhon", errorTypeRepository);

    // When
    ErrorTypeRepositoryDTO errorTypeRepositoryDTO = toErrorTypeRepoDto();

    // Then
    assertThat(errorTypeRepositoryDTO.getErrorTypes()
        .stream()
        .filter(errorType -> !errorType.getNamespace().equals("MULE"))
        .count(),
               is(1L));
  }

  @Test
  public void testBuildReturnsErrorTypeRepositoryDTOWithTwoErrorTypeDTOs_WhenBuildingFromAnErrorTypeRepositoryWithTwoErrorTypes() {
    // Given
    addErrorType("Juan", "Jhon", errorTypeRepository);

    addErrorType("Maria", "Mary", errorTypeRepository);

    // When
    ErrorTypeRepositoryDTO errorTypeRepositoryDTO = toErrorTypeRepoDto();

    // Then
    assertThat(errorTypeRepositoryDTO.getErrorTypes()
        .stream()
        .filter(errorType -> !errorType.getNamespace().equals("MULE"))
        .count(),
               is(2L));
  }

  @Test
  public void testBuildReturnsErrorTypeRepositoryDTOWithSameCharacteristics_WhenBuildingFromAnErrorTypeRepository() {
    // Given
    ComponentIdentifier componentIdentifier = addErrorType("boom", "CHAKA", errorTypeRepository);

    // When
    ErrorTypeRepositoryDTO errorTypeRepositoryDTO = toErrorTypeRepoDto();

    // Then
    assertThat(errorTypeRepositoryDTO.getErrorTypes()
        .stream()
        .filter(errorType -> !errorType.getNamespace().equals("MULE"))
        .count(),
               is(1L));
    assertThat(errorTypeRepositoryDTO.getInternalErrorTypes()
        .stream()
        .filter(errorType -> !errorType.getNamespace().equals("MULE"))
        .count(),
               is(0L));
    assertThat(errorTypeRepositoryDTO.getErrorNamespaces()
        .stream()
        .filter(ns -> !ns.equals("MULE"))
        .collect(toList()),
               containsInAnyOrder(errorTypeRepository.getErrorNamespaces().toArray()));
    assertThat(errorTypeRepositoryDTO.getAnyErrorType(), is(coreErrorTypeRepo.getAnyErrorType()));
    assertThat(errorTypeRepositoryDTO.getCriticalErrorType(), is(coreErrorTypeRepo.getCriticalErrorType()));
    assertThat(errorTypeRepositoryDTO.getSourceErrorType(), is(coreErrorTypeRepo.getSourceErrorType()));
    assertThat(errorTypeRepositoryDTO.getSourceResponseErrorType(), is(coreErrorTypeRepo.getSourceResponseErrorType()));
    assertThat(errorTypeRepositoryDTO.lookupErrorType(componentIdentifier),
               is(errorTypeRepository.lookupErrorType(componentIdentifier)));
  }

  @Test
  public void testBuildReturnsErrorTypeRepositoryDTOWithSameCharacteristics_WhenBuildingFromAnErrorTypeRepositoryWithTwoErrorTypes() {
    // Given
    ComponentIdentifier componentIdentifier = addErrorType("boom", "CHAKA", errorTypeRepository);

    ComponentIdentifier componentIdentifier2 = addErrorType("UNKNOWN", "EXT", errorTypeRepository);

    // When
    ErrorTypeRepositoryDTO errorTypeRepositoryDTO = toErrorTypeRepoDto();

    // Then
    assertThat(errorTypeRepositoryDTO.getErrorTypes()
        .stream()
        .filter(errorType -> !errorType.getNamespace().equals("MULE"))
        .count(),
               is(2L));
    assertThat(errorTypeRepositoryDTO.getInternalErrorTypes()
        .stream()
        .filter(errorType -> !errorType.getNamespace().equals("MULE"))
        .count(),
               is(0L));
    assertThat(errorTypeRepositoryDTO.getErrorNamespaces()
        .stream()
        .filter(ns -> !ns.equals("MULE"))
        .collect(toList()),
               containsInAnyOrder(errorTypeRepository.getErrorNamespaces().toArray()));
    assertThat(errorTypeRepositoryDTO.getAnyErrorType(), is(coreErrorTypeRepo.getAnyErrorType()));
    assertThat(errorTypeRepositoryDTO.getCriticalErrorType(), is(coreErrorTypeRepo.getCriticalErrorType()));
    assertThat(errorTypeRepositoryDTO.getSourceErrorType(), is(coreErrorTypeRepo.getSourceErrorType()));
    assertThat(errorTypeRepositoryDTO.getSourceResponseErrorType(), is(coreErrorTypeRepo.getSourceResponseErrorType()));
    assertThat(errorTypeRepositoryDTO.lookupErrorType(componentIdentifier),
               is(errorTypeRepository.lookupErrorType(componentIdentifier)));
    assertThat(errorTypeRepositoryDTO.lookupErrorType(componentIdentifier2),
               is(errorTypeRepository.lookupErrorType(componentIdentifier2)));
    assertThat(errorTypeRepositoryDTO.getErrorType(componentIdentifier),
               is(errorTypeRepository.getErrorType(componentIdentifier)));
    assertThat(errorTypeRepositoryDTO.getErrorType(componentIdentifier2),
               is(errorTypeRepository.getErrorType(componentIdentifier2)));
  }

  @Test
  public void testBuildReturnsErrorTypeRepositoryDTOWithSameCharacteristics_WhenBuildingFromAnErrorTypeRepositoryWithTwoErrorTypesAndTwoInternalErrorTypes() {
    // Given
    ComponentIdentifier componentIdentifier = addErrorType("boom", "CHAKA", errorTypeRepository);

    ComponentIdentifier componentIdentifier2 = addErrorType("UNKNOWN", "EXT", errorTypeRepository);

    ComponentIdentifier internalComponentIdentifier = addInternalErrorType("int", "ER", errorTypeRepository);

    ComponentIdentifier internalComponentIdentifier2 =
        addInternalErrorType("KNOWN", "EXT", errorTypeRepository);

    // When
    ErrorTypeRepositoryDTO errorTypeRepositoryDTO = toErrorTypeRepoDto();

    // Then
    assertThat(errorTypeRepositoryDTO.getErrorTypes()
        .stream()
        .filter(errorType -> !errorType.getNamespace().equals("MULE"))
        .count(),
               is(2L));
    assertThat(errorTypeRepositoryDTO.getInternalErrorTypes()
        .stream()
        .filter(errorType -> !errorType.getNamespace().equals("MULE"))
        .count(),
               is(2L));

    assertThat(errorTypeRepositoryDTO.getErrorNamespaces()
        .stream()
        .filter(ns -> !ns.equals("MULE"))
        .collect(toList()),
               containsInAnyOrder(errorTypeRepository.getErrorNamespaces().toArray()));

    assertThat(errorTypeRepositoryDTO.getAnyErrorType(), is(coreErrorTypeRepo.getAnyErrorType()));
    assertThat(errorTypeRepositoryDTO.getCriticalErrorType(), is(coreErrorTypeRepo.getCriticalErrorType()));

    assertThat(errorTypeRepositoryDTO.getSourceErrorType(), is(coreErrorTypeRepo.getSourceErrorType()));
    assertThat(errorTypeRepositoryDTO.getSourceResponseErrorType(), is(coreErrorTypeRepo.getSourceResponseErrorType()));

    assertThat(errorTypeRepositoryDTO.lookupErrorType(componentIdentifier),
               is(errorTypeRepository.lookupErrorType(componentIdentifier)));
    assertThat(errorTypeRepositoryDTO.lookupErrorType(componentIdentifier2),
               is(errorTypeRepository.lookupErrorType(componentIdentifier2)));

    assertThat(errorTypeRepositoryDTO.lookupErrorType(internalComponentIdentifier),
               is(errorTypeRepository.lookupErrorType(internalComponentIdentifier)));
    assertThat(errorTypeRepositoryDTO.lookupErrorType(internalComponentIdentifier2),
               is(errorTypeRepository.lookupErrorType(internalComponentIdentifier2)));

    assertThat(errorTypeRepositoryDTO.getErrorType(componentIdentifier),
               is(errorTypeRepository.getErrorType(componentIdentifier)));
    assertThat(errorTypeRepositoryDTO.getErrorType(componentIdentifier2),
               is(errorTypeRepository.getErrorType(componentIdentifier2)));
  }

  @Test
  public void testBuildReturnsErrorTypeRepositoryDTOWithSameCharacteristicsAsComposite_WhenBuildingFromACompositeErrorTypeRepositoryWithTwoErrorTypesAndTwoInternalErrorTypes() {
    // Given
    ArrayList<ErrorTypeRepository> children = new ArrayList<>();

    DefaultErrorTypeRepository defaultErrorTypeRepository1 = new DefaultErrorTypeRepository();
    children.add(defaultErrorTypeRepository1);

    DefaultErrorTypeRepository defaultErrorTypeRepository2 = new DefaultErrorTypeRepository();
    children.add(defaultErrorTypeRepository2);

    ComponentIdentifier componentIdentifier = addErrorType("boom", "CHAKA", defaultErrorTypeRepository1);

    ComponentIdentifier componentIdentifier2 = addErrorType("UNKNOWN", "EXT", defaultErrorTypeRepository2);

    ComponentIdentifier internalComponentIdentifier = addInternalErrorType("int", "ER", defaultErrorTypeRepository1);

    ComponentIdentifier internalComponentIdentifier2 = addInternalErrorType("KNOWN", "EXT", defaultErrorTypeRepository2);

    this.errorTypeRepository = new CompositeErrorTypeRepository(children);

    // When
    ErrorTypeRepositoryDTO errorTypeRepositoryDTO = toErrorTypeRepoDto();

    // Then
    assertThat(errorTypeRepositoryDTO.getErrorTypes()
        .stream()
        .filter(errorType -> !errorType.getNamespace().equals("MULE"))
        .count(),
               is(2L));
    assertThat(errorTypeRepositoryDTO.getInternalErrorTypes()
        .stream()
        .filter(errorType -> !errorType.getNamespace().equals("MULE"))
        .count(),
               is(2L));

    assertThat(errorTypeRepositoryDTO.getErrorNamespaces()
        .stream()
        .filter(ns -> !ns.equals("MULE"))
        .collect(toList()),
               containsInAnyOrder(errorTypeRepository.getErrorNamespaces().toArray()));

    assertThat(errorTypeRepositoryDTO.getAnyErrorType(), is(coreErrorTypeRepo.getAnyErrorType()));
    assertThat(errorTypeRepositoryDTO.getCriticalErrorType(), is(coreErrorTypeRepo.getCriticalErrorType()));

    assertThat(errorTypeRepositoryDTO.getSourceErrorType(), is(coreErrorTypeRepo.getSourceErrorType()));
    assertThat(errorTypeRepositoryDTO.getSourceResponseErrorType(), is(coreErrorTypeRepo.getSourceResponseErrorType()));

    assertThat(errorTypeRepositoryDTO.lookupErrorType(componentIdentifier),
               is(errorTypeRepository.lookupErrorType(componentIdentifier)));
    assertThat(errorTypeRepositoryDTO.lookupErrorType(componentIdentifier2),
               is(errorTypeRepository.lookupErrorType(componentIdentifier2)));

    assertThat(errorTypeRepositoryDTO.lookupErrorType(internalComponentIdentifier),
               is(errorTypeRepository.lookupErrorType(internalComponentIdentifier)));
    assertThat(errorTypeRepositoryDTO.lookupErrorType(internalComponentIdentifier2),
               is(errorTypeRepository.lookupErrorType(internalComponentIdentifier2)));

    assertThat(errorTypeRepositoryDTO.getErrorType(componentIdentifier),
               is(errorTypeRepository.getErrorType(componentIdentifier)));
    assertThat(errorTypeRepositoryDTO.getErrorType(componentIdentifier2),
               is(errorTypeRepository.getErrorType(componentIdentifier2)));
  }

  @Test
  public void testBuildReturnsErrorTypeRepositoryDTOWithSameCharacteristicsAsMuleCore_WhenBuildingFromAMuleCoreErrorTypeRepository() {
    // Given
    this.errorTypeRepository = CORE_ERROR_TYPE_REPOSITORY;

    // When
    ErrorTypeRepositoryDTO errorTypeRepositoryDTO = toErrorTypeRepoDto();

    // Then
    assertThat(errorTypeRepositoryDTO.getErrorTypes().size(), is(errorTypeRepository.getErrorTypes().size()));
    assertThat(errorTypeRepositoryDTO.getInternalErrorTypes().size(), is(errorTypeRepository.getInternalErrorTypes().size()));

    assertThat(errorTypeRepositoryDTO.getErrorNamespaces(),
               containsInAnyOrder(errorTypeRepository.getErrorNamespaces().toArray()));

    assertThat(errorTypeRepositoryDTO.getAnyErrorType(), is(errorTypeRepository.getAnyErrorType()));
    assertThat(errorTypeRepositoryDTO.getCriticalErrorType(), is(errorTypeRepository.getCriticalErrorType()));

    assertThat(errorTypeRepositoryDTO.getSourceErrorType(), is(errorTypeRepository.getSourceErrorType()));
    assertThat(errorTypeRepositoryDTO.getSourceResponseErrorType(), is(errorTypeRepository.getSourceResponseErrorType()));
  }

  @Test
  @Issue("MULE-19683")
  public void errorTypeNamespaceCaseInsensitive() {
    // Given
    ComponentIdentifier componentIdentifier = addErrorType("boom", "CHAKA", errorTypeRepository);

    // When
    ErrorTypeRepositoryDTO errorTypeRepositoryDTO = toErrorTypeRepoDto();

    Optional<ErrorType> lookupErrorType =
        errorTypeRepositoryDTO
            .lookupErrorType(ComponentIdentifier.builder().name("boom").namespace("CHAKA".toLowerCase()).build());
    assertThat(lookupErrorType.get().getNamespace(), is(componentIdentifier.getNamespace()));
    assertThat(lookupErrorType.get().getIdentifier(), is(componentIdentifier.getName()));
  }

  @Test
  @Issue("W-16237424")
  public void errorTypeNamespaceWithTrailingWhiteSpace() {
    // Given name and namespace with white spaces
    addErrorType("   boom     ", "    CHAKA    ", errorTypeRepository);

    // When
    ErrorTypeRepositoryDTO errorTypeRepositoryDTO = toErrorTypeRepoDto();

    Optional<ErrorType> lookupErrorType =
        errorTypeRepositoryDTO
            .lookupErrorType(ComponentIdentifier.builder().name("boom").namespace("CHAKA".toLowerCase()).build());
    assertThat(lookupErrorType.get().getNamespace(), is("CHAKA"));
    assertThat(lookupErrorType.get().getIdentifier(), is("boom"));
  }

  @Test
  @Issue("W-16237424")
  public void errorTypeNamespaceWithInvalidString_ShouldThrowIllegalStateException() {
    // Given name and namespace which are empty
    Exception exception = assertThrows(IllegalStateException.class, () -> {
      addErrorType("", "", errorTypeRepository);
    });

    assertThat(exception, instanceOf(IllegalStateException.class));
    assertThat(exception.getMessage(), containsString("Prefix URI must be not blank"));
  }

  protected ErrorTypeRepositoryDTO toErrorTypeRepoDto() {
    return errorTypeRepositoryDTOFactory.from(new CompositeErrorTypeRepository(asList(coreErrorTypeRepo, errorTypeRepository)));
  }

  private ComponentIdentifier addErrorType(String name, String namespace,
                                           ErrorTypeRepository errorTypeRepository) {
    ComponentIdentifier componentIdentifier = builder().name(name).namespace(namespace).build();
    errorTypeRepository.addErrorType(componentIdentifier, coreErrorTypeRepo.getAnyErrorType());
    return componentIdentifier;
  }

  private ComponentIdentifier addInternalErrorType(String name, String namespace,
                                                   ErrorTypeRepository errorTypeRepository) {
    ComponentIdentifier componentIdentifier = builder().name(name).namespace(namespace).build();
    errorTypeRepository.addInternalErrorType(componentIdentifier, coreErrorTypeRepo.getAnyErrorType());
    return componentIdentifier;
  }

}
